# coding: utf-8

import os
import re
import glob
import urllib

import requests

from sandbox.common.types.task import ReleaseStatus
from sandbox.common import errors

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import ssh

from sandbox.projects.juggler import resource_types as juggler_resources
from sandbox.projects.common import debpkg
from sandbox.projects.common import gnupg

DEBEMAIL = 'robot-juggling (robot-juggling) <robot-juggling@yandex-team.ru>'
DEBFULLNAME = 'robot-juggling (tech user)'

DUPLOAD_CONF = {
    'common': {
        'fqdn': "common.dupload.dist.yandex.ru",
        'method': "scpb",
        'login': "robot-juggling",
        'incoming': "/repo/common/mini-dinstall/incoming/",
        'dinstall_runs': 1,
    },
    'sysmon-common': {
        'fqdn': "sysmon-common.dupload.dist.yandex.ru",
        'method': "scpb",
        'login': "robot-juggling",
        'incoming': "/repo/sysmon-common/mini-dinstall/incoming/",
        'dinstall_runs': 1,
    },
    'yandex-precise': {
        'fqdn': "yandex-precise.dupload.dist.yandex.ru",
        'method': "scpb",
        'login': "robot-juggling",
        'incoming': "/repo/yandex-precise/mini-dinstall/incoming/",
        'dinstall_runs': 1,
    },
    'search-precise': {
        'fqdn': "search-precise.dupload.dist.yandex.ru",
        'method': "scpb",
        'login': "robot-juggling",
        'incoming': "/repo/search-precise/mini-dinstall/incoming/",
        'dinstall_runs': 1,
    },
    'search': {
        'fqdn': "search.dupload.dist.yandex.ru",
        'method': "scpb",
        'login': "robot-juggling",
        'incoming': "/repo/search/mini-dinstall/incoming/",
        'dinstall_runs': 1,
    }
}

CONDUCTOR_BRANCH_MAP = {
    ReleaseStatus.UNSTABLE: "unstable",
    ReleaseStatus.TESTING: "testing",
    ReleaseStatus.PRESTABLE: "prestable",
    ReleaseStatus.STABLE: "stable"
}

CONDUCTOR_URL = "http://c.yandex-team.ru"


def parse_changelog(checkout_path):
    proc = process.run_process(
        ['dpkg-parsechangelog'],
        outs_to_pipe=True, work_dir=checkout_path)
    result = {}
    value_list = None
    stdout_data, _ = proc.communicate()
    for line in stdout_data.splitlines():
        value = None
        match = re.match(r"^([A-Z][a-z]+):", line)
        if match is None and value_list is not None:
            value = line.rstrip()
        elif match is not None:
            value_list = result[match.group(1)] = []
            value = line.partition(':')[2].strip()
        if value:
            value_list.append(value)
    return {key: "\n".join(value_list) for key, value_list in result.items()}


def extract_changelog_version(checkout_path):
    version = parse_changelog(checkout_path).get("Version")
    if version is None:
        raise errors.SandboxEnvironmentError('No version found in debian changelog')
    return version


def extract_changelog_comment(checkout_path):
    changes = parse_changelog(checkout_path).get("Changes")
    if changes is None:
        raise errors.SandboxEnvironmentError('No comment found in debian changelog')
    return "\n".join(changes.splitlines()[2:])


def update_changelog(checkout_path, version_postfix, message='Hand-made build with empty changelog message'):
    """
    Write new version with given postfix to debian/changelog.
    """
    environ = {
        'DEBEMAIL': DEBEMAIL,
        'DEBFULLNAME': DEBFULLNAME,
    }
    with process.CustomOsEnviron(environ):
        version = extract_changelog_version(checkout_path)
        new_version = '{0}+{1}'.format(version, version_postfix)
        process.run_process(
            ['dch', '--check-dirname-level', '0', '-b', '--distributor', 'debian',
             '--newVersion', new_version, message],
            log_prefix='dch', work_dir=checkout_path)
        return new_version


def check_for_changelog(checkout_path):
    """
    Check last commit for changes in debian/changelog.
    """
    proc = process.run_process(
        ['git', 'diff', 'HEAD~1', 'debian/changelog'],
        outs_to_pipe=True, work_dir=checkout_path)
    stdout_data, _ = proc.communicate()
    if not stdout_data.strip():
        raise errors.SandboxEnvironmentError('debian/changelog is not modified in the last commit')
    return extract_changelog_version(checkout_path)


def build_deb(parent_task, checkout_path, repository_name, version, preserve_envvar=(),
              make_threads=1, upload_to_repo=1):
    """
    Build debian package and upload it to repository.
    """
    # retrieve GPG keys from Vault
    with gnupg.GpgKey(parent_task, 'JUGGLER', 'robot-juggling-gpg-private', 'robot-juggling-gpg-public'):
        args = ['debuild', '-d', '--check-dirname-level', '0', '--no-tgz-check', '--no-lintian']
        for envvar in preserve_envvar:
            args.extend(('-e', envvar))
        args.extend(('-krobot-juggling@yandex-team.ru', '-j%d' % make_threads))
        process.run_process(args, log_prefix='debuild', work_dir=checkout_path)

        # upload package to repository
        if upload_to_repo:
            upload_deb_package(parent_task, checkout_path, repository_name)

        # build packages meta
        for package_path in glob.glob(os.path.join(checkout_path, '..', '*.deb')):
            file_name = os.path.splitext(os.path.basename(package_path))[0]
            package_name, package_version, _ = file_name.split("_")
            if package_version != version:
                raise errors.SandboxEnvironmentError(
                    "Package version doesn't match build version: {0} vs {1}".format(package_version, version))
            parent_task._make_resource(path=package_path,
                                       version=version,
                                       resource_type=juggler_resources.JUGGLER_DEB_PACKAGE,
                                       attributes={"package_name": package_name, "version": version},
                                       description="{0} version {1}".format(package_name, version))


def upload_deb_package(parent_task, checkout_path, repository_name):
    with debpkg.DebRelease(DUPLOAD_CONF) as deb:
        with ssh.Key(parent_task, 'JUGGLER', 'robot-juggling-ssh'):
            for change_file in glob.glob(os.path.join(checkout_path, '..', '*.changes')):
                change_file = os.path.basename(change_file)
                proc = process.run_process(
                    ['ssh', '-o', 'StrictHostKeyChecking=no', '-T', '-n',
                     'robot-juggling@%s' % DUPLOAD_CONF[repository_name]['fqdn'],
                     'check4changes', repository_name, change_file],
                    check=False)
                if proc.returncode == 255:
                    # ssh should return 255 on error
                    raise errors.SandboxEnvironmentError(
                        'Unable to connect to %s using ssh' % DUPLOAD_CONF[repository_name]['fqdn'])
                elif proc.returncode != 1:
                    # check4changes should return code 1 if file exists
                    raise errors.SandboxEnvironmentError(
                        'File %s already exists in %s' % (change_file, repository_name))
            deb.debrelease(['--to', repository_name], work_dir=checkout_path)


def create_conductor_ticket(parent_task, release_status, excluded_packages=[]):
    branch = CONDUCTOR_BRANCH_MAP.get(release_status)
    if branch is None:
        return
    resource_list = channel.channel.sandbox.list_resources(
        resource_type=juggler_resources.JUGGLER_DEB_PACKAGE,
        task_id=parent_task.id
    )
    if not resource_list:
        return
    if excluded_packages:
        resource_list = filter(lambda x: x.attributes['package_name'] not in excluded_packages, resource_list)
    comment = parent_task.ctx.get("changelog_last_comment", "created from sandbox task {0}".format(parent_task.id))
    conductor_auth = parent_task.get_vault_data('JUGGLER', 'robot-juggling-conductor-auth')
    create_conductor_ticket2(resource_list, branch, conductor_auth, comment)


def create_conductor_ticket2(resource_list, branch, auth_key, comment=None):
    if comment is None:
        comment = "Automatic release from sandbox"

    query_args = [("ticket[branch]", branch), ("ticket[comment]", comment)]
    for idx, resource in enumerate(resource_list):
        query_args.append(("package[{0}]".format(idx), resource.attributes["package_name"]))
        query_args.append(("version[{0}]".format(idx), resource.attributes["version"]))
    query_url = "{0}/auth_update/ticket_add?{1}".format(CONDUCTOR_URL, urllib.urlencode(query_args, True))
    headers = {"Cookie": "conductor_auth={0}".format(auth_key)}
    requests.get(query_url, headers=headers).raise_for_status()
