# coding: utf-8

import os
import logging

from sandbox.sandboxsdk import process


def find_revision(checkout_path, spec):
    proc = process.run_process(
        ['git', 'rev-parse', 'refs/remotes/origin/{0}^{{commit}}'.format(spec)],
        outs_to_pipe=True, check=False, work_dir=checkout_path)
    if proc.returncode:
        proc = process.run_process(
            ['git', 'rev-parse', 'refs/tags/{0}^{{commit}}'.format(spec)],
            outs_to_pipe=True, check=False, work_dir=checkout_path)
        if proc.returncode:
            # maybe it's commit already
            proc = process.run_process(
                ['git', 'rev-parse', spec],
                outs_to_pipe=True, work_dir=checkout_path)
    commit, _ = proc.communicate()
    return commit.strip()


def checkout(git_url, checkout_path, tag, update_submodules=True):
    """
    Clone git repository and checkout to given branch/tag.
    """
    logging.info('fetching repo')
    assert tag, 'Trying to fetch project from git, but no tag specified'
    if os.path.exists(checkout_path):
        process.run_process(
            ['git', 'retry', '-v', 'fetch', git_url, '+refs/heads/*:refs/remotes/origin/*'],
            work_dir=checkout_path,
            log_prefix='git_fetch')
    else:
        process.run_process(
            ['git', 'retry', '-v', 'clone', git_url, checkout_path],
            log_prefix='git_clone')
    commit = find_revision(checkout_path, tag)
    process.run_process(
        ['git', 'checkout', '-f', commit],
        work_dir=checkout_path,
        log_prefix='git_checkout')
    if update_submodules:
        process.run_process(
            ['git', 'retry', '-v', 'submodule', 'update', '--init'],
            work_dir=checkout_path,
            log_prefix='git_submodule')
    return commit
