# coding: utf-8

import os
import contextlib
import six
import yaml

from sandbox.common import errors
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import process

from . import wheels
from . import bundle

WHEEL_VERSION = '0.29.0'


class ReqParser(object):

    def __init__(self):
        self.local_packages = []
        self.sandbox_resources = []
        self.remote_packages = []
        self.parent_packages = []

    def _parse_txt_requirements(self, path):
        with open(path, 'r') as fd:
            for line in fd:
                line = line.strip()
                if line.startswith('#') or not line:
                    continue
                elif line.startswith('-'):
                    # FIXME: options are currently ignored
                    continue
                else:
                    line = line.split("#", 1)[0]
                    name, _, version = line.partition('==')
                    name = name.strip()
                    version = version.strip()
                    if not version:
                        raise errors.SandboxEnvironmentError(
                            'No version for package {name} specified'.format(name=name))
                    self.remote_packages.append({
                        "package_name": name.lower(),
                        "package_version": version
                    })

    def _parse_yaml_requirements(self, path):
        with open(path, 'r') as fd:
            for name, params in (yaml.safe_load(fd) or {}).items():
                if "resource_id" in params:
                    self.sandbox_resources.append(params)

                elif "source_path" in params:
                    self.local_packages.append(params["source_path"])

                else:
                    if "version" not in params:
                        raise errors.SandboxEnvironmentError(
                            'No version for package {name} specified'.format(name=name)
                        )

                    result = {
                        "package_name": name.lower(),
                        "package_version": params["version"]
                    }
                    if "target" in params:
                        result["arcadia_target"] = params["target"]
                    if "artifact" in params:
                        result["arcadia_artifact"] = params["artifact"]
                    self.remote_packages.append(result)

    def parse(self, requirements_list):
        for path in requirements_list:
            if path.endswith((".yml", ".yaml")):
                self._parse_yaml_requirements(path)
            else:
                self._parse_txt_requirements(path)


@contextlib.contextmanager
def venv_context(parent_task, venv_path, requirements_list=None, checkout_path=None,
                 python_path=None, ignored_paths=None, target_prefix=None, root_path=None,
                 local_packages=None):
    if isinstance(requirements_list, six.string_types):
        requirements_list = [requirements_list]

    with environments.VirtualEnvironment(venv_path, use_system=False) as venv:
        # install dependencies into virtualenv
        parser = ReqParser()
        if requirements_list is not None:
            parser.parse(requirements_list)
        if not any(params["package_name"] == "wheel" for params in parser.remote_packages):
            parser.remote_packages.append({
                "package_name": "wheel",
                "package_version": WHEEL_VERSION
            })

        install_args = [
            venv.executable, os.path.join(venv.root_dir, 'bin', 'pip'),
            'install', '--no-deps', '--ignore-installed'
        ]
        install_args.extend(wheels.collect_wheels(
            parent_task, parser.remote_packages, wheels.get_abbr_impl(venv.executable)))
        process.run_process(install_args, log_prefix='pip_wheels')

        # generate install scripts
        install_scripts_args = [venv.executable, '-m', 'wheel', 'install-scripts']
        install_scripts_args.extend(params["package_name"] for params in parser.remote_packages)
        process.run_process(install_scripts_args, log_prefix='wheel_install_scripts')

        for params in parser.sandbox_resources:
            wheels.download_resource(venv.root_dir, **params)

        if root_path is None:
            root_path = checkout_path

        if root_path is not None:
            merged_local_packages = [os.path.join(root_path, path) for path in parser.local_packages]
            if checkout_path is not None:
                merged_local_packages.append(checkout_path)
            if local_packages is not None:
                merged_local_packages.extend(os.path.join(root_path, path) for path in local_packages)
            for package_path in merged_local_packages:
                # now install packages itself
                venv.pip('--no-deps --ignore-installed {0}'.format(package_path))
        elif parser.local_packages:
            raise errors.SandboxEnvironmentError('No checkout path specified but local packages used')

        if python_path is not None:
            bundle.make_bundle(venv.root_dir, python_path, ignored_paths, target_prefix)
        else:
            # update shebang
            bundle.fix_shebang(venv.root_dir, '/skynet/python/bin/python', ignored_paths)
            bundle.patch_activate_this(venv.root_dir)
            bundle.patch_real_prefix(venv.root_dir, '/skynet/python')
            bundle.remove_pyc(venv.root_dir)

        # remove activate script because it didn't work
        for script_name in ('activate.csh', 'activate.fish', 'activate'):
            os.unlink(os.path.join(venv.root_dir, 'bin', script_name))

        yield venv
