# coding: utf-8

import os
import tempfile
import platform
import shutil
import errno
import glob
import tarfile

from sandbox import common
from sandbox.projects.juggler import resource_types as juggler_resources
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import process
from sandbox.common.types.task import Status
from sandbox.common.platform import get_platform_alias

VERSION_SCRIPT = """
import sys

if hasattr(sys, 'pypy_version_info'):
    pyimpl = 'pp'
elif sys.platform.startswith('java'):
    pyimpl = 'jy'
elif sys.platform == 'cli':
    pyimpl = 'ip'
elif getattr(sys, "real_prefix", sys.prefix) == '/skynet/python':
    pyimpl = 'sky'
else:
    pyimpl = 'cp'

if pyimpl == 'pp':
    # as per https://github.com/pypa/pip/issues/2882
    version = "{0}{1}{2}".format(
        sys.version_info[0], sys.pypy_version_info.major, sys.pypy_version_info.minor)
else:
    version = "{0}{1}".format(sys.version_info[0], sys.version_info[1])

print("{0}{1}".format(pyimpl, version))
""".strip()


def get_abbr_impl(interpreter):
    with tempfile.NamedTemporaryFile() as fd:
        fd.write(VERSION_SCRIPT)
        fd.flush()
        proc = process.run_process([interpreter, fd.name], outs_to_pipe=True)
        stdout, _ = proc.communicate()
        return stdout.strip()


class JugglerWheelEnvironment(environments.SandboxEnvironment):

    resource_type = juggler_resources.JUGGLER_WHEEL

    def __init__(self, task_params, pyimpl):
        super(JugglerWheelEnvironment, self).__init__(version=task_params["package_version"])
        self.name = task_params["package_name"]
        self.task_params = task_params
        self.pyimpl = pyimpl

    def _environment_resource_query(self):
        query = {
            "resource_type": str(self.resource_type),
            "limit": 100,
            "status": sandboxapi.RESOURCE_READY,
            "all_attrs": {
                "package_name": self.name,
                "version": self.version,
                "pyimpl": self.pyimpl
            }
        }
        return query

    @property
    def released_compatible_resource(self):
        # we don't release wheels
        return None

    def prepare(self):
        pass

    @property
    def _context_key(self):
        return "%s_subtask" % self.name

    def _check_build_task(self, task):
        if self._context_key in task.ctx:
            build_task = channel.channel.sandbox.get_task(task.ctx[self._context_key])
            if build_task.is_failure():
                raise errors.SandboxTaskFailureError(
                    'Build package {name}=={version} in task {task_id} was failed.'.format(
                        name=self.name, version=self.version, task_id=build_task.id
                    )
                )
            else:
                raise common.errors.SandboxEnvironmentError(
                    'No resource with package {name}=={version} found in task {task_id}.'.format(
                        name=self.name, version=self.version, task_id=build_task.id
                    )
                )

    def _build_package(self, task):
        build_task = task.create_subtask(
            task_type='BUILD_JUGGLER_WHEEL',
            description='Build package {package_name}=={version} for {task_id} task'.format(
                package_name=self.name, version=self.version, task_id=task.id
            ),
            input_parameters=self.task_params,
            arch=get_platform_alias(platform.platform())
        )
        task.ctx[self._context_key] = build_task.id
        return build_task

    def build_resource(self, task):
        self._check_build_task(task)
        return self._build_package(task)


def collect_wheels(parent_task, packages, pyimpl):
    import api.copier
    import api.copier.errors

    task_list = []
    resource_list = []
    for params in packages:
        wheel = JugglerWheelEnvironment(params, pyimpl)
        try:
            resource_list.append(wheel.compatible_resource)
        except common.errors.SandboxEnvironmentError:
            task_list.append(wheel.build_resource(parent_task))

    if task_list:
        parent_task.wait_tasks(
            tasks=task_list,
            statuses=tuple(Status.Group.FINISH + Status.Group.BREAK),
            wait_all=True)

    # download wheels in parallel
    wheel_list = []
    root_path = parent_task.path()
    copier = api.copier.Copier()
    copier_list = []
    for resource in resource_list:
        path = os.path.join(root_path, os.path.basename(str(resource.path)))
        if os.path.exists(path):
            wheel_list.append(path)
        # elif random.random() < 0.01:
        #     wheel_list.append(channel.channel.task.sync_resource(resource.id))
        else:
            copier_list.append((
                resource,
                copier.handle(resource.skynet_id).get(dest=root_path, user=True)
            ))

    for resource, copier_task in copier_list:
        try:
            copier_files = copier_task.wait().files()

            if isinstance(copier_files, dict):
                filenames = (
                    os.path.join(root_path, x['path'])
                    for x in copier_files['files']
                )
            else:
                filenames = (x['name'] for x in copier_files)

            wheel_list.extend(filenames)
        except (api.copier.errors.CopierError, OSError):
            wheel_list.append(channel.channel.task.sync_resource(resource.id))

    return wheel_list


def mkdir(path):
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def download_resource(root_dir, resource_id, target_path=None,
                      is_archive=False, strip_components=None,
                      binary_path=None, python_module=False, **kwargs):

    source_path = channel.channel.task.sync_resource(resource_id)
    if target_path is not None:
        target_path = os.path.join(root_dir, target_path)
        mkdir(target_path)

    if is_archive:
        if target_path is None:
            raise errors.SandboxTaskFailureError("No target_path specified")
        cmd = ["tar"]
        if strip_components is not None:
            cmd.append("--strip-components={0}".format(strip_components))
        cmd.extend(("-C", target_path, "-zxf", source_path))
        process.run_process(cmd)

    elif python_module:
        with tarfile.open(source_path) as archive:
            for name in archive.getnames():
                if not name.endswith(".so"):
                    continue
                with open(os.path.join(root_dir, "lib", "python2.7", "site-packages", os.path.basename(name)), "wb") as stream:
                    shutil.copyfileobj(archive.extractfile(name), stream)

    else:
        if target_path is None:
            raise errors.SandboxTaskFailureError("No target_path specified")
        if os.path.isdir(source_path):
            if strip_components is not None:
                source_path = os.path.join(source_path, *['*']*int(strip_components))
            for fname in glob.glob(os.path.join(source_path, '*')):
                if os.path.isdir(fname):
                    shutil.copytree(fname, os.path.join(target_path, fname))
                else:
                    shutil.copy(fname, target_path)
        else:
            shutil.copy(source_path, target_path)

    if binary_path:
        environments.SandboxEnvironment.update_os_env("PATH", os.path.join(root_dir, binary_path))
