"""
An alternative to the native Sandbox release scheme with multi-stage and release rollbacks support

The module provides three general methods:

- `find_release`      - find current release
- `release`           - release the given resource
- `cancel_release`    - cancel release for the given resource (release rollback)

and some other supplementary methods:

- `has_release_attr`  - check if the given resource possesses a kosher release attribute for the given stage
- `get_release_time`  - get release time for the given resource and stage

"""

import datetime
import typing  # noqa: UnusedImport
import logging
import dateutil.parser as dt_parser

from sandbox.common.types import task as ctt

from sandbox.projects.common import time_utils
from sandbox.projects.common.kosher_release import private
from sandbox.common import rest


FIND_RELEASE_RESOURCE_LIMIT = 30
STAGE_TYPE = typing.Union[str, ctt.ReleaseStatus]
STR_OR_INT = typing.Union[str, int]

TIMEDELTA_MONTH = datetime.timedelta(days=30)
TIMEDELTA_THREE_MONTHS = datetime.timedelta(days=TIMEDELTA_MONTH.days * 3)
TIMEDELTA_HALF_A_YEAR = datetime.timedelta(days=TIMEDELTA_MONTH.days * 6)


@private.with_sb_rest_client()
def has_release_attr(resource_id, stage=ctt.ReleaseStatus.STABLE, sb_rest_client=None):
    # type: (STR_OR_INT, STAGE_TYPE, typing.Optional[rest.Client]) -> bool
    """
    Check whether the given resource possesses kosher release attribute for the given stage.

    :param resource_id: Sandbox resource ID
    :param stage: release stage (can be either a `ctt.ReleaseStatus` field or any string
    :param sb_rest_client: Sandbox REST client
    :return: True if the respective attribute is present in the given resource, False otherwise
    """

    release_attr_name = private.get_release_attr_name(stage=stage)
    resource = sb_rest_client.resource[resource_id].read()
    return release_attr_name in resource["attributes"]


@private.with_sb_rest_client()
def get_release_time(resource_id, stage=ctt.ReleaseStatus.STABLE, sb_rest_client=None):
    # type: (STR_OR_INT, STAGE_TYPE, typing.Optional[rest.Client]) -> typing.Optional[datetime.datetime]
    """
    Get release time for the given resource and stage

    :param resource_id: Sandbox resource ID
    :param stage: release stage (can be either a `ctt.ReleaseStatus` field or any string
    :param sb_rest_client: Sandbox REST client

    :return: Release time (as a datetime object) or None

    :raises ValueError:
        Raised if the appropriate release attribute exists and its value is neither empty nor a valid UTC ISO string
    """

    release_attr_name = private.get_release_attr_name(stage=stage)
    resource = sb_rest_client.resource[resource_id].read()

    if release_attr_name not in resource["attributes"]:
        logging.warning("Release attribute %s not found for resource #%s: %s", release_attr_name, resource_id, resource)
        return None

    release_time_iso_str = resource["attributes"][release_attr_name]

    if not release_time_iso_str:
        logging.warning("Release attribute %s value empty for resource #%s", release_attr_name, resource_id)
        return None

    return dt_parser.parse(release_time_iso_str)


@private.with_sb_rest_client()
def find_release(resource_type, stage=ctt.ReleaseStatus.STABLE, sb_rest_client=None, custom_attrs=None):
    # type: (str, STAGE_TYPE, typing.Optional[rest.Client], typing.Optional[dict]) -> typing.Optional[int]
    """
    Find latest release for the given resource type and stage.

    We detect the latest release by the max value of the k_released_<stage> resource attribute.
    In order to speed the search process up we perform a number of partial searches by limiting
    resource creation time. We start with a relatively small time interval and move the left bound
    until we find at least one resource with 'k_released_<stage>' attribute. If we do not succeed
    we drop all limitation and perform one more search. At each step we consider a limited number of
    resources (FIND_RELEASE_RESOURCE_LIMIT) that we believe is enough to detect the actual release.

    :param resource_type: Sandbox resource type
    :param stage: release stage (can be either a `ctt.ReleaseStatus` field or any string
    :param sb_rest_client: Sandbox REST client instance
    :param custom_attrs: an optional dict of custom attributes to filter resources

    :return: Resource ID if release found or None if no release found
    """

    logging.info("Find release: %s - %s", resource_type, stage)

    release_attr_name = private.get_release_attr_name(stage=stage)

    logging.info("Release attribute name: %s", release_attr_name)

    now = time_utils.datetime_utc()

    for creation_time_bound in [
        now - TIMEDELTA_THREE_MONTHS,
        now - TIMEDELTA_HALF_A_YEAR,
        None,
    ]:
        resource_id = private.find_release_creation_time_limited(
            resource_type,
            creation_time_bound,
            release_attr_name,
            FIND_RELEASE_RESOURCE_LIMIT,
            sb_rest_client,
            custom_attrs=custom_attrs,
        )

        if resource_id:
            return resource_id

    logging.warning("No resources found")


@private.with_sb_rest_client()
def release(
    resource_id,
    stage=ctt.ReleaseStatus.STABLE,
    sb_rest_client=None,
    skip_sandbox_native_release=False,
    cc="",
    message="",
    subject="",
):
    # type: (STR_OR_INT, STAGE_TYPE, typing.Optional[rest.Client], bool, str, str, str) -> None
    """
    Release the given resource to the given stage

    When `skip_sandbox_native_release` is not set and the `stage` is one of `ctt.ReleaseStatus` values
    we perform the native Sandbox release process as well. This is done to provide a compatibility layer
    between the new ('kosher') scheme and the default one. The same is done when cancelling a release.
    Release search method however does not rely on the native Sandbox release marker ('released' attribute).

    :param resource_id: Sandbox resource ID
    :param stage: release stage (can be either a `ctt.ReleaseStatus` field or any string
    :param sb_rest_client: Sandbox REST client
    :param skip_sandbox_native_release: skip Sandbox native release process (release button)
    :param cc: (optional) used for Sandbox native release
    :param message:  (optional) used for Sandbox native release
    :param subject:  (optional) used for Sandbox native release
    """

    release_attr_name = private.get_release_attr_name(stage=stage)
    release_attr_value = private.prepare_release_attr_value()

    resource_attribute_processor = sb_rest_client.resource[resource_id].attribute

    if has_release_attr(resource_id, stage=stage, sb_rest_client=sb_rest_client):
        sb_rest_method = resource_attribute_processor[release_attr_name].update
    else:
        sb_rest_method = resource_attribute_processor.create

    sb_rest_method(
        name=release_attr_name,
        value=release_attr_value,
    )

    if stage not in ctt.ReleaseStatus or skip_sandbox_native_release:
        return

    # Trigger Sandbox native release as well

    resource_task_id = sb_rest_client.resource[resource_id].read()["task"]["id"]

    sb_rest_client.release({
        "to": [],
        "params": {},
        "task_id": resource_task_id,
        "cc": cc,
        "message": message,
        "type": str(stage),
        "subject": subject,
    })


@private.with_sb_rest_client()
def cancel_release(resource_id, stage=ctt.ReleaseStatus.STABLE, sb_rest_client=None):
    # type: (STR_OR_INT, STAGE_TYPE, typing.Optional[rest.Client]) -> None
    """
    Cancel release (un-release) for the given resource at the given stage. Consider resource T
    released to stable. `cancel_release(T.id, 'stable')` will make sure that T is no longer considered
    to be stable. As a result, when searching for the latest stable release you are not getting T (rather
    you'd get the resource released to stable before T).

    In order to provide some kind of a compatibility layer with the Sandbox native release scheme
    we also release the given resource to 'cancelled' meaning that this release is considered to be
    bad, unsuitable for deployment. However one can always re-release previously cancelled resource
    which should work perfectly normal.

    The "release to cancelled" part is only done to the resources which already have the native Sandbox
    "released" attribute set.

    :param resource_id: Sandbox resource ID
    :param stage: release stage (can be either a `ctt.ReleaseStatus` field or any string
    :param sb_rest_client: Sandbox REST client
    """

    release_attr_name = private.get_release_attr_name(stage=stage)

    if not has_release_attr(resource_id, stage=stage, sb_rest_client=sb_rest_client):
        logging.warning("Resource #%s has never been released to %s", resource_id, str(stage))
        return

    sb_rest_client.resource[resource_id].attribute[release_attr_name].delete()

    if sb_rest_client.resource[resource_id].attribute["released"]:

        resource_task_id = sb_rest_client.resource[resource_id].task_id

        sb_rest_client.release({
            "task_id": resource_task_id,
            "message": "Cancel release",
            "type": "cancelled",
        })
