import typing  # noqa: UnusedImport
import datetime
import functools

from dateutil import tz

from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.projects.common import time_utils
from sandbox.common import rest


RELEASE_ATTR_NAME_PREFIX = "k_released_"
RESOURCE_ATTRIBUTES_KEY = "attributes"


def with_sb_rest_client(kwarg_key="sb_rest_client"):
    """
    Makes sure that the decorated function receives the desired Sandbox rest client object

    :param kwarg_key: This key is used to pass `rest.Client` object (as a kwarg) to the decorated function
    """

    def decorator(func):

        @functools.wraps(func)
        def wrapper(*args, **kwargs):

            sb_rest_client = kwargs.pop(kwarg_key, None)

            if sb_rest_client is None:
                sb_rest_client = rest.Client()

            kwargs[kwarg_key] = sb_rest_client

            return func(*args, **kwargs)

        return wrapper

    return decorator


def get_release_attr_name(stage=ctt.ReleaseStatus.STABLE):
    # type: (ctt.ReleaseStatus) -> str
    return "{}{}".format(RELEASE_ATTR_NAME_PREFIX, str(stage).lower())


def prepare_release_attr_value(dt_obj=None):
    # type: (typing.Optional[datetime.datetime]) -> str

    if not dt_obj:
        return time_utils.datetime_utc_iso(force_z=True)

    if dt_obj.tzinfo is None:
        raise ValueError("%s lacks timezone info")

    return time_utils.datetime_to_iso(dt_obj.astimezone(tz.tzutc()), force_z=True)


def find_release_creation_time_limited(
    resource_type,
    creation_time_limit,
    release_attr_name,
    limit,
    sb_rest_client,
    custom_attrs=None,
):
    # type: (str, typing.Optional[datetime.datetime], str, int, rest.Client, typing.Optional[dict]) -> typing.Optional[int]

    if creation_time_limit is None:
        creation_time_range = ""
    else:
        creation_time_range = "{}..{}".format(
            time_utils.datetime_to_iso(creation_time_limit, force_z=True),
            time_utils.datetime_utc_iso(force_z=True),
        )

    attrs = {
        release_attr_name: None,
    }

    if custom_attrs:
        attrs.update(custom_attrs)

    resources = sb_rest_client.resource.read(
        type=resource_type,
        status=ctr.State.READY,
        order="-id",
        attrs=attrs,
        limit=limit,
        created=creation_time_range,
    )["items"]

    if not resources:
        return

    latest_released_resource = max(
        resources,
        key=lambda resource_info: resource_info[RESOURCE_ATTRIBUTES_KEY][release_attr_name],
    )

    return latest_released_resource["id"]
