import click
import os

from sandbox.common import rest
from sandbox.projects.common import kosher_release


SANDBOX_TOKEN_ENV = "SANDBOX_TOKEN"


def get_sandbox_token():
    token = os.environ.get(SANDBOX_TOKEN_ENV)
    if not token:
        click.echo(f"No token found in env {SANDBOX_TOKEN_ENV}. Please provide one")
        exit(1)
    return token


def get_client():
    return rest.Client(auth=get_sandbox_token())


@click.group()
@click.pass_context
def main(ctx):
    pass


@main.command()
@click.argument("resource_id")
@click.argument("stage")
def release(resource_id, stage):
    """
    Release resource :param resource_id: to :param stage:
    """
    click.echo(f"Releasing {resource_id} to {stage}...")
    kosher_release.release(resource_id, stage, sb_rest_client=get_client())
    click.echo(f"Resource {resource_id} released to {stage}")


@main.command(context_settings=dict(
    ignore_unknown_options=True,
))
@click.argument("resource_type")
@click.argument("stage")
@click.argument("custom_attrs", nargs=-1, type=click.UNPROCESSED)
def find_release(resource_type, stage, custom_attrs):
    """
    Find release of :param resource_type: to :param stage:
    """
    if custom_attrs:
        custom_attrs = dict(item.split("=") for item in custom_attrs)
    else:
        custom_attrs = None
    resource_id = kosher_release.find_release(
        resource_type,
        stage,
        sb_rest_client=get_client(),
        custom_attrs=custom_attrs,
    )
    click.echo(resource_id)


if __name__ == "__main__":
    main(prog_name="k_release_tool")
