# coding: utf-8

from sandbox.projects.common import requests_wrapper

_LINK_TO_SB_ITEM = 'https://sandbox.yandex-team.ru/{item}/{id}/view'

_LINK_TO_ARCADIA = 'https://a.yandex-team.ru/{path}'
_LINK_TO_COMMIT = 'https://a.yandex-team.ru/{repo}/commit/{revision}'
_LINK_TO_REVIEW = 'https://a.yandex-team.ru/review/{review}/files'

_LINK_TO_RM_UI = 'https://rm.z.yandex-team.ru/component/{name}/{screen}'

_LINK_TO_ST = 'https://st.yandex-team.ru/{issue_id}'

_LINK_TO_STAFF = 'https://staff.yandex-team.ru/{nick}'

_LINK_TO_WIKI = "https://wiki.yandex-team.ru/{path}"

_LINK_TO_NANNY_RELEASE = "https://nanny.yandex-team.ru/ui/#/r/{release_request_id}/"

_LINK_TO_NANNY_SERVICE = "https://nanny.yandex-team.ru/ui/#/services/catalog/{service_id}/"

_LINK_TO_YAPPY_BETA = 'https://yappy.z.yandex-team.ru/b/{beta_name}'

_LINK_TO_YASM_PANEL = 'https://yasm.yandex-team.ru/template/panel/{}'


HREF_TO_ITEM = '<a href="{link}">{name}</a>'
WIKILINK_TO_ITEM = '(({link} {name}))'
ARC_HASH_LENGTH = 40


class LinkType(object):
    plain = 1
    wiki = 2
    href = 3


def format_link(link_type, link, name=None):
    if link_type is LinkType.wiki:
        return WIKILINK_TO_ITEM.format(link=link, name=name)
    if link_type is LinkType.href:
        return HREF_TO_ITEM.format(link=link, name=name)
    return link


def sb_item_link(item_id, item_name, link_name=None, plain=False):
    """
        Link to task or resource in sandbox UI
        :param item_id: Task id or resource id
        :param item_name: Either "task" or "resource"
        :param link_name: If None, use item_id as link name
        :param plain: Link without href. In this case link_name is not used.
    """
    link = _LINK_TO_SB_ITEM.format(
        item=item_name,
        id=item_id,
    )
    if plain:
        return link

    return HREF_TO_ITEM.format(
        link=link,
        name=link_name or item_id,
    )


def sb_item_wiki_link(item_id, item_name, link_name=None):
    """
        Link to sandbox item (task or resource) in wiki-format
        :param item_id: Task id or resource id
        :param item_name: Either "task" or "resource"
        :param link_name: If None, use item_id as link name
    """
    return WIKILINK_TO_ITEM.format(
        link=_LINK_TO_SB_ITEM.format(
            item=item_name,
            id=item_id,
        ),
        name=link_name or item_id,
    )


def wiki_page_link(path, plain=False, name=None):
    link = _LINK_TO_WIKI.format(path=path)
    if plain:
        return link
    return HREF_TO_ITEM.format(link=link, name=name or path)


def wiki_link_from_arcadia_path(path, name=None):
    """
        Link to wiki-link arcadia path.
        :param path: String, should starts with REPO_NAME.
        :return: link
    """
    if name is None:
        return WIKILINK_TO_ITEM.format(link=_LINK_TO_ARCADIA.format(path=path), name=path)
    return WIKILINK_TO_ITEM.format(link=_LINK_TO_ARCADIA.format(path=path), name=name)


def task_link(task_id, link_name=None, plain=False):
    return sb_item_link(task_id, "task", link_name, plain)


def resource_link(resource_id, link_name=None, plain=False):
    return sb_item_link(resource_id, "resource", link_name, plain)


def task_wiki_link(task_id, link_name=None):
    return sb_item_wiki_link(task_id, "task", link_name)


def resource_wiki_link(resource_id, link_name=None):
    return sb_item_wiki_link(resource_id, "resource", link_name)


def st_wiki_link(issue_id, link_name=None):
    link_to_st = st_link(issue_id, plain=True)
    return WIKILINK_TO_ITEM.format(
        link=link_to_st,
        name=link_name or issue_id,
    )


def revision_link(revision, link_name=None, link_type=LinkType.href):
    """
        Link to revision or arc_hash in html/wiki/plain format
    """
    if len(str(revision)) == ARC_HASH_LENGTH:
        return format_link(link_type, _LINK_TO_COMMIT.format(repo="arc_vcs", revision=revision), link_name or revision)
    else:
        return format_link(link_type, _LINK_TO_COMMIT.format(repo="arc", revision=revision), link_name or revision)


def staff_link(employee, link_name=None, link_type=LinkType.href):
    """ Link to staff """
    return format_link(link_type, _LINK_TO_STAFF.format(nick=employee), link_name or employee)


def rm_ui_link(component, screen="manage", link_name=None, link_type=LinkType.plain):
    """
    Link to Release Machine viewer component page
    :param component: component name in rm viewer
    :param screen: screen name in UI
    :param link_name: component name or arbitrary string
    :param link_type: LinkType
    :return: string, formatted by chosen type
    """
    return format_link(link_type, _LINK_TO_RM_UI.format(name=component, screen=screen), link_name or component)


def review_link(review, link_name=None, link_type=LinkType.href):
    """
        Link to review
    """
    return format_link(link_type, _LINK_TO_REVIEW.format(review=review), link_name or review)


def st_link(issue_id, link_name=None, plain=False):
    link = _LINK_TO_ST.format(issue_id=issue_id)
    if plain:
        return link
    return HREF_TO_ITEM.format(link=link, name=link_name or issue_id)


def href_to_arcadia(arcadia_path, link_name=None):
    url = _LINK_TO_ARCADIA.format(path=arcadia_path)
    if not link_name:
        link_name = url
    return HREF_TO_ITEM.format(
        link=url,
        name=link_name
    )


def yappy_beta_link(beta_name, link_name=None, plain=False):
    link = _LINK_TO_YAPPY_BETA.format(beta_name=beta_name)
    if plain:
        return link
    return HREF_TO_ITEM.format(link=link, name=link_name or beta_name)


def yappy_beta_status_chart_link(beta_name, link_name=None, time_from=None, time_to=None, plain=False):
    """
    Builds a link to the given beta consistency Yasm chart

    :param beta_name: Yappy beta name
    :param link_name: the result link name (if None then `beta_name` is used)
    :param time_from: if provided sets the left bound for the time range of the chart; should be given in milliseconds
    :param time_to: if provided sets the right bound for the time range of the chart; should be given in milliseconds
    :param plain: when True a plain link is returned, otherwise HTML <a>-tag is constructed and returned
    :return: str
    """

    link = _LINK_TO_YASM_PANEL.format(
        'yappy-beta-status/beta_name={beta_name}?from={time_from}&to={time_to}'.format(
            beta_name=beta_name,
            time_from=time_from or '',
            time_to=time_to or '',
        )
    )

    if plain:
        return link
    return HREF_TO_ITEM.format(
        link=link,
        name=link_name or beta_name,
    )


def nanny_wiki_link(release_request_id, link_name=None):
    """ Link to nanny release request """
    return WIKILINK_TO_ITEM.format(
        link=_LINK_TO_NANNY_RELEASE.format(release_request_id=release_request_id),
        name=link_name or release_request_id,
    ) if release_request_id else ""  # todo(dkvasov@): replace with nanny_link


def nanny_link(release_req_id, link_name=None, link_type=LinkType.href):
    return format_link(link_type, _LINK_TO_NANNY_RELEASE.format(release_request_id=release_req_id), link_name or release_req_id)


def nanny_service_wiki_link(service_id, link_name=None):
    """ Link to nanny service """
    return WIKILINK_TO_ITEM.format(
        link=_LINK_TO_NANNY_SERVICE.format(service_id=service_id),
        name=link_name or service_id,
    ) if service_id else ""  # todo(dkvasov@): replace with nanny_link


def nanny_service_link(service_id, link_name=None, link_type=LinkType.href):
    return format_link(link_type, _LINK_TO_NANNY_SERVICE.format(service_id=service_id), link_name or service_id)


def short_url(url_to_short):
    """
    Returns shortened url (uses nda.ya.ru)
    """
    return requests_wrapper.get_r("https://nda.ya.ru/--?url={}".format(url_to_short)).text
