# -*- coding: utf-8 -*-

import os
import shutil
import logging

from sandbox.projects.common import msparser
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.utils import splitthousands
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder
from sandbox.projects.common.environments import ValgrindEnvironment
from sandbox.sandboxsdk.process import run_process


def get_peak_mem_size(file_name):
    logging.info('get_peak_mem_size("{0}")'.format(file_name))

    data = msparser.parse_file(file_name)
    logging.info('file parsed, data = {0}'.format(data))

    if 'peak_snapshot_index' not in data:
        raise SandboxTaskFailureError("massif output is empty")
    peak_index = data['peak_snapshot_index']

    peak_snapshot = data['snapshots'][peak_index]

    mem_size = peak_snapshot['mem_heap'] + peak_snapshot['mem_stack'] + peak_snapshot['mem_heap_extra']

    return mem_size


_OUT_FILE_NAME = 'massif.out'


def generate_task(
    params_collection,
    base_class,
    start_timeout=sc.DEFAULT_START_TIMEOUT * 2,
    shutdown_timeout=sc.DEFAULT_SHUTDOWN_TIMEOUT * 4,
    tune_archive_settings=True,
):
    """
        generates task class based on parent task class
        to run search component with valgrind massif tool
        base class should have stub method init_search_component(self, component)
    """

    class MassifTaskPattern(base_class):
        type = None

        input_parameters = sc.tune_search_params(
            params_collection, base_class.input_parameters,
            start_timeout=start_timeout
        )

        def on_enqueue(self):
            base_class.on_enqueue(self)
            channel.task = self

            self.ctx['massif_out_resource_id'] = self.create_resource(
                self.descr,
                'massif_out',
                'VALGRIND_MASSIF_OUTPUT',
                arch='any'
            ).id

        def on_execute(self):
            resource, out_file_path = self._get_massif_out_resource_and_out_file_path()
            make_folder(resource.path)

            base_class.on_execute(self)

            process = run_process(['ms_print', out_file_path], timeout=60, log_prefix='ms_print')
            shutil.move(process.stdout_path, os.path.join(resource.path, 'massif.txt'))

            self.ctx['valgrind_mem_size'] = str(get_peak_mem_size(out_file_path))

        def _get_middlesearch_additional_params(self):
            return {
                "shutdown_timeout": shutdown_timeout,
            }

        def init_search_component(self, component):
            base_class.init_search_component(self, component)

            if tune_archive_settings:
                component.replace_config_parameter("Collection/UserParams/MemoryMappedArchive", None)
                component.replace_config_parameter("Collection/UserParams/ArchiveOpenMode", None)
                component.replace_config_parameter("Collection/UserParams/FileArchive-NoReuse", "")
            component.replace_config_parameter("Collection/CalculateBinaryMD5", 'no')

            resource, out_file_path = self._get_massif_out_resource_and_out_file_path()

            def cmd_patcher(cmd):
                ValgrindEnvironment().prepare()
                return ['valgrind', '--tool=massif', '--massif-out-file=' + out_file_path] + cmd

            component.run_cmd_patcher = cmd_patcher

        def _get_massif_out_resource_and_out_file_path(self):
            resource = channel.sandbox.get_resource(self.ctx['massif_out_resource_id'])
            out_file_path = os.path.join(resource.path, _OUT_FILE_NAME)
            return resource, out_file_path

        def get_results(self):
            if not self.is_completed():
                return 'Results are not ready yet.'

            return 'Mem: {}'.format(splitthousands(self.ctx.get('valgrind_mem_size')))

        def get_short_task_result(self):
            if not self.is_completed():
                return None

            return splitthousands(int(self.ctx['valgrind_mem_size']) / 1000000) + 'Mb'

    return MassifTaskPattern
