import os.path
import copy
import gzip
import logging
import subprocess
import time

from sandbox import common
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import process

from sandbox.projects.images.bans import ImagesTestMiddlesearchBan as test_task
from sandbox.projects.common import apihelpers
from sandbox.projects.common import decorators
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.mediasearch import yasm_task


_BUILD_TASK_ID = "build_task_id"
_BUILDER_FLAG = "builder_flag"
_RELEASE_SIGNAL = "release_signal"
_MONITORING_OPTION = "enable_push_monitoring"
_TEST_TASK_IDS = "test_task_ids"
_BASE_RELEASE_BAN_PARAMS_GROUP = "Base mediasearch ban release parameters"


class MaxTotalSizeParameter(parameters.SandboxIntegerParameter):
    name = "max_total_size"
    description = "Max possible total vector size (MB)"
    default_value = 1500
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class ForceBuildParameter(parameters.SandboxBoolParameter):
    name = "force_build"
    description = "Force build"
    default_value = False
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class DontReleaseParameter(parameters.SandboxBoolParameter):
    name = "dont_release"
    description = "Don't release"
    default_value = False
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class ReleaseStatusParameter(parameters.SandboxRadioParameter):
    name = "ban_release_status"
    description = "Release status"
    choices = [(x, x) for x in sandboxapi.RELEASE_STATUSES]
    default_value = sandboxapi.RELEASE_STABLE
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class RunTestsParameter(parameters.SandboxBoolParameter):
    name = "run_tests"
    description = "Run tests"
    default_value = False
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class EnableMonitoringParameter(parameters.SandboxBoolParameter):
    name = _MONITORING_OPTION
    description = "Enable push monitoring via Push API"
    default_value = False
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class RequestLatestTask(parameters.SandboxBoolParameter):
    name = "request_latest_task"
    description = "Request the latest task for monitoring"
    default_value = True
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class EnableSemaphoreParameter(parameters.SandboxBoolParameter):
    name = "enable_semaphore"
    description = "Semaphore to run only one task of this type on time"
    default_value = False
    group = _BASE_RELEASE_BAN_PARAMS_GROUP


class BaseReleaseBanTask(task.ForcedBackupTask):

    input_parameters = (MaxTotalSizeParameter, ForceBuildParameter, DontReleaseParameter,
                        ReleaseStatusParameter, RunTestsParameter, EnableMonitoringParameter, RequestLatestTask, EnableSemaphoreParameter)

    release_subject = None
    release_comment = None
    release_resources = None
    release_signal_value = None
    task_signal_value = None

    enable_semaphore = False

    client_tags = ctc.Tag.Group.LINUX

    @staticmethod
    def create_input_parameters(max_total_size=1500, run_tests=False, enable_semaphore=False):

        class _MaxTotalSizeParameter(MaxTotalSizeParameter):
            default_value = max_total_size

        class _RunTestsParameter(RunTestsParameter):
            default_value = run_tests

        class _EnableSemaphoreParameter(EnableSemaphoreParameter):
            default_value = enable_semaphore

        return (_MaxTotalSizeParameter, ForceBuildParameter, DontReleaseParameter,
                ReleaseStatusParameter, _RunTestsParameter, EnableMonitoringParameter, RequestLatestTask, _EnableSemaphoreParameter)

    def on_enqueue(self):
        task.ForcedBackupTask.on_enqueue(self)
        if _BUILDER_FLAG not in self.ctx and self.ctx[EnableSemaphoreParameter.name]:
            semaphore_name = self.type + ".lock"
            self.semaphores(ctt.Semaphores(
                acquires=[ctt.Semaphores.Acquire(name=semaphore_name, weight=1, capacity=1)],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            ))
            logging.info("Taken semaphore '{}'".format(semaphore_name))

    def on_execute(self):
        # check if we're the build task
        if _BUILDER_FLAG in self.ctx:
            total_size = float(self._build_ban()) / (1024 * 1024)
            logging.info("Total size ban size is {} MB".format(total_size))
            if total_size > self.ctx[MaxTotalSizeParameter.name]:
                raise errors.SandboxTaskFailureError("Total vector size limit exceeded")
            return

        # run build task
        if _BUILD_TASK_ID not in self.ctx:
            build_ctx = copy.deepcopy(self.ctx)
            build_ctx[_BUILDER_FLAG] = True
            build_task = self.create_subtask(
                task_type=self.type,
                description=self.descr,
                input_parameters=build_ctx,
                inherit_notifications=True,
                priority=self.priority
            )
            self.ctx[_BUILD_TASK_ID] = build_task.id

        # wait for the build task to finish
        build_task = channel.sandbox.get_task(self.ctx[_BUILD_TASK_ID])
        if not utils.is_task_stop_executing(build_task):
            self.wait_all_tasks_stop_executing([build_task])
            return

        build_task_status = build_task.new_status
        if build_task_status == self.Status.SUCCESS:
            if self._has_resources(build_task):
                # test
                if self.ctx.get(RunTestsParameter.name):
                    if _TEST_TASK_IDS not in self.ctx:
                        self.ctx[_TEST_TASK_IDS] = self._test_ban(build_task.id)

                    for test_task_id in self.ctx[_TEST_TASK_IDS]:
                        test_task = channel.sandbox.get_task(test_task_id)
                        if not utils.is_task_stop_executing(test_task):
                            self.wait_all_tasks_stop_executing([test_task])
                            return
                        else:
                            if test_task.new_status != self.Status.SUCCESS:
                                self._check_signals(False)
                                raise errors.SandboxTaskFailureError("Test task failed")

                # release
                if not self.ctx.get(DontReleaseParameter.name):
                    self.create_release(build_task.id,
                                        status=self.ctx.get(ReleaseStatusParameter.name),
                                        subject=self.release_subject.format(timestamp=int(time.time())),
                                        comments=self.release_comment,
                                        addresses_to=self.release_mailto)
                    self.wait_all_tasks_stop_executing([build_task])
                    return
        elif build_task_status != self.Status.RELEASED:
            self._check_signals(False)
            raise errors.SandboxTaskFailureError("Build task failed")
        else:
            self.release_signal_value = build_task.ctx.get(_RELEASE_SIGNAL, None)

        # successfully released
        self._check_signals()

    def _has_resources(self, build_task):
        for resource in self.release_resources:
            if not apihelpers.list_task_resources(build_task.id, resource):
                logging.info("No output resources was found")
                return False
        return True

    def _get_latest_task(self, status, check_build_id=False):
        tasks = channel.sandbox.list_tasks(task_type=self.type, status="FINISHED", get_id_only=True)
        for task_id in sorted(tasks, key=lambda x: int(x), reverse=True):
            check_task = channel.sandbox.get_task(task_id)
            if check_task and check_task.new_status == status:
                if not check_task.ctx.get(_MONITORING_OPTION, None):
                    continue
                if check_build_id and _BUILD_TASK_ID not in check_task.ctx:
                    continue
                return check_task
        return None

    def _check_signals(self, success=True):
        if not utils.get_or_default(self.ctx, EnableMonitoringParameter):
            return

        if success:
            if self.timestamp_start:
                self.task_signal_value = int(self.timestamp_start)
            else:
                logging.error("Not found timestamp_start of the current task")

        if utils.get_or_default(self.ctx, RequestLatestTask):
            if self.release_signal_value is None:
                latest_released_task = self._get_latest_task(self.Status.RELEASED)
                if latest_released_task:
                    logging.info("Using the results of the latest released task {} now".format(latest_released_task.id))
                    self.release_signal_value = latest_released_task.ctx.get(_RELEASE_SIGNAL, None)
                else:
                    logging.error("Not found the latest released task")

            if self.task_signal_value is None:
                latest_task = self._get_latest_task(self.Status.SUCCESS, True)
                if latest_task:
                    if latest_task.timestamp_start:
                        logging.info("The latest successful task {} now".format(latest_task.id))
                        self.task_signal_value = int(latest_task.timestamp_start)
                    else:
                        logging.error("Not found timestamp_start of the latest successful task {}".format(latest_task.id))
                else:
                    logging.error("Not found the latest successful task")

        self._monitor_ban()

    def _skip_build(self, version_path, version_resource):
        if self.ctx.get(ForceBuildParameter.name):
            return False

        released_resource = apihelpers.get_last_released_resource(version_resource)
        if not released_resource:
            return False

        released_path = self.sync_resource(released_resource.id)

        with open(version_path) as version_file:
            with open(released_path) as released_file:
                return version_file.read().strip() == released_file.read().strip()

    def _tool(self, resource_type, parameter_name=None):
        if parameter_name is not None and self.ctx.get(parameter_name):
            tool_id = self.ctx[parameter_name]
        else:
            tool_id = utils.get_and_check_last_released_resource_id(resource_type, arch=sandboxapi.ARCH_LINUX)
        return self.sync_resource(tool_id)

    def _build_ban(self):
        raise NotImplementedError()

    def _test_ban(self, build_task_id):
        raise NotImplementedError()

    def _monitor_ban(self):
        pass

    def _set_release_signal(self, signal_value=None):
        if signal_value is None:
            signal_value = int(time.time())
        self.ctx[_RELEASE_SIGNAL] = signal_value

    def _register_ban(self, *args, **kwargs):
        resource = self.create_resource(*args, **kwargs)
        self.mark_resource_ready(resource.id, force_backup=True)


class ImagesBaseReleaseBanTask(yasm_task.YasmTask,
                               nanny.ReleaseToNannyTask,
                               BaseReleaseBanTask):
    """
        Builds rotten ban resources for Yandex.Images service
    """

    release_mailto = "images-data-releases@yandex-team.ru"

    execution_space = 4 * 1024
    cores = 1
    required_ram = 4 * 1024

    @property
    def nanny_token(self):
        return self.get_vault_data('IMAGES-SANDBOX', 'nanny-oauth-token')

    def _monitor_ban(self):
        # TemporaryError is not compatible with 'Sequence run' schedulers
        if self.task_signal_value is not None:
            self._yasm_notify(value=self.task_signal_value)
        if self.release_signal_value is not None:
            self._yasm_notify(task_status="released", value=self.release_signal_value, signal_ttl=365*86400)

    def _test_task(self, build_task_id, resource_type, rule_name="ImgFilterBanned", max_removal_delta=None):
        new_ban_resources = apihelpers.list_task_resources(build_task_id, resource_type)
        if not new_ban_resources:
            raise errors.SandboxTaskFailureError('Failed to find new ban resource {}'.format(resource_type))
        sub_ctx = {
            test_task.FirstBanParameter.name: utils.get_and_check_last_released_resource_id(resource_type),
            test_task.SecondBanParameter.name: new_ban_resources[0].id,
            test_task.RuleNameParameter.name: rule_name,
        }
        if max_removal_delta is not None:
            sub_ctx[test_task.MaxRemovalDeltaParameter.name] = max_removal_delta
        return self.create_subtask(
            task_type=test_task.ImagesTestMiddlesearchBan.type,
            description=self.descr,
            input_parameters=sub_ctx,
            inherit_notifications=True
        ).id


class VideoBaseReleaseBanTask(BaseReleaseBanTask):
    """
        Builds rotten ban resources for Yandex.Video service
    """

    _SERVICE_ID = "production_vidban"

    release_mailto = "video-monitoring@yandex-team.ru"

    @staticmethod
    def _is_changed_resource(resource_type, skynet_id=None, path=None):
        if not skynet_id:
            skynet_id = common.share.skynet_share(os.path.dirname(path), os.path.basename(path))
        released_data = apihelpers.get_last_released_resource(resource_type)
        if released_data and released_data.skynet_id == skynet_id:
            return False
        return True

    def _update_resource(self, resource_type, skynet_id=None, path=None):
        should_create = self.ctx.get(ForceBuildParameter.name)
        should_create = should_create or self._is_changed_resource(resource_type, skynet_id=skynet_id, path=path)
        if not should_create:
            logging.info("No new {} found".format(resource_type.basename))
            return None
        logging.info("New {} found".format(resource_type.basename))

        if resource_type not in self.release_resources:
            self.release_resources = self.release_resources + (resource_type,)
            logging.info("Release resources updated: {}".format(self.release_resources))

        return self.create_resource(self.descr, path, resource_type)

    @staticmethod
    def _open_file(path):
        if path.endswith(".gz"):
            return gzip.open(path)
        return open(path, "r")

    @classmethod
    def _format_file(cls, src_path, dst_path, format_func, append=False):
        with cls._open_file(src_path) as input, open(dst_path, "a" if append else "w") as output:
            for line in input:
                formated_line = format_func(line)
                if formated_line:
                    output.write(formated_line)


@decorators.retries(max_tries=3, delay=1)
def grimhold_call(grimhold_tool, grimhold_cfg, grimhold_data, probe_only=False):
    grimhold_args = [
        grimhold_tool,
        "-c", grimhold_cfg,
        "extract",
        "-s", os.path.basename(grimhold_data),
        "-d", os.path.dirname(grimhold_data),
    ]

    if not probe_only:
        process.run_process(grimhold_args, outputs_to_one_file=False, log_prefix="grimhold", check=True)
        return grimhold_data

    grimhold_proc = process.run_process(
        grimhold_args + ["-p"], stdout=subprocess.PIPE, wait=False,
        outputs_to_one_file=False, log_prefix="grimhold_probe")
    stdout, stderr = grimhold_proc.communicate()
    if grimhold_proc.returncode:
        raise errors.SandboxTaskFailureError("Failed to get grimhold rbtorrent")
    return stdout.strip()
