import os
import itertools
import logging

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk import parameters

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.build.YaMake as build_task


class BuildBinariesParameter(parameters.SandboxBoolParameter):
    name = 'build_binaries'
    description = 'Build binaries'
    default_value = True


class BuildConfigsParameter(parameters.SandboxBoolParameter):
    name = 'build_configs'
    description = 'Build configs'
    default_value = True


class BaseMediaBuildTask(build_task.YaMakeTask):
    type = None

    client_tags = ~ctc.Tag.Group.OSX
    cores = 24

    input_parameters = build_params.get_arcadia_params() \
        + build_task.ya_make_build_params() \
        + [BuildBinariesParameter, BuildConfigsParameter] \
        + build_params.get_aapi_parameters_as_default()

    def get_binaries(self):
        """Returns list of binary resources to build and pack"""

        raise NotImplementedError()

    def get_configs(self):
        """Returns list of configuration files to pack"""

        return []

    def get_branch_overrides(self, branch):
        return {}

    def get_targets(self):
        return [os.path.dirname(elt[0]) for elt in self.get_binaries_if_selected()]

    def get_arts(self):
        return [{"path": elt[0], "dest": self.get_outdir()}
                for elt in self.get_binaries_if_selected()]

    def get_arts_source(self):
        return [{"path": elt[0], "dest": self.get_outdir()}
                for elt in self.get_configs_if_selected()]

    def get_resources(self):
        result = {}
        resource_iterator = itertools.chain(self.get_binaries_if_selected(),
                                            self.get_configs_if_selected())
        for path, resource in resource_iterator:
            base_path = os.path.basename(path)
            result[base_path] = {
                "description": self.descr,
                "resource_path": os.path.join(self.get_outdir(), base_path),
                "resource_type": resource
            }
        return result

    def get_binaries_if_selected(self):
        if self.ctx[BuildBinariesParameter.name]:
            branch = self.ctx.get('ap_arcadia_branch')
            relocations = self.get_branch_overrides(branch)

            result = [(relocations.get(elt.arcadia_build_path, elt.arcadia_build_path), elt)
                      for elt in self.get_binaries()]
            logging.info("Selected binaries: %s" % "\n".join([str((i, j)) for i, j in result]))
            return result
        else:
            return []

    def get_configs_if_selected(self):
        if self.ctx[BuildConfigsParameter.name]:
            branch = self.ctx.get('ap_arcadia_branch')
            relocations = self.get_branch_overrides(branch)

            return ((relocations.get(elt.arcadia_path, elt.arcadia_path), elt)
                    for elt in self.get_configs())
        else:
            return []

    def get_outdir(self):
        return "out"
