# coding: utf-8
import json
import time
import urllib
import requests
import datetime

# Internal settings
_FETCH_TIMEOUT = 3.0


class RPSLimiter:
    "Inserts delays with respect to the specified RPS limit"

    def __init__(self, limit):
        "limit -- RPS"

        self.min_gap = 1.0/limit
        self.last = time.time()

    def gap(self):
        "Sleep before the next restricted event"

        t = time.time()
        gap_de_facto = t - self.last
        if gap_de_facto < self.min_gap:
            time.sleep(self.min_gap - gap_de_facto)
        self.last = time.time()


class StringLogger:
    "Accumulate log events in a string"

    def __init__(self):
        self.messages = ""

    def info(self, msg, *args):
        self.messages += "{} {}\n".format(datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S,%f')[:-3], msg % args)


class Counter(dict):
    def inc(self, key):
        self[key] = self.get(key, 0) + 1


class BaseControl(object):
    """
    Базовый класс логики задачи

    Символы класса используются для настройки и управления
    Экземпляр класса создаётся в процессах для обработки одного запроса
    """

    email_notification = True
    email_subject_template = "[online] {descr} results"

    chunk_size = 1
    rps_per_process = 10

    default_notification_list = ""
    default_max_rps = 100
    default_queries_limit = 1000

    default_build_output = False
    default_ordered_output = False

    # bool(default_query_url_template) != True
    # leads to corresponding task parameter
    default_query_url_template = None

    # -- Обязательно определить в дочернем классе

    @staticmethod
    def iterate_queries(input_path, queries_limit):
        """
        Итерация запросов входного файла
        NB staticmethod

        :input_path: Path to read input from
        :queries_limit: Max queries to generate

        Yields:
        * query -- string to be urlencoded and substituted in the template
        * (query, data)
            data -- value to be passed to process_query
        query is not None (which is used as a special value in wrapper)
        """
        raise NotImplementedError()

    def process_query(self, query_url):
        raise NotImplementedError()

    @staticmethod
    def template_data(counter):
        raise NotImplementedError()

    # -- Опционально
    @classmethod
    def prepare(cls, extra_resource, context):
        """Start-process callback

        Executed in worker process upon start.
        :param extra_resource: is a path
        :param context: is a dictionary (task.ctx in Sandbox)
        Both params are optional and can be None.
        """

        pass

    # Базовая функциональность

    def __init__(self, query_url_template=None):
        self.logger = StringLogger()
        self.limiter = RPSLimiter(self.rps_per_process)
        self.counter = Counter()
        self.output = ""
        self.query_url_template = query_url_template

    def create_query_url(self, query):
        return self.query_url_template % urllib.quote(query)

    def fetch(self, url, on_404, on_error, timeout=_FETCH_TIMEOUT):
        "Fetching URL, counting, raising problems"
        try:
            code_404 = False
            self.limiter.gap()
            r = requests.get(url, verify=False, timeout=timeout)
            self.logger.info("%d %s", r.status_code, url)
            if r.status_code == 404:
                code_404 = True
            r.raise_for_status()
            return r.content
        except Exception as e:
            self.logger.info("XXX %s %s %s", url, str(type(e)), str(e))
            if code_404:
                self.counter.inc(on_404)
            else:
                self.counter.inc(on_error)
            raise

    def fetch_json(self, url, on_404, on_error, timeout=_FETCH_TIMEOUT):
        return json.loads(self.fetch(url, on_404, on_error, timeout))

    def run_query(self, url, query_data):
        try:
            if query_data:
                self.logger.info("Context: %s", str(query_data))
            self.counter.inc("total_queries")
            self.process_query(url)
        except Exception as e:
            self.logger.info("Error: %s", e)
            self.counter.inc("any_query_fail")
        finally:
            self.logger.info("%s %s", url, repr(self.counter))
            return (self.counter, self.logger.messages, self.output)
