# coding: utf-8
"""
Turn class inheriting control.Basecontrol into Sandbox Task
"""
from sandbox.sdk2.helpers import misc
from sandbox.common.types.misc import OSFamily
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import ResourceSelector, LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects.common import apihelpers

import wrapper

import os
import re
import json
import time
import jinja2


# Finish task when it's less than specified seconds till kill_timeout
_GAP_TO_ESCAPE_BEFORE_TIMEOUT = 60

# Resulting data
_RESULTS = "results"
_OUTPUT_RESOURCE_ID = "output_resource_id"

# Task paths
_OUTPUT_PATH = "output"
_INFO_LOG_PATH = "checker.log"
_DEBUG_LOG_PATH = "checker.debug.log"

# Input parameter names
_INPUT_RESOURCE_ID = "input_resource_id"
_QUERY_URL_TEMPLATE = "query_url_template"
_QUERIES_LIMIT = "queries_limit"
_RPS_LIMIT = "rps_limit"
_EXTRA_RESOURCE_ID = "extra_resource_id"
_NOTIFICATION_LIST = "notification_list"
_ATTRIBUTES = "attributes"
_ORDERED_OUTPUT = "ordered_output"
_BUILD_OUTPUT = "build_output"


class BaseOnlineTask(SandboxTask):

    cores = 1
    required_ram = 4096
    execution_space = 4096

    input_resource_type = None
    input_resource_description = "Plan"
    input_resource_attr_name = ""
    input_resource_attr_value = None
    output_resource_type = None
    extra_resource_type = None
    extra_resource_description = None
    ignore_built_output_flag = True

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        create_output = bool(self.output_resource_type) and (self.ignore_built_output_flag or self.ctx.get(_BUILD_OUTPUT))
        if create_output:
            self.ctx[_OUTPUT_RESOURCE_ID] = self.create_resource(
                self.descr,
                _OUTPUT_PATH,
                self.output_resource_type,
                arch=OSFamily.ANY,
                attributes=json.loads(self.ctx[_ATTRIBUTES])
            ).id

    def on_execute(self):
        self.ensure_input_resource()

        # Fetch query data
        query_data = self.sync_resource(self.ctx[_INPUT_RESOURCE_ID])
        queries_limit = int(self.ctx[_QUERIES_LIMIT])
        max_rps = self.ctx[_RPS_LIMIT]
        query_url_template = self.ctx.get(_QUERY_URL_TEMPLATE, None)

        extra_resource_id = self.ctx.get(_EXTRA_RESOURCE_ID)
        extra_resource = self.sync_resource(extra_resource_id) if extra_resource_id else None

        build_output = self.ctx.get(_BUILD_OUTPUT)
        ordered_output = self.ctx.get(_ORDERED_OUTPUT)

        # Log outputs
        with open(self.log_path(_INFO_LOG_PATH), "w") as info_log:
            with open(self.log_path(_DEBUG_LOG_PATH), "w") as debug_log:

                # Output resource (if expected)
                create_output = bool(self.output_resource_type) and (self.ignore_built_output_flag or build_output)
                with open(_OUTPUT_PATH, "w") if create_output else wrapper.none_stream() as output:

                    # Task status line
                    with misc.ProgressMeter("Charge...") as status:

                        # Iterate queries
                        it = wrapper.query_iterator(self.control, query_data, queries_limit)
                        for results in wrapper.multiprocess(it, self.control, max_rps, query_url_template,
                                                            info_log, debug_log, output, extra_resource, self.ctx, ordered_output):

                            # Submit results to ctx
                            self.ctx[_RESULTS] = results

                            # We'd prefer not to have resource rather than incomplete one
                            if not create_output:
                                time_passed = time.time() - self.updated
                                if self.ctx["kill_timeout"] - time_passed < _GAP_TO_ESCAPE_BEFORE_TIMEOUT:
                                    break

                            # Calculate progress
                            status.value = results.get("total_queries", 0)
                            status.maxval = queries_limit

        if self.control.email_notification:
            self._send_mail()

    def ensure_input_resource(self):
        if self.ctx.get(_INPUT_RESOURCE_ID):
            return

        resource = apihelpers.get_last_resource_with_attribute(
            self.input_resource_type,
            self.input_resource_attr_name,
            self.input_resource_attr_value
        )

        if not resource:
            raise SandboxTaskFailureError('Unable to find input resource')

        self.ctx[_INPUT_RESOURCE_ID] = resource.id

    def _send_mail(self):
        """Results notification via email"""

        emails = [x for x in re.split(r'[, ]+', self.ctx[_NOTIFICATION_LIST]) if len(x) > 0]

        if emails:
            message = self.footer
            channel.sandbox.send_email(
                emails, [],
                self.control.email_subject_template.format(descr=self.descr.strip()),
                message, content_type='text/html')

    @property
    def footer(self):
        """Task footer rendered by template"""

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))

        if _RESULTS in self.ctx:
            data_to_render = {"results": self.control.template_data(self.ctx[_RESULTS])}
        else:
            data_to_render = {}

        return env.get_template("footer.html").render(data_to_render)


def empower(control):
    def class_wrapper(cls):
        cls.type = re.sub('([a-z])([A-Z])', r'\1_\2', cls.__name__).upper()

        class InputResource(ResourceSelector):
            name = _INPUT_RESOURCE_ID
            description = cls.input_resource_description
            resource_type = cls.input_resource_type

        class QueryUrlTemplate(SandboxStringParameter):
            name = _QUERY_URL_TEMPLATE
            description = 'URL template (%s for query)'
            default_value = control.default_query_url_template
            required = True

            @classmethod
            def cast(cls, value):
                if value.count("%s") != 1:
                    raise ValueError
                return super(QueryUrlTemplate, cls).cast(value)

        class QueriesLimit(SandboxIntegerParameter):
            name = _QUERIES_LIMIT
            description = 'Number of top queries'
            default_value = control.default_queries_limit
            required = True

        class RPSLimit(SandboxIntegerParameter):
            name = _RPS_LIMIT
            description = 'Maximum total RPS'
            default_value = control.default_max_rps
            required = True

        class ExtraResource(LastReleasedResource):
            name = _EXTRA_RESOURCE_ID
            description = cls.extra_resource_description
            resource_type = cls.extra_resource_type

        class NotificationList(SandboxStringParameter):
            name = _NOTIFICATION_LIST
            description = 'Notification list (comma-separated)'
            default_value = control.default_notification_list

        class Attributes(SandboxStringParameter):
            name = _ATTRIBUTES
            description = 'Output resource attributes (JSON)'
            default_value = "null"

        class BuildOutput(SandboxBoolParameter):
            name = _BUILD_OUTPUT
            description = 'Build output'
            default_value = control.default_build_output
            required = True

        class OrderedOutput(SandboxBoolParameter):
            name = _ORDERED_OUTPUT
            description = 'Ordered output'
            default_value = control.default_ordered_output
            required = True

        input_parameters = tuple(cls.input_parameters) if hasattr(cls, "input_parameters") else ()
        input_parameters += (InputResource, QueriesLimit, RPSLimit)

        if control.default_query_url_template:
            input_parameters += (QueryUrlTemplate,)
        if control.email_notification:
            input_parameters += (NotificationList,)
        if cls.output_resource_type:
            input_parameters += (Attributes,)
        if cls.extra_resource_type and cls.extra_resource_description:
            input_parameters += (ExtraResource,)
        input_parameters += (BuildOutput, OrderedOutput,)

        cls.input_parameters = input_parameters
        cls.control = control

        return cls

    return class_wrapper
