# coding: utf-8
"""
Wrappers for iterating, multiprocessing, etc of online investigations
driven by control.BaseControl classes
"""
import functools
import contextlib
import multiprocessing
import sys


class Results(dict):
    def __iadd__(self, another):
        mykeys = set(self.iterkeys())
        anotherkeys = set(another.iterkeys())
        for key in mykeys & anotherkeys:
            self[key] += another[key]
        for key in anotherkeys - mykeys:
            self[key] = another[key]
        return self


def process_query(control, query_url_template, query_item):
    """
    Fetch SERP, fetch thumbs, calculate stats

    This method is used as a multiprocessing.Pool worker and must be
    module-level. Control is a module-level class for the same reason.
    IMPORTANT The whole system must be pickleable.

    :param query: search query text
    :return: counter dict, log messages string, output string
    """

    if query_item[0] is None:
        return (None, query_item[1], None)

    return control(query_url_template).run_query(query_item[0], query_item[1])


# TODO(anskor): return using Result class when sanbox fix issue with storing
# dict ancestor to task context (SANDBOX-5579)
def append_result(results, counter):
    mykeys = set(results.iterkeys())
    anotherkeys = set(counter.iterkeys())
    for key in mykeys & anotherkeys:
        results[key] += counter[key]
    for key in anotherkeys - mykeys:
        results[key] = counter[key]
    return results


def multiprocess(it, control, max_rps, query_url_template,
                 info_log_stream=None, debug_log_stream=None, output_stream=None, extra_resource=None, context=None, ordered_output=False):
    """
    Process queries from an iterator in multiple processes

    This method is shared by Sandbox Task class and command line method.

    :param it: Queries iterator
    :param log: Output stream for logs
    """

    process_number = min(max(1, int(max_rps)/control.rps_per_process), 100)
    pool = multiprocessing.Pool(process_number, control.prepare, [extra_resource, context])
    bound_process_query = functools.partial(process_query, control, query_url_template)
    # TODO(anskor): return using Result class when sanbox fix issue with storing
    # dict ancestor to task context (SANDBOX-5579)
    # results = Results()
    results = {}

    imap = pool.imap if ordered_output else pool.imap_unordered
    for (counter, messages, output) in imap(bound_process_query, it, control.chunk_size):

        if counter is None:
            if info_log_stream:
                info_log_stream.write("\n")
                info_log_stream.write(messages)
            if debug_log_stream:
                debug_log_stream.write("\n")
                debug_log_stream.write(messages)
            raise Exception(messages)

        # TODO(anskor): return using Result class when sanbox fix issue with storing
        # dict ancestor to task context (SANDBOX-5579)
        # results += counter
        append_result(results, counter)

        if info_log_stream and counter.get("any_query_fail", 0):
            info_log_stream.write("\n")
            info_log_stream.write(messages)
        if debug_log_stream:
            debug_log_stream.write("\n")
            debug_log_stream.write(messages)
        if output_stream:
            output_stream.write(output)

        yield results


def query_iterator(control, query_data_filename, queries_limit):
    """Auxiliary method creating proper query iterator"""

    try:
        for q in control.iterate_queries(query_data_filename, queries_limit):
            if isinstance(q, tuple):
                if len(q) > 1:
                    yield (q[0], q[1])
                else:
                    yield (q[0], None)
            else:
                yield (q, None)
    except:
        exc_info = sys.exc_info()
        yield (None, "Iterator error {0}: {1}\n".format(*exc_info))


@contextlib.contextmanager
def none_stream():
    """Can be used to pass None into "with" section"""
    yield None
