import logging
import time

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import task

from sandbox.projects.common import apihelpers
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny


_BUILD_TASK_ID = "build_task_id"
_TEST_TASK_IDS = "test_task_ids"
_RELEASE_GROUP = 'Release parameters'


class ForceBuildParameter(parameters.SandboxBoolParameter):
    name = "force_build"
    description = "Force build"
    group = _RELEASE_GROUP
    default_value = False


class DontReleaseParameter(parameters.SandboxBoolParameter):
    name = "dont_release"
    description = "Don't release"
    group = _RELEASE_GROUP
    default_value = False


class RunTestsParameter(parameters.SandboxBoolParameter):
    name = "run_tests"
    description = "Run tests"
    group = _RELEASE_GROUP
    default_value = False


class EnableMonitoringParameter(parameters.SandboxBoolParameter):
    name = "enable_push_monitoring"
    description = "Enable push monitoring via Push API"
    default_value = False
    group = _RELEASE_GROUP


class BaseReleaseTask(task.SandboxTask):

    input_parameters = (
        ForceBuildParameter,
        DontReleaseParameter,
        RunTestsParameter,
        EnableMonitoringParameter,
    )

    cores = 1
    required_ram = 4096
    execution_space = 4096

    release_subject = None
    release_comment = None
    release_resources = None
    release_mailto = "images-data-releases@yandex-team.ru"

    client_tags = ctc.Tag.Group.LINUX

    def on_execute(self):
        # build
        if _BUILD_TASK_ID not in self.ctx:
            self.ctx[_BUILD_TASK_ID] = self._do_build() if not self._skip_build() else None

        if not self.ctx[_BUILD_TASK_ID]:
            self._do_monitor()
            return

        build_task = channel.sandbox.get_task(self.ctx[_BUILD_TASK_ID])
        if not utils.is_task_stop_executing(build_task):
            self.wait_all_tasks_stop_executing([build_task])
            return

        build_task_status = build_task.new_status
        if build_task_status == self.Status.SUCCESS:
            # test
            if self.ctx.get(RunTestsParameter.name):
                if _TEST_TASK_IDS not in self.ctx:
                    self.ctx[_TEST_TASK_IDS] = self._do_test(build_task.id)

                for test_task_id in self.ctx[_TEST_TASK_IDS]:
                    test_task = channel.sandbox.get_task(test_task_id)
                    if not utils.is_task_stop_executing(test_task):
                        self.wait_all_tasks_stop_executing([test_task])
                        return
                    else:
                        if test_task.new_status != self.Status.SUCCESS:
                            raise errors.SandboxTaskFailureError("Test task failed")

            # release
            if self._can_release(build_task):
                self.create_release(build_task.id,
                                    subject=self.release_subject.format(timestamp=int(time.time())),
                                    comments=self.release_comment,
                                    addresses_to=self.release_mailto)
                self.wait_all_tasks_stop_executing([build_task])
                return
        elif build_task_status != self.Status.RELEASED:
            raise errors.SandboxTaskFailureError("Build task failed")

        if utils.get_or_default(self.ctx, EnableMonitoringParameter):
            self._do_monitor()

    def _can_release(self, build_task):
        if not build_task or self.ctx.get(DontReleaseParameter.name):
            return False

        for resource in self.release_resources:
            if not apihelpers.list_task_resources(build_task.id, resource):
                logging.info("Nothing to release")
                return False
        return True

    def _skip_build(self):
        if self.ctx.get(ForceBuildParameter.name):
            return False
        return True

    def _tool(self, resource_type, parameter_name=None):
        if parameter_name is not None and self.ctx.get(parameter_name):
            tool_id = self.ctx[parameter_name]
        else:
            tool_id = utils.get_and_check_last_released_resource_id(resource_type, arch=sandboxapi.ARCH_LINUX)
        return self.sync_resource(tool_id)

    def _do_build(self):
        raise NotImplementedError()

    def _do_test(self, build_task_id):
        raise NotImplementedError()

    def _do_monitor(self):
        raise NotImplementedError()


class BaseNannyTask(nanny.ReleaseToNannyTask):
    @property
    def nanny_token(self):
        return self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
