# coding: utf-8

import time
import logging

import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm
import sandbox.common.errors as ce
import sandbox.common.utils as cu

from sandbox import sdk2
from sandbox.projects.common.mediasearch import scraper as sc
from sandbox.projects.common.mediasearch import ui
from sandbox.projects.common import link_builder as lb

_DEFAULT_SCRAPER_VAULT_OWNER_NAME = "robot-images-st:scraper_token"
_SCRAPER_BATCH_URL_TMPL = "https://scraper.yandex-team.ru/batch/{batch_id}"
_LOG = logging.getLogger(__name__)
_SAVE_CONTEXT_INTERVAL = 60


class SimplyLastResource(sdk2.parameters.Resource):
    """
    Parameter that uses last resource of specified type if none is given
    """
    @cu.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs=cls.attrs,
            state=ctr.State.READY,
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class ScraperTask(sdk2.Task):
    """
    Base task for running batches in scraper and processing results
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 3522

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        description = "ScraperTask"
        max_restarts = 10
        kill_timeout = 7200

        # custom parameters
        poll_interval = sdk2.parameters.Integer("Delay between scraper polls (seconds)", default=300)
        sleep_wait = sdk2.parameters.Bool("Use sandbox WaitTime to sleep while scraper is running", default=True)
        scraper_token = sdk2.parameters.String("Scraper token Vault owner:name", default=_DEFAULT_SCRAPER_VAULT_OWNER_NAME)
        scraper_batch_id = sdk2.parameters.Integer("Scraper batch id (process only)", required=False, default=None)

    @property
    def stats(self):
        """
        Data for footer table
        """
        return sorted(self.Context.counters.iteritems())

    def backup_context(self):
        """
        Save context to server

        Footer data update as a side effect
        """
        now = int(time.time())
        if self.Context.last_saved_ts == ctm.NotExists or now - self.Context.last_saved_ts >= _SAVE_CONTEXT_INTERVAL:
            self.Context.save()
            self.Context.last_saved_ts = now

    def counter_inc(self, key, value=1):
        """
        Shortcut for incrementing counter by a value
        """
        self.counters[key] = self.counters.get(key, 0) + value

    @property
    def counters(self):
        """
        Reference to counters

        Feel free to update contents.
        Saves context to server if was last accessed more than a minute ago.
        """
        self.backup_context()
        return self.Context.counters

    @property
    def counters_ro(self):
        """
        Reference to counters

        Counters cannot be changed here. For use on server (footer etc).
        """
        return self.Context.counters

    def counter_percentage(self, numerator, divider, error="&mdash;"):
        """
        Render percentage of one counter relative to another

        First counter value is also rendered after percents.
        For use in result table in footer.
        """
        if self.Context.counters.get(divider):
            return "{:.2f}% ({})".format(
                100 * float(self.Context.counters.get(numerator, 0)) / self.Context.counters[divider],
                self.Context.counters.get(numerator, 0)
            )
        else:
            return error

    def counter_ratio(self, numerator, divider, error="&mdash;"):
        """
        Render ratio of one counter relative to another

        For use in result table in footer.
        """
        if self.Context.counters.get(divider):
            return "{:.2f}".format(
                float(self.Context.counters.get(numerator, 0)) / self.Context.counters[divider]
            )
        else:
            return error

    @property
    def footer(self):
        """
        Footer
        """
        if self.Context.scraper_batch_id == ctm.NotExists and self.Context.counters == ctm.NotExists:
            return "Don't panic"
        result = []

        if self.Context.counters != ctm.NotExists:
            result.append({
                "content": [{"Name": k, "Value": v} for k, v in self.stats],
                "helperName": ""
            })

        if self.Context.scraper_batch_id != ctm.NotExists:
            batch_info = self._scraper_batch_link
            if self.Context.scraper_batch_status != ctm.NotExists:
                batch_info += " (" + self.Context.scraper_batch_status + ")"
                batch_info += " " + ui.progress_bar(
                    self.Context.scraper_batch_completed,
                    self.Context.scraper_batch_requested
                )
            result.append({
                "content": batch_info,
                "helperName": ""
            })

        return result

    @property
    def _scraper_batch_link(self):
        """
        Link to scraper batch of this task
        """
        return lb.HREF_TO_ITEM.format(
            link=_SCRAPER_BATCH_URL_TMPL.format(batch_id=self.Context.scraper_batch_id),
            name="Scraper batch"
        )

    def queries_iterator(self, feedback):
        """
        Iterator that yields queries for scraper batch tasks

        :param feedback: Progress bar handle
        """
        raise NotImplementedError()

    @property
    def batch_properties(self):
        """
        Scraper batch properties description
        """
        raise NotImplementedError()

    def process_results(self, results_iterator, feedback):
        """
        Process scraper batch results

        :param results_iterator: Iterator that yields serps
        :param feedback: Progress bar handle
        """
        raise NotImplementedError()

    def results_processor(self, scraper, scraper_batch_id, feedback):
        """
        Process scraper batch results

        Default implementation
        """
        self.process_results(scraper.results_iterator(scraper_batch_id), feedback)

    def on_enqueue(self):
        super(ScraperTask, self).on_enqueue()
        if len(self.Parameters.scraper_token.split(":")) != 2:
            raise ce.TaskFailure(
                "Invalid Scraper token {} (\"owner:name\" format expected)".format(self.Parameters.scraper_token)
            )

    def on_execute(self):
        """
        Core logic
        """
        scraper_token = sdk2.Vault.data(*self.Parameters.scraper_token.split(":"))
        _LOG.debug("Scraper token length is %d", len(scraper_token))
        scraper = sc.Scraper(scraper_token)

        if self.Parameters.scraper_batch_id:
            self.Context.scraper_batch_id = self.Parameters.scraper_batch_id

        if self.Context.counters == ctm.NotExists:
            self.Context.counters = {}

        # Preparing scraper task and starting batch
        if self.Context.scraper_batch_id == ctm.NotExists:

            # Starting from the begining (for example after Exception)
            self.Context.counters.clear()
            self.set_info("[1/3] Preparing batch")

            with ui.context_action("[1/3] Preparing batch") as feedback:
                self.Context.scraper_batch_id = scraper.run_batch(
                    self.queries_iterator(feedback),
                    self.batch_properties
                )

            self.Context.counters_backup = self.Context.counters.copy()

            self.set_info("[2/3] Scraper work\n" + self._scraper_batch_link, do_escape=False)
            if self.Parameters.sleep_wait:
                raise sdk2.WaitTime(self.Parameters.poll_interval)
            else:
                ui.set_action("[2/3] Scraper work")

        # Waiting for scraper batch completion
        while True:
            digest = scraper.get_batch_status(self.Context.scraper_batch_id)

            self.Context.scraper_batch_status = digest["status"]
            self.Context.scraper_batch_completed = int(digest["completed-serps"])
            self.Context.scraper_batch_requested = int(digest["requested-serps"])

            # Scraper batch statuses
            # Terminal: COMPLETE, FAIL, CANCELED
            # Other: PAUSED, PROGRESS, CREATING

            if digest["status"] in ("PROGRESS", "PAUSED", "CREATING"):
                if self.Parameters.sleep_wait:
                    raise sdk2.WaitTime(self.Parameters.poll_interval)
                else:
                    ui.set_progress(
                        "[2/3] Scraper work ({})".format(digest["status"]),
                        int(digest["completed-serps"]),
                        int(digest["requested-serps"])
                    )
                    time.sleep(self.Parameters.poll_interval)

            elif digest["status"] == "COMPLETE":
                self.set_info("[3/3] Processing results")

                # Restore stats (for the case of restart with specified batch_id)
                if self.Context.counters_backup != ctm.NotExists:
                    self.Context.counters = self.Context.counters_backup.copy()
                else:
                    self.Context.counters = {}

                # Process scraper batch results
                with ui.context_action("[3/3] Processing results") as feedback:
                    feedback.set_total(int(digest["completed-serps"]))
                    self.results_processor(scraper, self.Context.scraper_batch_id, feedback)
                break

            else:
                raise ce.TaskError("Invalid scraper status: {}".format(digest["status"]))
