"""
Miscellaneous Sandbox UI hacks
"""
import sandbox.common.rest as cr
import datetime
import logging


def set_action(text):
    """
    Simply set current action to text
    """
    api = cr.Client()
    api.task.current.execution.update(action=text)


def progress_bar(done, total):
    """
    Task status pseudographics
    """

    percents = 100.0 * done / total

    completed = 20 * int(percents) / 100
    left = 20 - completed
    return "|" * completed + "." * left + " ({:.0f}%)".format(percents)


def set_progress(description, done, total):
    """
    Simply set current action to decription with progress bar
    """
    message = "{} {}".format(description, progress_bar(done, total))
    set_action(message)


def context_action(description):
    """
    Set current action for executing task.

    Based on current_action from sandbox
    :param description: string with action status description
    :return: ActionStatus context manager
    """
    class ActionStatus(object):
        """
        Progress bar handle
        """

        def __init__(self, description):
            """
            Init handle

            :param decription: Progress action title
            """
            self.description = description
            self.current_action = None
            self.total = None
            self.done = 0

        def __enter__(self):
            logging.info("[STARTED] %s", self.description)
            self.started = datetime.datetime.now()
            message = "{} started".format(self.description)
            self._update(message)
            return self

        def __exit__(self, *args, **kw):
            message = "{} finished".format(self.description)
            self._update(message)
            logging.debug("[FINISHED] %s", self.description)

        @property
        def _delta_from_start(self):
            delta = datetime.datetime.now() - self.started
            return str(delta).split('.')[0]

        def _update(self, description):
            """
            Internal: Render on Sandbox

            Time delta from start is added to the end of the text
            :param description: Text to render
            """
            try:
                if description != self.current_action:
                    if description:
                        set_action("{} [{}]".format(description, self._delta_from_start))
                    else:
                        set_action("")
                    self.current_action = description
            except Exception:
                logging.exception("Can not set task current action")

        def set_total(self, total):
            """
            Setup progress bar goal

            Thus you can omit setting total field in later progress updates.
            """
            self.total = total

        def action(self, action):
            """
            Current action notification
            """
            self._update("{} -- {}".format(self.description, action))

        def progress(self, done=None, total=None):
            """
            Advance progress bar
            """
            if total is None:
                total = self.total
            if done is None:
                self.done += 1
            else:
                self.done = done
            if total:
                message = "{} {}".format(self.description, progress_bar(self.done, total))
            else:
                message = "{} {} done".format(self.description, self.done)
            self._update(message)

    return ActionStatus(description)
