# -*- coding: utf-8 -*-

import json
import logging
import os
import requests

import sandbox.common.errors as err
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh

METRICS_URL = "https://metrics.yandex-team.ru/"
METRICS_LAUNCH_MANAGER_URL = "https://mlm.yandex-team.ru/"

REGIONS = {
    "RU": "ru",
    "UA": "ua",
    "BY": "by",
    "KZ": "kz",
    "TR": "com.tr",
    "ID": "com",
    "UZ": "uz",
    "WORLD": "ru",
}


class MLMError(Exception):
    pass


class MLMLaunchStartError(Exception):
    """
    Wraps all exception classes that should be retried during MLM launch start.
    See RMDEV-2235 for more details.
    """
    pass


class MetricsLauncher(object):
    def __init__(self, oauth_token, launch_manager=METRICS_LAUNCH_MANAGER_URL, user_report_server=METRICS_URL):
        self.session = requests.Session()
        self.session.headers["Authorization"] = "OAuth {}".format(oauth_token)
        self.session.headers["Accept"] = "application/json"
        self.session.headers["Content-Type"] = "application/json"
        self.launch_manager = launch_manager
        self.user_report_server = user_report_server

    def _get_url(self, *args):
        return os.path.join(self.launch_manager, *args)

    @decorators.retries(3, delay=300, exceptions=MLMLaunchStartError, raise_class=err.TemporaryError)
    def launch_template(self, data):
        launch_url = self._get_url("api", "launch-set")
        logging.info("Launch url: '%s'\nLaunch data:\n%s\n\n", launch_url, json.dumps(data, indent=2))

        try:
            resp = self.session.post(launch_url, data=json.dumps(data))
        except Exception as exc:
            # handle temporary errors, e.g network connection problems, see RMDEV-2235
            eh.log_exception("MLM start failed", exc)
            raise MLMLaunchStartError(str(exc))

        try:
            # please do not remove this debug info from logs
            json_object = json.loads(resp.text)
            text = json.dumps(json_object, indent=2)
            logging.debug("MLM response json:\n%s\n\n", text)
        except Exception:
            logging.debug("MLM response text (cannot parse json):\n%s\n\n", resp.text)

        # Turn task into FAILURE. This case of MLM response MUST NOT be retried,
        # it indicates some trouble with input parameters.
        eh.ensure(
            not 400 <= resp.status_code < 500,
            "{} Client Error: {} for url: {}".format(resp.status_code, resp.reason, resp.url),
        )

        if resp.status_code != 200:
            # strange response code, let's retry it
            raise MLMLaunchStartError("{} Error: {} for url: {}".format(resp.status_code, resp.reason, resp.url))

        return resp.json()

    @decorators.retries(3, delay=300, raise_class=MLMError)
    def get_launch_info(self, launch_id):
        return self._get_json(self._get_url("api", "launch-set", launch_id))

    @decorators.retries(3, delay=300, raise_class=MLMError)
    def get_launchset_tasks(self, launch_id):
        """
        Obtain launch tasks tree info with timings and stages info.
        Useful for RM Timeline.
        """
        return self._get_json(self._get_url('api', 'launch-set', launch_id, 'tasks'))

    @decorators.retries(3, delay=300, raise_class=MLMError)
    def get_launch_sla_info(self, launch_id):
        return self._get_json(self._get_url("qex", "launch-set", launch_id))

    @decorators.retries(3, delay=300, raise_class=MLMError)
    def get_qex_launch_info(self, launch_id):
        return self._get_json("{url}?launch-set-id={lsid}".format(url=self._get_url("qex", "launch"), lsid=launch_id))

    @decorators.retries(3, delay=10, raise_class=MLMError)
    def get_template_info(self, template_id):
        return self._get_json(self._get_url("api", "launch-set-template", template_id))

    @decorators.retries(3, delay=10, raise_class=MLMError)
    def stop_launch(self, launch_id):
        url = self._get_url("api", "launch-set", launch_id)
        return self._get_json(url, self.session.delete)

    def _get_json(self, url, session_method=None):
        """
        :param url: url to fetch
        :param session_method: method to use (self.session.get by default)
        """
        if session_method is None:
            session_method = self.session.get
        response = session_method(url)
        response.raise_for_status()
        return response.json()


def get_mlm_link(template_id=None, launch_id=None, critical_only=False):
    """
    :param template_id: main MLM launch id (aka launchset id)
    :param launch_id: Launch id in launchset (one square block on the top of MLM UI).
    When all parameters are empty, link to MLM root will be formed.
    """
    if template_id is None:
        return METRICS_LAUNCH_MANAGER_URL

    link = os.path.join(METRICS_LAUNCH_MANAGER_URL, "sets", template_id or "", launch_id or "")
    if critical_only:
        link += "?critical=true"

    return link
