import six
import logging
import json
import re

from sandbox.projects.common import time_utils


# TODO: use enums in code and in LaunchMetrics code

class LaunchStatus:
    COMPLETED = "COMPLETED"
    CANCELED = "CANCELED"


class LaunchFinalStatus:
    UNKNOWN = "UNKNOWN"
    CRITICAL = "CRITICAL"
    WARN = "WARN"


class MetricStatus:
    UNDEFINED = "UNDEFINED"
    VALID = "VALID"
    WARN = "WARN"
    CRITICAL = "CRITICAL"


def get_final_status(mlm_launch_info):
    return mlm_launch_info.get("readyFinalStatus")


def get_failed(launch_group, group_type):
    failed_tests = []
    for test in launch_group:
        failed_metrics = [m for m in test["metrics"] if m["status"] in ["WARN", "CRITICAL", "UNKNOWN"]]
        if not failed_metrics:
            continue

        unified_test = {
            "url": test["detailUrl"],
            "metrics": failed_metrics,
        }
        if group_type == "serp":
            unified_test["name"] = u"{} [{}]".format(test["name"], test["filterName"])
        elif group_type == "test":
            if test["variantName"]:
                unified_test["name"] = u"{} / {}".format(test["typeName"], test["variantName"])
            else:
                unified_test["name"] = test["typeName"]
        failed_tests.append(unified_test)

    return failed_tests


def get_failed_tests(region):
    serps = get_failed(region["diffQueryGroups"], "serp")
    tests = get_failed(region["diffAcceptanceTests"], "test")
    return serps + tests


def get_first_failed_launch_id(mlm_launch_info):
    if "launches" not in mlm_launch_info:
        return None

    for launch in mlm_launch_info["launches"]:
        launch_id = launch.get("id")
        if not launch_id:
            logging.debug("[get_first_failed_launch_id] Launch without id:\n%s\n\n", json.dumps(launch, indent=4))
            return None

        if "diffQueryGroups" not in launch and "diffAcceptanceTests" not in launch:
            # Seems to be an incomplete run (aka UB)
            logging.info("[get_first_failed_launch_id] No diff blocks found for launch `%s`", launch_id)
            return launch_id

        for test in launch["diffQueryGroups"] + launch["diffAcceptanceTests"]:
            # CRITICAL or FATAL metric value
            for metric in test["metrics"]:
                # FIXME: remove UNDEFINED
                if metric["status"] not in ["VALID", "WARN", "UNDEFINED"]:
                    logging.info(
                        "[get_first_failed_launch_id] Found failed metric: launch %s, metric:\n%s\n\n",
                        launch_id, json.dumps(metric, indent=4),
                    )
                    return launch_id

    return None


def get_launch_fails(mlm_launch_info):
    launch_fails = {}
    for launch_block in mlm_launch_info["launches"]:
        failed_tests = get_failed_tests(launch_block)
        if failed_tests:
            launch_fails[launch_block["name"]] = failed_tests
    logging.info("Failed launches count: %d", len(launch_fails))
    return launch_fails


def verify_ignore_list(metrics_ignore_list):
    for critical_key, date in six.iteritems(metrics_ignore_list):
        assert isinstance(critical_key, tuple)
        assert len(critical_key) in [2, 3]
        assert isinstance(date, six.string_types)
        assert re.match(r"\d\d\d\d-\d\d-\d\d", date)


def calculate_adjusted_final_status(
    final_status,
    launch_fails,
    metrics_ignore_list,
    current_date=time_utils.date_ymd(),
):
    """
    Check metrics in critical state by ignore list and adjust final launch status
    See details in RMDEV-3170
    """
    if final_status != LaunchFinalStatus.CRITICAL:
        # bypass all other statuses
        return final_status

    # all metrics are calculated, but some critical checks failed
    failures = False
    for block_name, failed_tests in six.iteritems(launch_fails):
        for unified_test in failed_tests:
            for m in unified_test["metrics"]:
                if m["status"] == "CRITICAL":
                    have_criticals = True

                    critical_key = block_name, unified_test["name"], m["metricName"]
                    logging.info("Critical metric: %s", critical_key)
                    deadline = metrics_ignore_list.get(critical_key)

                    if deadline is None:
                        # wildcarded test name check: all tests with same metric name
                        critical_key = block_name, m["metricName"]
                        deadline = metrics_ignore_list.get(critical_key)

                    if deadline is not None and current_date < deadline:
                        logging.info(
                            "Critical metric %s deadline not exceeded: %s < %s",
                            critical_key, current_date, deadline,
                        )
                        # this metric does not affect adjusted final status
                        continue

                    failures = True
    if failures:
        logging.info("Template final status was not affected by ignore list")
        return final_status

    assert have_criticals, (
        "At least one metric should be in critical state when template final status is CRITICAL. "
        "Please report to Release Machine Support."
    )

    logging.info("Template final status adjusted to WARN")
    return LaunchFinalStatus.WARN
