# -*- coding: utf-8 -*-

from sandbox.sandboxsdk import parameters as sp

import sandbox.common.types.client as ctc

from sandbox.projects.common import utils
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.base_search_quality import settings as bss


Basesearch1Params = sc.create_basesearch_params(1)
Basesearch2Params = sc.create_basesearch_params(2)

BASE1_PORT = sc.DEFAULT_BASESEARCH_PORT
BASE2_PORT = sc.DEFAULT_BASESEARCH_PORT + 1

"""
    SingleHost tasks family start 2 basesearches and one middlesearch,
    and they eat at least MIN_PHYSMEM_GB * 2 (plus middlesearch itself)
    See also SEARCH-1352.

    Why not just round 64? See https://rb.yandex-team.ru/arc/r/108987/
"""
META_MIN_PHYSMEM_GB = 61


class RunProfilerParameter(sp.SandboxBoolParameter):
    name = 'run_profiler'
    description = 'Run profiler'
    group = 'Profiling'
    default_value = False


class UseGPerfToolsParameter(sp.SandboxBoolParameter):
    name = 'use_gperftools'
    description = 'Use GPerfTools'
    group = 'Profiling'
    default_value = False


class IgnoreBadDHErrors(sp.SandboxBoolParameter):
    """
    When benchmarking int with production plan we should filter search requests only
    or ignore "Bad dh parameter" messages on underlying basesearches
    """
    name = 'ignore_bad_dh_errors'
    description = 'Ignore bad dh errors'
    group = 'Consistency'
    default_value = False


class ReplaceBasesearches(sp.SandboxBoolParameter):
    name = 'replace_basesearches'
    description = 'Replace basesearches in middlesearch config'
    sub_fields = {'true': [param.name for param in Basesearch1Params.params + Basesearch2Params.params]}
    default_value = True


PARAMS = (
    sc.DefaultMiddlesearchParams.params + (ReplaceBasesearches,) +
    Basesearch1Params.params + Basesearch2Params.params + (
        # profiling
        RunProfilerParameter,
        UseGPerfToolsParameter,
        IgnoreBadDHErrors,
    )
)


class MiddlesearchSingleHostTask(bugbanner.BugBannerTask):

    client_tags = ctc.Tag.LINUX_PRECISE

    execution_space = (
        bss.MAX_MIDDLE_CACHE_SIZE +
        bss.RESERVED_SPACE
    )
    required_ram = META_MIN_PHYSMEM_GB << 10

    # to shut up checker
    replace_basesearches = None
    basesearch1 = None
    basesearch2 = None

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.WebMiddleSearch)
        self.replace_basesearches = utils.get_or_default(self.ctx, ReplaceBasesearches)

        if self.replace_basesearches:
            param1, param2 = self._get_basesearch_params()
            port1, port2 = self._get_basesearch_ports_for_basesearch()
            self.basesearch1 = self._create_basesearch(param1, port1)
            self.basesearch2 = self._create_basesearch(param2, port2)
            sc.mkl.configure_mkl_environment(self.basesearch1)
            sc.mkl.configure_mkl_environment(self.basesearch2)

        middlesearch = self._create_middlesearch_component()
        sc.mkl.configure_mkl_environment(middlesearch)

        if self.replace_basesearches:
            self.basesearch1.start()
            self.basesearch2.start()
            self.basesearch1.wait()
            self.basesearch2.wait()

            ignore_bad_dh_errors = utils.get_or_default(self.ctx, IgnoreBadDHErrors)
            self.basesearch1.ignore_bad_dh_errors = ignore_bad_dh_errors
            self.basesearch2.ignore_bad_dh_errors = ignore_bad_dh_errors

        self._use_middlesearch_component(middlesearch)

        if self.replace_basesearches:
            self.basesearch2.stop()
            self.basesearch1.stop()

    def _create_middlesearch_component(self):
        params = self._get_middlesearch_additional_params()

        use_profiler = self.ctx.get(RunProfilerParameter.name, False)
        use_gperftools = self.ctx.get(UseGPerfToolsParameter.name, False)

        basesearches = None
        if self.replace_basesearches:
            base_port1, base_port2 = self._get_basesearch_ports_for_middlesearch()
            basesearches = [
                {
                    'basesearch_type': 'WEB',
                    'collection': self.basesearch1.get_collection(),
                    'hosts_and_ports': [('localhost', base_port1)]
                },
                {
                    'basesearch_type': 'FOREIGN',
                    'collection': self.basesearch2.get_collection(),
                    'hosts_and_ports': [('localhost', base_port2)]
                }
            ]

        middlesearch = sc.get_middlesearch(
            basesearches=basesearches,
            disable_timeouts=True,
            use_profiler=use_profiler or use_gperftools,
            use_gperftools=use_gperftools,
            **params
        )

        self.init_search_component(middlesearch)

        return middlesearch

    def _create_basesearch(self, params_, port_):
        return sc.get_basesearch(params=params_, port=port_)

    def _get_basesearch_params(self):
        return Basesearch1Params, Basesearch2Params

    def _get_basesearch_ports_for_basesearch(self):
        """
            override in subclasses
        """
        return BASE1_PORT, BASE2_PORT

    def _get_basesearch_ports_for_middlesearch(self):
        """
            override in subclasses
        """
        return self._get_basesearch_ports_for_basesearch()

    def _get_middlesearch_additional_params(self):
        """
            override in subclasses
        """
        return {}

    def init_search_component(self, search_component):
        """
            override in subclasses
        """
        pass

    def _use_middlesearch_component(self, middlesearch):
        """
            override in subclasses
        """
        raise Exception("not implemented")
