# -*- coding: utf-8 -*-
import tempfile
import urlparse
import logging

from .utils import double_quote_string, unpack_args, get_trailing_part
from sandbox.sandboxsdk.svn import Svn
from sandbox.sandboxsdk.parameters import SandboxRadioParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process


class VCS(object):
    class CommitInfo:
        def __init__(self, commit_id, author, date, message):
            self.commit_id = commit_id
            self.author = author
            self.date = date
            self.message = message

    REVISION_DELIMITER = '@@'
    cmd = ''

    def __init__(self, url, path):
        self.url = url
        self.path = path

    def get_url_rev(self):
        scheme, netloc, path, query, _ = urlparse.urlsplit(self.url)
        path, rev = self.REVISION_DELIMITER in path and path.rsplit(self.REVISION_DELIMITER, 1) or (path, None)
        url = urlparse.urlunsplit((scheme, netloc, path, query, ''))
        return url, rev

    def obtain(self):
        raise NotImplementedError

    def commit(self, message):
        raise NotImplementedError

    def switch_to_branch(self, branch_name):
        raise NotImplementedError

    def switch_to_dev_branch(self):
        raise NotImplementedError

    def create_branch(self, branch_name):
        raise NotImplementedError

    def create_tag(self, tag_name):
        raise NotImplementedError

    def get_tag_url(self, tag_name):
        raise NotImplementedError

    def get_log(self, end_commit_id=None, start_commit_id=None, limit=None):
        raise NotImplementedError


class GitVCS(VCS):
    DEFAULT_USER_NAME = 'Zomb Mobsearch'
    DEFAULT_USER_EMAIL = "zomb-mobsearch@yandex-team.ru"
    DEFAULT_DEV_BRANCH_NAME = 'master'

    cmd = 'git'

    def __init__(self, url, path, dry_run=False, dev_branch_name=DEFAULT_DEV_BRANCH_NAME,
                 user_name=DEFAULT_USER_NAME, user_email=DEFAULT_USER_EMAIL):
        super(GitVCS, self).__init__(url, path)
        self._dev_branch_name = dev_branch_name
        self._user_name = user_name
        self._user_email = user_email
        self._dry_run = dry_run
        self._current_branch_name = 'master'

    def obtain(self):
        url, rev = self.get_url_rev()
        logging.info('Cloning %s to %s' % (url, self.path))
        self._run_git('clone', url, self.path, work_dir=None)
        if rev is not None:
            self.switch_to_branch(rev)
        self._configure_user()

    def commit(self, message):
        logging.info('Commiting changes with message "%s"' % message)
        self._run_git('add', '-u')
        self._run_git('commit', '-m', message)
        self._push(self._current_branch_name)

    def switch_to_branch(self, branch_name):
        logging.info('Switching to %s' % branch_name)
        self._run_git('checkout', branch_name)
        self._current_branch_name = branch_name

    def switch_to_dev_branch(self):
        self.switch_to_branch(self._dev_branch_name)

    def create_branch(self, branch_name):
        logging.info('Creating branch %s' % branch_name)
        self._run_git('checkout', '-b', branch_name)
        self._push(branch_name, '--set-upstream')
        self._current_branch_name = branch_name

    def create_tag(self, tag_name):
        logging.info('Creating tag %s' % tag_name)
        self._run_git('tag', tag_name)
        self._push(tag_name)

    def get_tag_url(self, tag_name):
        return self.url + VCS.REVISION_DELIMITER + tag_name

    def get_log(self, end_commit_id=None, start_commit_id=None, limit=None):
        if start_commit_id is not None:
            revision_range_arg = '%s^...%s' % (start_commit_id, end_commit_id or 'HEAD')
        elif end_commit_id is not None:
            revision_range_arg = end_commit_id
        else:
            revision_range_arg = None
        git_args = ['log', '--format=%H{tab}%an <%ae>{tab}%ad{tab}%s %b'.format(tab='%x09')]
        if limit is not None:
            git_args.extend(['-n', str(limit)])
        if revision_range_arg is not None:
            git_args.append(revision_range_arg)
        with tempfile.TemporaryFile() as stdout:
            self._run_git(git_args, stdout=stdout)
            stdout.seek(0)
            log_lines = stdout.readlines()
        log_entries = []
        for line in log_lines:
            if not line.strip():
                continue
            parts = line.strip().split('\t', 3)
            log_entries.append(VCS.CommitInfo(parts[0], parts[1], parts[2], parts[3]))
        return log_entries

    def _run_git(self, *args, **kwargs):
        work_dir = kwargs['work_dir'] if 'work_dir' in kwargs else self.path
        stdout = kwargs['stdout'] if 'stdout' in kwargs else None
        run_process([self.cmd] + unpack_args(*args), self.cmd, work_dir=work_dir, stdout=stdout)

    def _configure_user(self):
        self._run_git('config', 'user.name', self._user_name)
        self._run_git('config', 'user.email', self._user_email)

    def _push(self, ref_name=None, *args):
        if self._dry_run:
            logging.info('Skipping push, dry run is enabled')
            return
        logging.info('Pushing to remote')
        git_args = ['push'] + unpack_args(*args)
        if ref_name is not None:
            git_args.extend(['origin', ref_name])
        self._run_git(git_args)


class SvnVCS(VCS):
    DEFAULT_REPOSITORY_ROOT = 'arcadia.yandex.ru/arc'
    DEFAULT_BRANCHES_PATH = '^/branches/mobile-report'
    DEFAULT_TAGS_PATH = '^/tags/mobile-report'
#    DEFAULT_TRUNK_PATH = '^/trunk/arcadia/mobile-report'

    cmd = 'svn'
    svn = Svn

    def __init__(self, url, path, root_path=DEFAULT_REPOSITORY_ROOT, branches_path=DEFAULT_BRANCHES_PATH,
                 tags_path=DEFAULT_TAGS_PATH, trunk_path=None):
        super(SvnVCS, self).__init__(url, path)
        if root_path not in url:
            raise ValueError('Repository root is not consistent with repository URL')
        self.root_path = root_path
        self.branches_path = branches_path
        self.tags_path = tags_path
        self.trunk_path = trunk_path or self._get_trunk_path_from_url(url, root_path)

    @staticmethod
    def _get_trunk_path_from_url(url, repository_root):
        return '^' + get_trailing_part(url, repository_root.rstrip('/'))

    def obtain(self):
        url, rev = self.get_url_rev()
        logging.info('Checking out %s (at %s) to %s' % (url, rev or 'HEAD', self.path))
        self.svn.checkout(self.url, self.path, revision=rev)

    def commit(self, message):
        logging.info('Commiting changes with message "%s"' % message)
        self.svn.commit(self.path, message)

    def switch_to_branch(self, branch_name):
        self._switch('%s/%s' % (self.branches_path, branch_name))

    def switch_to_dev_branch(self):
        self._switch(self.trunk_path)

    def create_branch(self, branch_name):
        self._copy('.', '%s/%s' % (self.branches_path, branch_name), 'Created branch %s' % branch_name)

    def create_tag(self, tag_name):
        self._copy('.', '%s/%s' % (self.tags_path, tag_name), 'Created tag %s' % tag_name)

    def get_tag_url(self, tag_name):
        scheme = urlparse.urlsplit(self.url).scheme
        tag_path = self.tags_path.lstrip('^').rstrip('/') + '/' + tag_name
        return urlparse.urlunsplit((scheme, self.root_path.rstrip('/'), tag_path, None, None))

    def get_log(self, end_commit_id=None, start_commit_id=None, limit=None):
        revision_from = end_commit_id or 'BASE'
        revision_to = start_commit_id or 1
        svn_log = self.svn.log(self.path, revision_from, revision_to, limit=limit)
        return map(lambda r: VCS.CommitInfo(r['revision'], r['author'], r['date'], r['msg']), svn_log)

    def _run_svn(self, *args):
        run_process([self.cmd] + unpack_args(*args), self.cmd, work_dir=self.path)

    def _switch(self, path):
        logging.info('Switching to %s' % path)
        self._run_svn('switch', '--ignore-ancestry', path)

    def _copy(self, src_path, dst_path, message):
        logging.info('Copying %s to %s with message "%s"' % (src_path, dst_path, message))
        self._run_svn('cp', src_path, dst_path, '-m', double_quote_string(message))


class VcsTypeParam(SandboxRadioParameter):
    VCS_TYPE_GIT = 'git'
    VCS_TYPE_SVN = 'svn'

    per_line = 4
    name = 'vcs_type'
    description = 'VCS type'
    choices = [('svn', VCS_TYPE_SVN), ('git', VCS_TYPE_GIT)]
    default_value = VCS_TYPE_GIT


class VcsUrlParam(SandboxStringParameter):
    name = 'vcs_url'
    description = 'VCS url'
    default_value = 'https://github.yandex-team.ru/mobile-search/report.git'
    required = True


VCS_REGISTRY = dict(
    svn=SvnVCS,
    git=GitVCS
)


def get_latest_commit_id(vcs):
    log = vcs.get_log(limit=1)
    return log[0].commit_id if log else None
