# -*- coding: utf-8 -*-
import logging
import os

from .utils import unpack_args
from sandbox.sandboxsdk.process import run_process


class MavenProject(object):
    MAVEN_CMD = 'mvn'
    SED_CMD = 'sed'
    POM_FILE_NAME = 'pom.xml'

    def __init__(self, path):
        self.path = path

    def get_module(self, module_name):
        return MavenProject(os.path.join(self.path, module_name))

    def runner(self):
        return self._Runner(self)

    def update_version(self, new_version):
        self.runner().goal('versions:set').with_args('-DnewVersion=%s' % new_version).run()

    def update_pom_property(self, property_name, property_value):
        sed_expr = 's,<{name}>.*</{name}>,<{name}>{value}</{name}>,g'.format(name=property_name, value=property_value)
        run_process([self.SED_CMD, '-i', '-e', sed_expr, self.POM_FILE_NAME], self.SED_CMD,
                    work_dir=self.path)

    class _Runner:
        def __init__(self, project):
            self._project = project
            self._goals = []
            self._args = []
            self._modules = None

        def goal(self, goal_name):
            self._goals.append(goal_name)
            return self

        def clean(self):
            return self.goal('clean')

        def compile(self):
            return self.goal('compile')

        def test(self):
            return self.goal('test')

        def package(self):
            return self.goal('package')

        def install(self):
            return self.goal('install')

        def on_modules(self, *modules):
            self._modules = unpack_args(*modules)
            return self

        def with_args(self, *args):
            self._args.extend(unpack_args(*args))
            return self

        def with_property(self, property_name, property_value=None):
            arg = '-D%s' % property_name
            if property_value is not None:
                arg += '=%s' % property_value
            return self.with_args(arg)

        def with_tests(self, should_run_tests):
            if not should_run_tests:
                return self.with_property('skipTests')
            return self

        def in_parallel(self, threads_num=None):
            return self.with_args('-T', threads_num or '1C')

        def run(self):
            cmd_args = []
            if self._modules is not None:
                cmd_args.extend(['-pl', ','.join(self._modules), '-am'])
            cmd_args.extend(self._args)
            cmd_args.extend(self._goals)
            self._run_maven(cmd_args)

        def _run_maven(self, args):
            logging.info('Running maven at %s with args %s' % (self._project.path, args))
            run_process([MavenProject.MAVEN_CMD] + args, MavenProject.MAVEN_CMD, work_dir=self._project.path)
