# -*- coding: utf-8 -*-
import logging
import re

from .mobvcs import VCS_REGISTRY, get_latest_commit_id
from .mvn import MavenProject


class ProjectInfo(object):
    def __init__(self, vcs_url, vcs_type, path):
        self.vcs_url = vcs_url
        self.vcs_type = vcs_type
        self.path = path


class VcsReleaseInfo(object):
    def __init__(self, report_revision, viewportese_revision, tickets):
        self.report_revision = report_revision
        self.viewportese_revision = viewportese_revision
        self.tickets = tickets

    def to_dict(self):
        return self.__dict__.copy()

    @staticmethod
    def from_dict(d):
        return VcsReleaseInfo(d['report_revision'], d['viewportese_revision'], d['tickets'])


class MobileReportReleaseMaker(object):
    def __init__(self, report_project_info, viewportese_project_info, release_version, config_type, skip_tests):
        self._release_version = release_version
        self._config_type = config_type
        self._skip_tests = skip_tests
        self._report_vcs = VCS_REGISTRY[report_project_info.vcs_type](report_project_info.vcs_url, report_project_info.path)
        self._viewportese_vcs = VCS_REGISTRY[viewportese_project_info.vcs_type](viewportese_project_info.vcs_url, viewportese_project_info.path)
        self._report_project = MavenProject(report_project_info.path)
        self._viewportese_project = MavenProject(viewportese_project_info.path)

    def process(self):
        logging.info('Obtaining sources')
        self._report_vcs.obtain()
        self._viewportese_vcs.obtain()

        logging.info('Validating projects and creating/switching to release branches')
        report_release_version = _ReportVersion(self._release_version)
        viewportese_release_version = _ViewporteseVersion(self._release_version)
        if self._release_version.is_major():
            self._validate_projects()
            report_release_revision = get_latest_commit_id(self._report_vcs)
            viewportese_release_revision = get_latest_commit_id(self._viewportese_vcs)
            self._report_vcs.create_branch(report_release_version.to_branch_name())
            self._report_vcs.switch_to_branch(report_release_version.to_branch_name())
        else:
            self._report_vcs.switch_to_branch(report_release_version.to_branch_name())
            self._viewportese_vcs.switch_to_branch(_ViewporteseVersion(self._release_version.previous_minor()).to_branch_name())
            report_release_revision = get_latest_commit_id(self._report_vcs)
            viewportese_release_revision = get_latest_commit_id(self._viewportese_vcs)
            self._validate_projects()
        self._viewportese_vcs.create_branch(viewportese_release_version.to_branch_name())
        self._viewportese_vcs.switch_to_branch(viewportese_release_version.to_branch_name())

        logging.info('Collecting ticket ids from commit messages')
        ticket_ids = set()
        ticket_ids.update(_collect_tickets_from_commit_messages(self._report_vcs, _ReportVersion(self._release_version, True)))
        ticket_ids.update(_collect_tickets_from_commit_messages(self._viewportese_vcs, _ViewporteseVersion(self._release_version, True)))

        logging.info('Updating versions for release')
        self._update_versions(report_release_version, viewportese_release_version)

        logging.info('Building release')
        self._build(False)

        logging.info('Creating tags')
        self._report_vcs.create_tag(report_release_version.to_tag_name())
        self._viewportese_vcs.create_tag(viewportese_release_version.to_tag_name())

        logging.info('Updating version in release branches')
        next_minor_version = self._release_version.next_minor()
        self._update_versions(_ReportVersion(next_minor_version, True), _ViewporteseVersion(next_minor_version, True))

        if self._release_version.is_major():
            logging.info('Switching to dev branches')
            self._report_vcs.switch_to_dev_branch()
            self._viewportese_vcs.switch_to_dev_branch()
            logging.info('Updating version in dev branches')
            next_major_version = self._release_version.next_major()
            self._update_versions(_ReportVersion(next_major_version, True), _ViewporteseVersion(next_major_version, True))

        return VcsReleaseInfo(report_release_revision, viewportese_release_revision, sorted(ticket_ids))

    def _update_versions(self, new_report_version, new_viewportese_version):
        # update project versions
        self._report_project.update_version(new_report_version)
        self._viewportese_project.update_version(new_viewportese_version)
        # update dependency versions
        self._report_project.get_module('viewport-build').update_pom_property('viewportese.version', new_viewportese_version)
        self._viewportese_project.update_pom_property('viewports.version', new_report_version)
        # commit changes
        for vcs, version in [(self._report_vcs, new_report_version), (self._viewportese_vcs, new_viewportese_version)]:
            vcs.commit(_get_changed_version_commit_message(version))

    def _build(self, should_run_tests):
        viewport_modules = ['viewport-api', 'viewport-validate', 'viewport-vml', 'vml-maven-plugin']
        self._report_project.runner().on_modules(viewport_modules).clean().install().with_tests(False).run()
        self._viewportese_project.runner().clean().install().with_tests(should_run_tests).run()
        self._report_project.runner().clean().package().with_property('config', self._config_type).\
            with_tests(should_run_tests).in_parallel().run()

    def _validate_projects(self):
        run_tests = not self._skip_tests
        logging.info('Compiling project' + ' and running tests' if run_tests else '')
        self._build(run_tests)


class _ProjectVersion(object):
    def __init__(self, version, is_snapshot=False):
        self.version = version
        self.is_snapshot = is_snapshot

    def to_tag_name(self):
        return 'R-%s' % self._base_version()

    def to_branch_name(self):
        raise NotImplementedError

    def to_string(self):
        if self.is_snapshot:
            return '%s-SNAPSHOT' % self._base_version()
        return self._base_version()

    def _base_version(self):
        raise NotImplementedError

    def __str__(self):
        return self.to_string()


class _ReportVersion(_ProjectVersion):
    def _base_version(self):
        return '%d.%d' % (self.version.major, self.version.minor)

    def to_branch_name(self):
        return 'R-%d' % self.version.major


class _ViewporteseVersion(_ProjectVersion):
    def _base_version(self):
        return '%d.%d-1.0' % (self.version.major, self.version.minor)

    def to_branch_name(self):
        return 'R-%d.%d' % (self.version.major, self.version.minor)


_TICKET_ID_REGEXP = re.compile('[A-Z]+-[0-9]+')
_COMMITS_BATCH_SIZE = 100


def _get_changed_version_commit_message(version):
    return 'Changed project version to %s' % version


def _collect_tickets_from_commit_messages(project_vcs, project_current_version):
    ticket_ids = set()
    lookup_end_commit_message = _get_changed_version_commit_message(project_current_version)
    log = project_vcs.get_log(limit=_COMMITS_BATCH_SIZE)
    while log:
        for commit_info in log:
            message = commit_info.message
            author = commit_info.author
            logging.info('Processing commit: %s - "%s"' % (author, message))
            if lookup_end_commit_message in message:
                return ticket_ids
            ticket_ids.update(re.findall(_TICKET_ID_REGEXP, message))
        log = project_vcs.get_log(log[-1].commit_id, limit=_COMMITS_BATCH_SIZE)[1:]
    return ticket_ids
