# -*- coding: utf-8 -*-
import logging
import time

from .startrek_client.client import Startrek
from .startrek_client.models import StartrekError


class StartrackReleaseInfo(object):
    def __init__(self, release_ticket_id, changelog):
        self.release_ticket_id = release_ticket_id
        self.changelog_tickets = changelog


class StartrackTicketsProcessor(object):
    DEFAULT_STARTRACK_API_URL = "https://st-api.yandex-team.ru"
    DEFAULT_RELEASE_QUEUE = 'SEAREL'
    DEFAULT_PROJECT_QUEUE = 'GATEWAY'
    _TIMEOUT = 100

    def __init__(self, release_version, client_auth_token, client_user_agent, st_api_url=DEFAULT_STARTRACK_API_URL,
                 release_queue=DEFAULT_RELEASE_QUEUE, project_queue=DEFAULT_PROJECT_QUEUE):
        self._st_client = Startrek(client_auth_token, client_user_agent, path=st_api_url,
                                   timeout=StartrackTicketsProcessor._TIMEOUT)
        self._release_version = release_version
        self._release_queue = release_queue
        self._project_queue = project_queue

    def update_tickets(self, release_ticket_content, release_ticket_assignee, release_ticket_followers,
                       release_ticket_qa_engineer, ticket_ids):
        release_ticket = self._get_release_ticket(release_ticket_content, release_ticket_assignee,
                                                  release_ticket_followers, release_ticket_qa_engineer)
        version = self._get_release_version()
        changelog_tickets = self._link_tickets_and_create_changelog(ticket_ids, release_ticket, version)
        return StartrackReleaseInfo(release_ticket.key, changelog_tickets)

    def add_comment_to_release_ticket(self, message):
        logging.info('Adding comment to release ticket: "%s"' % message)
        release_ticket = self._find_release_ticket()
        if release_ticket is not None:
            release_ticket.add_comment(message)
        else:
            logging.warn('Release ticket not found, unable add comment')

    def _get_release_ticket(self, release_ticket_content, release_ticket_assignee, release_ticket_followers,
                            release_ticket_qa_engineer):
        release_ticket = None
        if not self._release_version.is_major():
            release_ticket = self._find_release_ticket()

        if release_ticket is None:
            release_ticket = self._create_release_ticket(release_ticket_content, release_ticket_assignee,
                                                         release_ticket_followers, release_ticket_qa_engineer)
        else:
            self._update_release_ticket(release_ticket, release_ticket_content)

        return release_ticket

    def _get_release_ticket_name(self):
        return 'mobile-report/R-{major}'.format(major=self._release_version.major)

    def _find_release_ticket(self):
        logging.info('Searching for release ticket')
        query = 'Queue: {queue} and Summary: "{name}"'.format(queue=self._release_queue,
                                                              name=self._get_release_ticket_name())
        issues = self._st_client.find_issues(query, limit=1)
        release_ticket = next(issues, None)
        logging.info('Release ticket: %s' % 'Not found' if release_ticket is None else release_ticket.key)
        return release_ticket

    def _create_release_ticket(self, description, assignee, followers, qa_engineer):
        logging.info('Creating release ticket')
        ticket_data = {
            'queue': self._release_queue,
            'summary': self._get_release_ticket_name(),
            'description': description,
            'assignee': assignee,
            'followers': followers,
            'qaEngineer': qa_engineer
        }
        ticket = self._st_client.create_issue(ticket_data)
        logging.info('Release ticket id: %s' % ticket.key)
        return ticket

    @staticmethod
    def _update_release_ticket(ticket, description_extension):
        logging.info('Updating release ticket')
        new_description = '\n'.join((ticket.description, description_extension))
        ticket.update(description=new_description)

    def _get_release_version(self):
        version_name = 'R-{major}.{minor}'.format(major=self._release_version.major, minor=self._release_version.minor)
        return self._find_project_version(version_name) or self._create_project_version(version_name)

    def _find_project_version(self, version_name):
        logging.info('Searching for version: %s' % version_name)
        versions = self._st_client.get_queue_versions(self._project_queue)
        for version in versions:
            if version.name == version_name:
                return version
        return None

    def _create_project_version(self, version_name):
        logging.info('Creating version: %s' % version_name)
        version = self._st_client.add_version(self._project_queue, version_name)
        time.sleep(5)  # Wait for version to be created
        return version

    def _link_tickets_and_create_changelog(self, ticket_ids, release_ticket, version):
        ticket_id_to_title = dict()
        for ticket_id in sorted(set(ticket_ids)):
            logging.info('Processing ticket %s' % ticket_id)
            try:
                ticket = self._st_client.get_issue(ticket_id)
            except StartrekError as e:
                logging.warn('Unable to access ticket %s: %s' % (ticket_id, e))
                continue
            if self._project_queue in ticket.key:
                ticket.update(fixVersions={'add': [version.id]})
            try:
                ticket.add_link('relates', release_ticket.key)
            except StartrekError as e:
                logging.warn('Failed to create link between %s and %s: %s' % (ticket.key, release_ticket.key, e))
            ticket_id_to_title[ticket_id] = ticket.summary
        return ticket_id_to_title
