# coding: utf-8

import json
import logging
import operator
import sys

from .uriutils import expand
from .uriutils import Matcher

logger = logging.getLogger(__name__)
matcher = Matcher()


def s(value):
    if sys.version < '3':
        return unicode(value).encode('utf-8')
    else:
        return value


class ReferenceEncoder(json.JSONEncoder):

    def default(self, obj):
        if isinstance(obj, Reference) or isinstance(obj, Resource):
            return {"id": obj.id}
        return json.JSONEncoder.default(self, obj)


class Resource(object):

    _resource = None
    str_params = 'id name'

    def __init__(self, client):
        self._client = client
        self._session = client.session
        self._url = client.path + '/v2/' + self._resource
        self._value = None
        self._next = None

    def __str__(self):
        return '{0}{{{1}}}'.format(
            self.__class__.__name__,
            ':'.join(
                s(param)
                for param in self.get_str_params()
            )
        )

    def get_str_params(self):
        return [
            operator.attrgetter(param)(self)
            for param in self.str_params.split()
        ]

    @classmethod
    def register(cls):
        matcher.add('v2/' + cls._resource, cls)

    def get(self, id):
        self._get(expand(self._url, {'id': id}))

    def _get(self, uri):
        r = self._session.get(uri, timeout=self._client.timeout)
        self._check_status(r)
        self._build(json.loads(r.text))

    def get_all(self):
        return self._get_all(expand(self._url))

    def _get_all(self, uri):
        r = self._session.get(uri, timeout=self._client.timeout)
        self._check_status(r)

        for element in map(self._init_build, json.loads(r.text)):
            yield element

        while self._next is not None:
            r = self._session.get(self._next['url'], timeout=self._client.timeout)
            self._check_status(r)

            for element in map(self._init_build, json.loads(r.text)):
                yield element

    def _init_build(self, object):
        value = self.__class__(self._client)
        value._build(object)
        return value

    def create(self, data):
        self._create(expand(self._url, {}), data)

    def _create(self, uri, data):
        r = self.post(uri, data)
        self._check_status(r)
        self._build(json.loads(r.text))

    def _link(self, uri, resource, rel):
        r = self._session.request('LINK', uri, headers={
            'Link': '<%s>; rel="%s"' % (resource, rel)
        }, timeout=self._client.timeout)
        self._check_status(r)
        self._build(json.loads(r.text))

    def _unlink(self, uri, resource, rel):
        r = self._session.request('UNLINK', uri, headers={
            'Link': '<%s>; rel="%s"' % (resource, rel)
        }, timeout=self._client.timeout)
        self._check_status(r)

    def _patch(self, uri, data):
        r = self._session.patch(uri, data=json.dumps(data, cls=ReferenceEncoder), timeout=self._client.timeout)
        self._check_status(r)
        self._build(json.loads(r.text))

    # TODO: remove?
    def post(self, url, data):
        r = self._session.post(url, data=json.dumps(data), timeout=self._client.timeout)
        self._check_status(r)
        return r

    def _check_status(self, r):
        if r.status_code == 404:
            raise NotFound(r)
        elif r.status_code >= 400:
            raise StartrekError(r)

        self._next = r.links.get('next')

    def _build(self, raw_object):
        obj = {}
        for field, value in raw_object.items():
            obj[field] = self._build_value(field, value)
        self._value = obj
        return self

    def _build_value(self, field, value):
        if field == 'customFields':
            return value
        elif isinstance(value, list):
            return [self._build_value(field, element) for element in value]
        elif isinstance(value, dict) and 'self' in value:
            return Reference(field, value, self._client)
        else:
            return value

    def __repr__(self):
        return self.__str__()

    def __getattr__(self, name):
        if name.startswith('_'):
            return getattr(super(Resource, self), name)

        if name in self._value:
            return self._value[name]

        raise AttributeError(
            'Object "%s" has no attribute %s' % (self.__class__.__name__, name)
        )


class Reference(object):

    def __init__(self, field, value, client):
        self._client = client
        self.id = value['id']
        self.self = value['self']
        self.type = matcher.match(self.self.replace(self._client.path, ''))
        self.outward = value.get('outward')
        self.inward = value.get('inward')
        if 'display' in value:
            self.name = value['display']
        else:
            self.name = value['id']
        self._value = None

    def __repr__(self):
        return self.__str__()

    def self_type(self):
        if self.type is None:
            return self.self
        else:
            return self.type.__name__

    def __str__(self):
        if self._value is None:
            return '[{0}:{1}]'.format(self.self_type(), s(self.name))
        else:
            return self._value.__str__()

    def __getattr__(self, name):
        if self._value is None:
            resource_descriptor = matcher.match(
                self.self.replace(self._client.path, ''))
            if resource_descriptor is None:
                raise Exception('Unknown resource: ' + self.self)
            resource = resource_descriptor(self._client)
            resource._get(self.self)
            self._value = resource

        return getattr(self._value, name)


class IssueType(Resource):
    _resource = 'issuetypes/{id}'

    def get_by_queue(self, queue):
        uri = expand(
            self._client.path + '/v2/queues/{queue}/issuetypes', {'queue': queue})
        return self._get_all(uri)


class Priority(Resource):
    _resource = 'priorities/{id}'


class User(Resource):
    _resource = 'users/{id}'


class Group(Resource):
    _resource = 'groups/{id}'


class Status(Resource):
    _resource = 'statuses/{id}'


class Resolution(Resource):
    _resource = 'resolutions/{id}'


class Version(Resource):
    _resource = 'versions/{id}'

    def get_by_queue(self, queue):
        uri = expand(
            self._client.path + '/v2/queues/{queue}/versions', {'queue': queue})
        return self._get_all(uri)

    def create_for_queue(self, queue, version_name, **kwargs):
        data = {'queue': queue, 'name': version_name}
        data.update(kwargs)
        self.create(data)


class Component(Resource):
    _resource = 'components/{id}'

    def get_by_queue(self, queue):
        uri = expand(
            self._client.path + '/v2/queues/{queue}/components',
            {'queue': queue},
        )
        return self._get_all(uri)


class Project(Resource):
    _resource = 'projects/{id}'
    str_params = 'id type.key'

    def get_by_queue(self, queue):
        uri = expand(
            self._client.path + '/v2/queues/{queue}/projects',
            {'queue': queue})
        return self._get_all(uri)


class Issue(Resource):
    _resource = 'issues/{id}{?perPage}'
    str_params = 'key type.name summary'

    def get_all(self, query, limit):
        r = self._session.post(
            expand(self._url, {'id': '_search', 'perPage': limit}),
            data=json.dumps({'query': query}),
            timeout=self._client.timeout)
        self._check_status(r)

        for element in map(self._init_build, json.loads(r.text)):
            yield element

        while self._next is not None:
            print(self._next)
            r = self._session.post(self._next['url'], data=json.dumps({'query': query}), timeout=self._client.timeout)
            self._check_status(r)

            for element in map(self._init_build, json.loads(r.text)):
                yield element

    def __getattr__(self, name):
        if name == 'transitions':
            if 'transitions' in self._value:
                return self._value['transitions']
            else:
                return self._client.get_transitions(self.key)
        elif name == 'comments':
            if 'comments' in self._value:
                return self._value['comments']
            else:
                return self._client.get_comments(self.key)
        elif name == 'links':
            if 'links' in self._value:
                return self._value['links']
            else:
                return self._client.get_links(self.key)
        elif name == 'attachments':
            if 'attachments' in self._value:
                return self._value['attachments']
            else:
                return self._client.get_attachments(self.key)
        elif name == 'changelog':
            return self._client.get_changelog(self.key)
        return super(Issue, self).__getattr__(name)

    def transition_to(self, transition, **kwargs):
        uri = expand(self._url, {"id": self.key}) + expand(
            '/transitions/{transition}/_execute', {'transition': transition})
        self.post(uri, data=kwargs)

    def update(self, **kwargs):
        self._patch(self.self, kwargs)

    def update_one(self, id, **kwargs):
        self._patch(expand(self._url, {'id': id}), kwargs)

    def add_comment(self, text, *args):
        return self._client.add_comment(self.key, text, *args)

    def add_link(self, relationship, to):
        return self._client.add_link(self.key, relationship, to)

    def link(self, resource, relationship):
        return self._client.link(self.key, resource, relationship)

    def unlink(self, resource, relationship):
        return self._client.unlink(self.key, resource, relationship)


class Queue(Resource):
    _resource = 'queues/{id}'
    str_params = 'key name'

    def __getattr__(self, name):
        if name == 'versions':
            return self._client.get_queue_versions(self.key)
        elif name == 'components':
            return self._client.get_queue_components(self.key)
        elif name == 'projects':
            return self._client.get_queue_projects(self.key)
        elif name in ('issuetypes', 'issue_types', 'issueTypes'):
            return self._client.get_queue_issue_types(self.key)
        return super(Queue, self).__getattr__(name)

    def add_version(self, name, **kwargs):
        return self._client.add_version(self.key, name, **kwargs)


class Transition(Resource):
    _resource = 'issues/{issue}/transitions/{id}'
    str_params = 'id to'

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))


class Comment(Resource):
    _resource = 'issues/{issue}/comments/{id}'
    str_params = 'id text'

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))

    def create(self, issue, text, *args):
        data = {'text': text}
        if args:
            data['attachmentIds'] = [
                Attachment(self._client).upload_temp(file)
                for file in args
            ]
        self._create(expand(self._url, {'issue': issue}), data=data)

    def update(self, text, *args):
        if self._value is None:
            raise Exception('Comment is not loaded')
        data = {'text': text}
        if args:
            data['attachmentIds'] = [
                Attachment(self._client).upload_temp(file)
                for file in args
            ]
        self._patch(self.self, data)

    def update_one(self, issue, id, text, *args):
        data = {'text': text}
        if args:
            data['attachmentIds'] = [
                Attachment(self._client).upload_temp(file)
                for file in args
            ]
        self._patch(
            expand(
                self._url, {
                    'issue': issue, 'id': id}), data)

    def delete(self):
        if self._value is None:
            raise Exception('Comment is not loaded')
        r = self._session.delete(self.self, timeout=self._client.timeout)
        self._check_status(r)

    def delete_one(self, issue, id):
        r = self._session.delete(expand(self._url, {'issue': issue, 'id': id}), timeout=self._client.timeout)
        self._check_status(r)


class IssueLink(Resource):
    _resource = 'issues/{issue}/links/{id}'

    def get_str_params(self):
        return self.object.key, getattr(self.type, self.direction)

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))

    def create(self, issue, relationship, to):
        self._create(
            expand(self._url, {'issue': issue}),
            data={'relationship': relationship, 'issue': to}
        )

    def delete(self):
        if self._value is None:
            raise Exception('IssueLink is not loaded')
        r = self._session.delete(self.self, timeout=self._client.timeout)
        self._check_status(r)

    def delete_one(self, issue, id):
        r = self._session.delete(expand(self._url, {'issue': issue, 'id': id}), timeout=self._client.timeout)
        self._check_status(r)

    def link(self, issue, resource, relationship):
        self._link(expand(self._client.path + '/v2/issues/{issue}', {'issue': issue}), resource, relationship)

    def unlink(self, issue, resource, relationship):
        self._unlink(expand(self._client.path + '/v2/issues/{issue}', {'issue': issue}), resource, relationship)


class Attachment(Resource):
    _resource = 'issues/{issue}/attachments/{id}'
    str_params = 'name'

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))

    def upload(self, issue, file):
        files = {'file': open(file, 'rb')}
        r = self._session.post(
            expand(self._url, {'issue': issue}),
            headers={'Content-Type': None},
            files=files,
            timeout=self._client.timeout
        )
        self._check_status(r)

    def upload_temp(self, file):
        files = {'file': open(file, 'rb')}
        r = self._session.post(
            self._client.path + '/v2/attachments',
            headers={'Content-Type': None},
            files=files,
            timeout=self._client.timeout
        )
        self._check_status(r)
        self._build(json.loads(r.text))
        return int(self.id)

    def download_to(self, path):
        r = self._session.get(self.content, timeout=self._client.timeout)
        self._check_status(r)
        with open(path + '/' + self.name, 'wb') as file:
            file.write(r.content)

    def delete(self):
        if self._value is None:
            raise Exception('Attachment is not loaded')
        r = self._session.delete(self.self, timeout=self._client.timeout)
        self._check_status(r)

    def delete_one(self, issue, id):
        r = self._session.delete(expand(self._url, {'issue': issue, 'id': id}), timeout=self._client.timeout)
        self._check_status(r)


class ChangelogEntry(Resource):
    _resource = 'issues/{issue}/changelog/{id}{?sort,field*,type*}'
    str_params = 'id type transport'

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue, sort, fields, types):
        return self._get_all(expand(self._url, {'issue': issue, 'sort': sort, 'field': fields, 'type': types}))


class StartrekError(Exception):

    def __init__(self, r):
        self.code = r.status_code

        logger.error('Status code %d', self.code)
        logger.error('Response body %s', r.text)

        self.messages = []
        if self.code < 500:
            error = json.loads(r.text)
            if 'errors' in error:
                self.messages = error['errors']
            else:
                self.messages = error['errorMessages']
            if 'invocation-info' in error:
                self._host = error['invocation-info']['hostname']

    def __str__(self):
        return 'Error[{0}:{1}]'.format(
            self.code,
            ''.join(message.encode('utf-8') for message in self.messages))


class NotFound(StartrekError):

    def __str__(self):
        return 'NotFound[{0}]'.format(
            ''.join(message.encode('utf-8') for message in self.messages))


Issue.register()
Queue.register()
IssueType.register()
Priority.register()
User.register()
Group.register()
Status.register()
Resolution.register()
Version.register()
Component.register()
Project.register()
Transition.register()
Attachment.register()
Comment.register()
IssueLink.register()
ChangelogEntry.register()
