# -*- coding: utf-8 -*-

import os
import logging

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.common.environments import SandboxMavenEnvironment
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process

from ..mobvcs import VcsTypeParam, VcsUrlParam, VCS_REGISTRY


class BaseMavenBuildTask(SandboxTask):
    client_tags = ctc.Tag.Group.LINUX & ctc.Tag.IPV4
    environment = (
        SvnEnvironment(),
        SandboxJavaJdkEnvironment('1.8.0'),
        SandboxMavenEnvironment('3.2.2'),
    )

    input_parameters = (VcsTypeParam, VcsUrlParam)
    resources = []
    archs_for_bundle = ('linux',)
    vcs_dir = 'checkout'
    dist_dir = 'dist'

    def on_execute(self):
        self.get_src()
        self.run_mvn()
        self.make_resource()
        self.set_info('Task completed successfully.')

    def get_src(self):
        logging.info('Obtaining source tree using %s' % self.ctx['vcs_type'])
        vcs = VCS_REGISTRY[self.ctx['vcs_type']](self.ctx['vcs_url'], self.path(self.vcs_dir))
        if self.ssh_key():
            with self.ssh_key():
                vcs.obtain()
        else:
            vcs.obtain()

    def run_mvn(self):
        run_dir = self.path(self.vcs_dir)
        logging.info('Running mvn in %s with args %s' % (run_dir, self.mvn_args()))
        run_process(['mvn ' + self.mvn_args()], 'mvn', shell=True, timeout=10800, check=True, work_dir=run_dir)

    def make_resource(self):
        run_dir = self.path(self.vcs_dir)
        dist_dir = os.path.join(self.abs_path(), self.dist_dir, '')
        try:
            make_folder(dist_dir)
            for res in self.resources:
                resource_path = os.path.join(run_dir, res[2], res[0])
                if res[3] and not os.path.exists(resource_path):
                    continue
                copy_path(resource_path, os.path.join(dist_dir, res[0]))
                self.create_resource(self.descr, os.path.join(dist_dir, res[0]), res[1])
        except IOError, e:
            raise SandboxTaskFailureError('Unable to copy file. %s' % e)

    def mvn_args(self):
        raise NotImplementedError('Method mvn_args is not implemented')

    def ssh_key(self):
        return None

    def arcadia_info(self):
        """
            Получение информации о задаче при релизе
            Может быть переопределён в наследниках для уточнения возвращаемых данных
            @return список из трёх значений revision, tag, branch
        """
        return 'test', 'test', 'trunk'


__Task__ = BaseMavenBuildTask
