# -*- coding: utf-8 -*-
import logging
import json
import os
import re
from collections import namedtuple
from copy import deepcopy
from six.moves import http_client

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox.common import errors
from sandbox.common import rest


SandboxFileUpdate = namedtuple(
    "SandboxFileUpdate",
    (
        "local_path",
        "resource_id",
        "resource_type",
        "task_id",
        "task_type",
        "is_dynamic",  # default: False
        "extract_path",  # default: ""
    ),
)
SandboxFileUpdate.__new__.__defaults__ = (
    False,  # is_dynamic
    "",  # extract_path
)

StaticFileUpdate = namedtuple(
    "StaticFileUpdate",
    (
        "local_path",
        "content",
        "is_dynamic",
    )
)
StaticFileUpdate.__new__.__defaults__ = (
    False,  # is_dynamic
)


class TaskGroupStatus(object):
    DONE = "DONE"
    MERGED = "MERGED"
    NEW = "NEW"
    COMMITTED = "COMMITTED"
    REJECTED = "REJECTED"


class NannyApiException(Exception):
    """Ошибка при работе с Nanny API."""
    def __init__(self, msg, http_error=None):
        self.msg = msg
        self.http_error = http_error

    def __str__(self):
        return self.msg


def _handle_http_error(e):
    """
    Если был получен ответ от сервера, надо сформировать исключение NannyApiException с телом ответа,
    т.к. в нём может содержаться полезная информация об ошибке.

    :type e: requests.HTTPError
    """
    message = 'Nanny responded with code {}. Body:\n{}'.format(e.response.status_code, e.response.content)
    return NannyApiException(message, http_error=e)


class NannyClient(object):
    def __init__(self, api_url, oauth_token, ca_certs='/etc/ssl/certs/ca-certificates.crt'):
        self._api_url = api_url.rstrip('/')
        self._oauth_token = oauth_token
        self._session = requests.Session()
        if ca_certs and os.path.isfile(ca_certs):
            self._session.verify = ca_certs
        retries = Retry(total=3, backoff_factor=1.0, status_forcelist=[429, 500, 502, 503, 504])
        self._session.mount(self._api_url, HTTPAdapter(max_retries=retries))
        self._session.headers.update({'Content-Type': 'application/json'})
        if self._oauth_token:
            self._session.headers.update({'Authorization': 'OAuth {}'.format(self._oauth_token)})

    def create_release(self, data):
        url = '{}/v1/requests/sandbox_releases/'.format(self._api_url)
        return self._post_url(url, json.dumps(data))

    def get_release_request_info(self, release_request_id):
        url = '{}/v1/requests/{}/'.format(self._api_url, release_request_id)
        return self._get_url(url)

    def create_release2(self, data):
        """
        Creates release using "new" API.
        """
        url = '{}/api/tickets/CreateRelease/'.format(self._api_url)
        return self._post_url(url, json.dumps(data))

    def create_gencfg_release(self, data):
        url = '{}/v1/requests/gencfg_releases/'.format(self._api_url)
        self._post_url(url, json.dumps(data))

    def create_service(self, _id, info_attrs, auth_attrs=None, runtime_attrs=None, comment="-"):
        url = '{}/v2/services/'.format(self._api_url)
        data = {
            'id': _id,
            'comment': comment,
            'info_attrs': info_attrs,
        }
        if auth_attrs:
            data['auth_attrs'] = auth_attrs
        if runtime_attrs:
            data['runtime_attrs'] = runtime_attrs
        self._post_url(url, json.dumps(data))

    def create_event(self, service_id, event_data):
        url = '{}/v2/services/{}/events/'.format(self._api_url, service_id)
        return self._post_url(url, json.dumps(event_data))

    def update_service_resources(self, service_id, resources_data):
        url = '{}/v2/services/{}/runtime_attrs/resources/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps(resources_data))

    def update_service_instance_spec(self, service_id, instance_spec_data):
        url = '{}/v2/services/{}/runtime_attrs/instance_spec/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps(instance_spec_data))

    def update_service_instances(self, service_id, instances_data):
        # TODO: maybe current_state/instances ???
        url = '{}/v2/services/{}/runtime_attrs/instances/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps(instances_data))

    def get_service(self, service_id):
        url = '{}/v2/services/{}/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_target_state(self, service_id):
        url = '{}/v2/services/{}/target_state/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_current_state(self, service_id):
        url = '{}/v2/services/{}/current_state/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_snapshot_state_changes(self, snapshot_id, limit=None, skip=None):
        query = {"snapshotId": snapshot_id}
        if limit is not None:
            query["limit"] = limit
        if skip is not None:
            query["skip"] = skip
        url = '{}/api/repo/ListSnapshotStateChanges/'.format(self._api_url)
        return self._post_url(url, data=json.dumps(query))

    def get_service_current_instances(self, service_id):
        url = '{}/v2/services/{}/current_state/instances/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_info_attrs(self, service_id):
        url = '{}/v2/services/{}/info_attrs/'.format(self._api_url, service_id)
        return self._get_url(url)

    def set_service_info_attrs(self, service_id, info_attrs):
        url = '{}/v2/services/{}/info_attrs/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps(info_attrs))

    def get_service_runtime_attrs(self, service_id):
        url = '{}/v2/services/{}/runtime_attrs/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_runtime_attrs_history(self, service_id, limit=None, skip=None):
        url = '{}/v2/services/{}/history/runtime_attrs_infos/'.format(self._api_url, service_id)
        params = {}
        if limit is not None:
            params['limit'] = limit
        if skip is not None:
            params['skip'] = skip
        return self._get_url(url, params=params)

    def update_service_runtime_attrs(self, service_id, snapshot_id, content, comment="-"):
        url = '{}/v2/services/{}/runtime_attrs/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps({
            'snapshot_id': snapshot_id,
            'comment': comment,
            'content': content,
        }))

    def update_service_runtime_attrs_section(self, service_id, section_name, content, comment="-"):
        url = '{}/v2/services/{}/runtime_attrs/{}/'.format(self._api_url, service_id, section_name)
        return self._put_url(url, json.dumps({
            'comment': comment,
            'content': content,
        }))

    def get_service_instances(self, service_id):
        url = '{}/v2/services/{}/runtime_attrs/instances/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_instance_spec(self, service_id):
        url = '{}/v2/services/{}/runtime_attrs/instance_spec/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_resources(self, service_id):
        url = '{}/v2/services/{}/runtime_attrs/resources/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_engines(self, service_id):
        url = '{}/v2/services/{}/runtime_attrs/engines/'.format(self._api_url, service_id)
        return self._get_url(url)

    def update_service_engines(self, service_id, engines_data):
        url = '{}/v2/services/{}/runtime_attrs/engines/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps(engines_data))

    def get_service_active_runtime_attrs(self, service_id):
        url = '{}/v2/services/{}/active/runtime_attrs/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_description(self, service_id):
        url = '{}/v2/services/{}/'.format(self._api_url, service_id)
        return self._get_url(url)

    def get_service_auth_attrs(self, service_id):
        url = '{}/v2/services/{}/auth_attrs/'.format(self._api_url, service_id)
        return self._get_url(url)

    def set_service_auth_attrs(self, service_id, snapshot_id, content, comment="-"):
        url = '{}/v2/services/{}/auth_attrs/'.format(self._api_url, service_id)
        return self._put_url(url, json.dumps({
            'snapshot_id': snapshot_id,
            'comment': comment,
            'content': content,
        }))

    def list_services(self, skip, limit, order_by=None):
        url = '{}/v2/services/'.format(self._api_url)
        params = {'skip': skip, 'limit': limit}
        if order_by:
            params['order_by'] = order_by
        return self._get_url(url, params=params)

    def list_services_by_category(self, category):
        limit = 200
        skip = 0

        result = []
        while True:
            url = '{}/v2/services/?category={}&limit={}&skip={}'.format(self._api_url, category, limit, skip)
            data = self._get_url(url)
            result += data['result']
            if len(data['result']) < limit:
                break
            skip += limit

        return {'result': result}

    def copy_service(
        self,
        src_service_id,
        new_service_id,
        category,
        description,
        copy_auth_attrs=False,
        copy_instances=False,
        abc_group_id=None,
        copy_secrets=False,
        copy_infra_id=False,
    ):
        url = '{}/v2/services/{}/copies/'.format(self._api_url, src_service_id)
        post_data = {
            'id': new_service_id,
            'copy_auth_attrs': copy_auth_attrs,
            'copy_instances': copy_instances,
            "copy_secrets": copy_secrets,
            "copy_infra_id": copy_infra_id,
            'category': category,
            'desc': description,
        }
        if abc_group_id is not None:
            post_data["abc_group"] = abc_group_id
        return self._post_url(url, json.dumps(post_data))

    def delete_service(self, service_id):
        url = '{}/v2/services/{}/'.format(self._api_url, service_id)
        return self._delete_url(url)

    def set_snapshot_state(
        self, service_id, snapshot_id, state, comment, recipe=None, prepare_recipe=None, set_as_current=False
    ):
        url = '{}/v2/services/{}/events/'.format(self._api_url, service_id)
        post_data = {
            'type': 'SET_SNAPSHOT_STATE',
            'content':
                {
                    'snapshot_id': snapshot_id,
                    'state': state,
                    'comment': comment
                }
        }

        if recipe is not None:
            post_data['content']['recipe'] = recipe

        if prepare_recipe is not None:
            post_data['content']['prepare_recipe'] = prepare_recipe

        if set_as_current:
            post_data['content']['set_as_current'] = True

        return self._post_url(url, json.dumps(post_data))

    def delete_all_snapshots(self, service_id):
        url = '{}/v2/services/{}/events/'.format(self._api_url, service_id)
        post_data = {
            'type': 'REMOVE_ALL_SNAPSHOTS',
            'content': {'comment': 'Remove all snapshots to remove service.'}
        }
        return self._post_url(url, json.dumps(post_data))

    def get_snapshot_instances(self, service_id, snapshot_id):
        url = '{}/v2/services/instances/{}/sn/{}/'.format(self._api_url, service_id, snapshot_id)
        return self._get_url(url)

    def set_service_gencfg_groups(self, service_id, gencfg_groups, release):
        url = '{}/v2/services/{}/runtime_attrs/'.format(self._api_url, service_id)
        runtime_attrs_doc = self._get_url(url)
        runtime_attrs_doc['content']['instances'].pop(runtime_attrs_doc['content']['instances']['chosen_type'].lower())
        runtime_attrs_doc['content']['instances']['chosen_type'] = 'EXTENDED_GENCFG_GROUPS'
        runtime_attrs_doc['content']['instances']['extended_gencfg_groups'] = {
            'tags': [],
            'groups': [
                {
                    'release': release,
                    'limits': {
                        'cpu_policy': 'normal',
                        'io_policy': 'normal'
                    },
                    'tags': [],
                    'name': group
                }
                for group in gencfg_groups
            ],
        }

        post_data = {
            'snapshot_id': runtime_attrs_doc['_id'],
            'comment': 'Set gencfg group',
            'content': runtime_attrs_doc['content']
        }
        return self._put_url(url, json.dumps(post_data))

    def get_history_runtime_attrs(self, snapshot_id):
        url = '{}/v2/history/services/runtime_attrs/{}/'.format(self._api_url, snapshot_id)
        return self._get_url(url)

    def get_history_runtime_attrs_states(self, service, snapshot_id):
        url = '{}/v2/services/{}/history/runtime_attrs/{}/states/'.format(self._api_url, service, snapshot_id)
        return self._get_url(url)

    def list_recipes(self):
        url = '{}/v1/alemate/yaml_recipes/'.format(self._api_url)
        return self._get_url(url)

    def get_recipe_content(self, recipe):
        url = '{}/v1/alemate/yaml_recipes/{}/'.format(self._api_url, recipe)
        return self._get_url(url)

    def run_recipe(self, recipe, parameters=None):
        if not parameters:
            parameters = {}
        url = '{}/v1/alemate/yaml_recipes/apply/'.format(self._api_url)
        recipe_content = self.get_recipe_content(recipe)['body']
        post_data = {'content': recipe_content, 'parameters': parameters}
        return self._post_url(url, json.dumps(post_data))

    def get_taskgroup_status(self, task_id):
        url = '{}/v1/alemate/task_groups/{}/status/'.format(self._api_url, task_id)
        return self._get_url(url)

    def get_taskgroup_info(self, task_id):
        url = '{}/v1/alemate/task_groups/{}/info/'.format(self._api_url, task_id)
        return self._get_url(url)

    def set_taskgroup_status(self, task_id, status=TaskGroupStatus.REJECTED):
        url = '{}/v1/alemate/task_groups/{}/status/'.format(self._api_url, task_id)
        data = {'status': status}
        return self._post_url(url, json.dumps(data))

    def get_taskgroup_children(self, task_id):
        url = '{}/v1/alemate/task_groups/{}/children/'.format(self._api_url, task_id)
        return self._get_url(url)

    def send_taskgroup_job_commands(self, task_id, job_id, data):
        url = '{}/v1/alemate/task_groups/{}/tasks/{}/commands/'.format(self._api_url, task_id, job_id)
        return self._post_url(url, json.dumps(data))

    def get_dashboard(self, dashboard):
        # FIXME: deprecated method to be removed, use get_dashboard_content instead
        url = '{}/v2/services_dashboards/catalog/{}/services/'.format(self._api_url, dashboard)
        return self._get_url(url)

    def get_dashboard_state(self, dashboard):
        url = '{}/v2/services_dashboards/catalog/{}/state/'.format(self._api_url, dashboard)
        return self._get_url(url)

    def get_dashboard_services(self, dashboard):
        service_ids = self.get_dashboard_services_groups(dashboard, groups=None)
        # Sort is necessary to be compatible with old implementation of getting service ids
        service_ids.sort()
        return service_ids

    def create_dashboard(self, _id, content):
        url = "{}/v2/services_dashboards/catalog/".format(self._api_url)
        data = {
            "_id": _id,
            "content": content,
        }
        self._post_url(url, json.dumps(data))

    def update_dashboard(self, _id, content):
        url = "{}/v2/services_dashboards/catalog/{}/".format(self._api_url, _id)
        data = {
            "comment": "Update dashboard",
            "content": content,
        }
        self._put_url(url, json.dumps(data))

    def delete_dashboard(self, _id):
        url = "{}/v2/services_dashboards/catalog/{}/".format(self._api_url, _id)
        return self._delete_url(url)

    def get_dashboard_content(self, dashboard_id):
        # This API method is much faster than /v2/services_dashboards/catalog/<dashboard_id>/services/
        url = '{}/v2/services_dashboards/catalog/{}/'.format(self._api_url, dashboard_id)
        return self._get_url(url)

    def get_dashboard_services_groups(self, dashboard, groups=None):
        if groups is not None and not isinstance(groups, (list, tuple)):
            raise Exception('{} is not a list or tuple of group ids'.format(groups))

        service_ids = []

        dashboard_info = self.get_dashboard_content(dashboard)
        for dashboard_group in dashboard_info['content'].get('groups', []):
            logging.debug('get_dashboard_services_groups: dashboard_group {}'.format(dashboard_group))
            if groups is None or dashboard_group['id'] in groups:
                for s in dashboard_group['services']:
                    service_ids.append(s['service_id'])

        return service_ids

    def create_recipe_in_dashboard(self, dashboard, id, content):
        url = '{}/v2/services_dashboards/catalog/{}/recipes/'.format(self._api_url, dashboard)
        data = {
            "id": id,
            "content": content,
        }
        logging.debug("Data %s", data)
        return self._post_url(url, json.dumps(data))

    def update_recipe_in_dashboard(self, dashboard, id, content):
        url = '{}/v2/services_dashboards/catalog/{}/recipes/{}/'.format(self._api_url, dashboard, id)
        data = {
            "comment": "Update recipe",
            "content": content,
        }
        logging.debug("Data %s", data)
        return self._put_url(url, json.dumps(data))

    def delete_recipe_from_dashboard(self, dashboard_id, recipe_id):
        url = '{}/v2/services_dashboards/catalog/{}/recipes/{}/'.format(self._api_url, dashboard_id, recipe_id)
        return self._delete_url(url)

    def get_dashboard_recipes(self, dashboard):
        url = '{}/v2/services_dashboards/catalog/{}/recipes/'.format(self._api_url, dashboard)
        return self._get_url(url)

    def safe_get_dashboard_recipe(self, dashboard, recipe_name):
        recipes = self.get_dashboard_recipes(dashboard)
        recipes = [x for x in recipes['result'] if x['id'] == recipe_name]
        if not recipes:
            logging.error("There is no recipe %s in dashboard %s", recipe_name, dashboard)
            return None
        return recipes[0]

    def get_dashboard_recipe(self, dashboard, recipe_name):
        recipes = self.get_dashboard_recipes(dashboard)
        return [x for x in recipes['result'] if x['id'] == recipe_name][0]

    def get_dashboard_latest_snapshots(self, dashboard):
        dashboard_services = self.get_dashboard_services(dashboard)
        return {service: self.get_service_resources(service)['snapshot_id'] for service in dashboard_services}

    def get_dashboard_taskgroups(self, dashboard, statuses=(TaskGroupStatus.MERGED, ), limit=10):
        filters = ['dashboard_id={}'.format(dashboard)]
        if statuses:
            filters.append('statuses={}'.format(','.join(statuses)))
        params = {
            'filter': '&'.join(filters),
            'limit': limit,
        }
        url = '{}/v1/alemate/task_groups/'.format(self._api_url)
        return self._get_url(url, params=params)

    def get_dashboard_deployment(self, dashboard, deployment_id):
        url = '{}/v2/services_dashboards/catalog/{}/deployments/{}/'.format(self._api_url, dashboard, deployment_id)
        return self._get_url(url)

    def deploy_dashboard_yaml(self, dashboard, recipe, service_snapshots=None, pass_additional_service_info=()):
        """Deploys dashboard's services via yaml recipe
        :param: dashboard: dashboard identifier
        :param: recipe: recipe name, e.g. my_recipe_name.yaml
        :param: service_snapshots: dict, service_id: snapshot_id
        :param: pass_additional_service_info: additional service info keys, that will be added to recipe's context
        """
        url = '{}/v2/services_dashboards/catalog/{}/deployments/'.format(self._api_url, dashboard)

        if not service_snapshots:
            raise NannyApiException('service_snapshots must be specified!')

        dashboard_services = self.get_dashboard_services(dashboard)
        services = {}
        for service_id, snapshot_id in service_snapshots.items():
            if service_id not in dashboard_services:
                logging.warning("Service %s does not belong to dashboard %s, skipping it", service_id, dashboard)
                continue

            services[service_id] = {
                'runtime_attrs_id': snapshot_id,
            }
            if pass_additional_service_info:
                full_service_info = self.get_service(service_id)
                if 'labels' in pass_additional_service_info:
                    services[service_id].update({
                        'labels': {l['key']: l.get('value') for l in full_service_info['info_attrs']['content']['labels']},
                    })
                if 'gencfg_groups' in pass_additional_service_info:
                    services[service_id].update({
                        'gencfg_groups': full_service_info['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups'],
                    })

        deploy_data = {
            'type': 'yaml',
            'yaml_recipe_id': recipe,
            'yaml_recipe_context': {
                'services': services,
                'service_ids': list(services.keys()),
            }
        }
        logging.debug("Deployment payload: %s", json.dumps(deploy_data, indent=2))

        return self._post_url(url, json.dumps(deploy_data))

    def deploy_dashboard(
        self, dashboard, recipe,
        service_snapshots=None,
        use_latest_snapshot=False,
        set_its_values=None,
        set_zk_values=None,
    ):
        url = '{}/v2/services_dashboards/catalog/{}/deployments/'.format(self._api_url, dashboard)
        recipe_data = self.get_dashboard_recipe(dashboard, recipe)
        if service_snapshots and use_latest_snapshot:
            raise NannyApiException('Only service_snapshots and use_latest_snapshot must be specified!')
        if use_latest_snapshot:
            service_snapshots = self.get_dashboard_latest_snapshots(dashboard)
            logging.debug("Got latest snapshots: {}".format(service_snapshots))

        for _id, task in enumerate(recipe_data['content']['tasks']):
            if task['data']['name'] in ['set_snapshot_target_state']:
                service_id = task['data']['params']['service_id']
                if service_id in service_snapshots:
                    logging.debug('Assign snapshot for task type {}: {}, snapshot: {}'.format(
                        task['data']['name'],
                        service_id,
                        service_snapshots[service_id]
                    ))
                    recipe_data['content']['tasks'][_id]['data']['vars']['snapshot_id'] = service_snapshots[service_id]
                else:
                    raise NannyApiException("Can't find latest snapshot for %s" % service_id)
            elif (
                task['data']['name'] in ['set_its_value'] and
                isinstance(set_its_values, dict) and
                len(set_its_values) > 0
            ):
                its_path = task['data']['params']['path']
                if its_path in set_its_values:
                    logging.debug('Assign value {} to its path: {}'.format(set_its_values[its_path], its_path))
                    recipe_data['content']['tasks'][_id]['data']['vars']['path'] = its_path
                    recipe_data['content']['tasks'][_id]['data']['vars']['str_value'] = set_its_values[its_path]
            elif (
                task['data']['name'] in ['update_zookeeper_flag'] and
                isinstance(set_zk_values, dict) and
                len(set_zk_values) > 0
            ):
                zk_node = task['data']['params']['node']
                zk_action = task['data']['params']['action']
                if zk_node in set_zk_values and zk_action == 'set':
                    zk_data = set_zk_values[zk_node]
                    logging.debug('Perform set operation for zk node {} with content: {}'.format(
                        zk_node,
                        zk_data
                    ))
                    recipe_data['content']['tasks'][_id]['data']['vars']['action'] = zk_action
                    recipe_data['content']['tasks'][_id]['data']['vars']['node'] = zk_node
                    recipe_data['content']['tasks'][_id]['data']['vars']['data'] = zk_data
        return self._post_url(url, json.dumps(recipe_data))

    def _get_url(self, url, **kwargs):
        try:
            r = self._session.get(url, **kwargs)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise NannyApiException('Failed to get url "{}"\nError: {}'.format(url, e))
        return r.json()

    def _delete_url(self, url, **kwargs):
        try:
            r = self._session.delete(url, **kwargs)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise NannyApiException('Failed to delete url "{}"\nError: {}'.format(url, e))
        if r.status_code == http_client.NO_CONTENT:
            return {}
        return r.json()

    def _post_url(self, url, data):
        try:
            r = self._session.post(url=url, data=data)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise NannyApiException('Failed to post url "{}"\nError: {}'.format(url, e))
        return r.json()

    def _put_url(self, url, data):
        try:
            r = self._session.put(url=url, data=data)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise NannyApiException('Failed to put url "{}"\nError: {}'.format(url, e))
        return r.json()

    def _deploy(self, service_id, snapshot_id, recipe='common', comment=None, prepare_recipe=None, ignore_disabled=False):
        if comment is None:
            comment = 'Deploy new version'
        target_state = self.get_service_target_state(service_id)
        if ignore_disabled or target_state['content']['is_enabled']:
            snapshot_to_deploy_id = snapshot_id
            payload = {
                'type': 'SET_TARGET_STATE',
                'content': {
                    'is_enabled': True,
                    'snapshot_id': snapshot_to_deploy_id,
                    'comment': comment,
                    'recipe': recipe,
                },
            }
            if prepare_recipe is not None:
                payload['content']['prepare_recipe'] = prepare_recipe
            return self.create_event(service_id, payload)

    def shutdown(self, service_id, comment=None):
        if comment is None:
            comment = 'Shutdown service'
        target_state = self.get_service_target_state(service_id)
        if target_state['content']['is_enabled']:
            payload = {
                'type': 'SET_TARGET_STATE',
                'content': {
                    'is_enabled': False,
                    'comment': comment,
                },
            }
            return self.create_event(service_id, payload)

    @staticmethod
    def get_updated_files_info(files, file_updates, create_if_not_exists=False):
        """
        Updates files info according to file_updates dict

        :param files: service resources description
        :param file_updates: dict with new resource info with local_path string as key and one of (SandboxFileUpdate, StaticFileUpdate) as value
        :param create_if_not_exists: flag, if set, resources with paths not present in current resource configuration will be added.
            Otherwise, these resources will be ignored
        """
        updated_files = deepcopy(files)
        updated_paths = set()
        for _file in updated_files:
            local_path = _file["local_path"]
            if local_path in file_updates:
                file_update = file_updates[local_path]
                _file.update(file_update._asdict())
                updated_paths.add(local_path)

        if create_if_not_exists:
            for non_updated_file_path in set(file_updates.keys()) - updated_paths:
                file_update = file_updates[non_updated_file_path]
                updated_files.append(file_update._asdict())
        else:
            logging.warning("Resources with local_paths %s are not present in service and won't be created",
                            ", ".join(set(file_updates.keys()) - updated_paths))

        return updated_files

    def update_service_files(self, service_id, sandbox_file_updates=None, static_file_updates=None, comment=None, create_if_not_exists=False):
        """
        Updates files info according to file_updates dicts and creates new snapshot.

        :param service_id: service to update files
        :param sandbox_file_updates: dict with new resource info with local_path string as key and SandboxFileUpdate as value
        :param static_file_updates: dict with new resource info with local_path string as key and StaticFileUpdate as value
        :param create_if_not_exists: flag, if set, resources with paths not present in current resource configuration will be added.
            Otherwise, these resources will be ignored
        """
        resources = self.get_service_resources(service_id)
        snapshot_id = resources['snapshot_id']

        update_comment = ""

        if sandbox_file_updates:
            resources['content']['sandbox_files'] = self.get_updated_files_info(
                resources['content']['sandbox_files'], sandbox_file_updates,
                create_if_not_exists=create_if_not_exists
            )
            update_comment += "\n".join([
                "Updating sandbox file {local_path} to {update.resource_type}({update.resource_id})".format(
                    local_path=local_path, update=update
                )
                for local_path, update in sandbox_file_updates.items()
            ])
        if static_file_updates:
            resources['content']['static_files'] = self.get_updated_files_info(
                resources['content']['static_files'], static_file_updates,
                create_if_not_exists=create_if_not_exists
            )
            update_comment += "\n".join([
                "Updating static file {}".format(local_path) for local_path in static_file_updates
            ])

        logging.info("Updates are: %s", update_comment)

        data = self.update_service_resources(service_id, {
            'content': resources['content'],
            'comment': comment or update_comment,
            'snapshot_id': snapshot_id,
        })

        return data

    def update_service_sandbox_resources(self, service_id, task_type, task_id, deploy=False,
                                         recipe='common', comment=None, deploy_comment=None,
                                         raise_if_not_changed=False, prepare_recipe=None):
        """
        Обновляет всё Sandbox-ресурсы сервиса (будь то sandbox file или sandbox shard),
        обновляя task_id у ресурсов с типом task_type.

        :param service_id: сервис, который нужно обновить
        :param task_type: тип таска для обновления. Все ресурсы в конфигурации сервиса, ссылающиеся на этот тип таска
                          будут обновлены.
        :param task_id: новый ID таска, который нужно подставить в подходящие по типу таска ресурсы.
        :param deploy: выкатывать ли сервис после обновления
        :param recipe: рецепт, с которым выкатываться.
        :param comment: комментарий к обновлению конфигурации ресурса
        :param deploy_comment: комментарий к задаче выкладки
        :param raise_if_not_changed: бросать исключение если нечего обновлять
        :param prepare_recipe: id рецепта, c которым делать принесение данных.

        :rtype: None

        В комментариях доступны следующие подстановки (str.format(<name>=<val)):
            {updates}   список "типов" обновленных ресурсов сервиса и их количество через запятую:
                            sandbox file
                            sandbox shard
                        Пример:
                            sandbox file: 2, sandbox shard: 1

            {files}     список имен обновленных файлов sandbox через запятую
                        Пример:
                            code-core.tar.gz, code-config.conf, additional-data.tar.gz

            {shard}     идентификатор обновленного шарда: тип sandbox-ресурса для SANDBOX_SHARD

            {task_id}   новый ID таска

            {task_type} тип таска, по которому производится обновление
        """
        resources = self.get_service_resources(service_id)

        snapshot_id = resources['snapshot_id']

        updates = {'sandbox file': [],
                   'sandbox shard': [],
                   'sandbox shardmap': []}

        updated = False

        # пытаемся обновить sandbox-шард
        sandbox_bsc_shard = resources['content'].get('sandbox_bsc_shard')
        if sandbox_bsc_shard:
            if sandbox_bsc_shard['chosen_type'] == 'SANDBOX_SHARD':
                if sandbox_bsc_shard['task_type'] == task_type:
                    updates['sandbox shard'].append(sandbox_bsc_shard['resource_type'])

                    sandbox_bsc_shard['task_id'] = task_id
                    updated = True

            elif sandbox_bsc_shard['chosen_type'] == 'SANDBOX_SHARDMAP':
                if sandbox_bsc_shard['sandbox_shardmap']['task_type'] == task_type:
                    updates['sandbox shardmap'].append(sandbox_bsc_shard['sandbox_shardmap']['resource_type'])

                    sandbox_bsc_shard['sandbox_shardmap']['task_id'] = task_id
                    updated = True

        # пытаемся обновить sandbox-файл
        for sandbox_file in resources['content']['sandbox_files']:
            if sandbox_file['task_type'] == task_type:
                updates['sandbox file'].append(sandbox_file['local_path'])

                sandbox_file['task_id'] = task_id
                # SWAT-3885: unset old resource_id
                sandbox_file.pop('resource_id', None)
                updated = True

        if not updated and raise_if_not_changed:
            raise errors.TaskFailure(
                'There is no task type {0} in nanny service {1}'.format(task_type, service_id)
            )

        # Высчитываем агрегированную статистику по обновлениям конфигурации для подстановки в комментарии
        updates_stats = []
        for key, val in updates.items():
            if val:
                updates_stats.append("{}: {}".format(key, len(val)))

        if comment is None:
            comment = 'Update {files}, {shard}: update {task_type} task id to {task_id}'

        comment = comment.format(
            updates=', '.join(updates_stats),
            files=', '.join(updates['sandbox file']),
            shard=', '.join(
                updates['sandbox shard'] or
                updates['sandbox shardmap']
            ),
            task_type=task_type,
            task_id=task_id
        )

        data = self.update_service_resources(service_id, {
            'content': resources['content'],
            'comment': comment,
            'snapshot_id': snapshot_id,
        })
        if deploy:
            if deploy_comment:
                deploy_comment = deploy_comment.format(
                    updates=', '.join(updates_stats),
                    files=', '.join(updates['sandbox file']),
                    shard=', '.join(
                        updates['sandbox shard'] or
                        updates['sandbox shardmap']
                    ),
                    task_type=task_type,
                    task_id=task_id
                )

            snapshot_to_deploy_id = data['runtime_attrs']['_id']
            return self._deploy(service_id, snapshot_to_deploy_id, recipe=recipe, comment=deploy_comment,
                                prepare_recipe=prepare_recipe)

        return None

    def _update_service_sandbox_bsc_shard_resource(self, service_id, task_type, task_id, deploy, recipe, comment,
                                                   deploy_comment, shard_type, prepare_recipe=None):
        resources = self.get_service_resources(service_id)

        snapshot_id = resources['snapshot_id']
        sandbox_bsc_shard = resources['content'].get('sandbox_bsc_shard')
        if not sandbox_bsc_shard:
            raise NannyApiException('Service does not have a shard')
        if sandbox_bsc_shard['chosen_type'] != shard_type:
            raise NannyApiException(
                'Service has a shard of different type: {}'.format(sandbox_bsc_shard['chosen_type'])
            )

        if shard_type == 'SANDBOX_SHARD':
            sandbox_task_info = sandbox_bsc_shard
        elif shard_type == 'SANDBOX_SHARDMAP':
            sandbox_task_info = sandbox_bsc_shard.get('sandbox_shardmap')
        else:
            raise NannyApiException('Unknown shard type: {}'.format(shard_type))

        if sandbox_task_info['task_type'] != task_type:
            raise NannyApiException(
                'Service has a different shard resource type: {}'.format(sandbox_task_info['task_type'])
            )
        if sandbox_task_info['task_id'] == task_id:  # Do not create new snapshot and don't deploy
            logging.debug("Service has correct shard task id: {}".format(task_id))
            return None

        old_task_id = sandbox_task_info['task_id']
        sandbox_task_info['task_id'] = task_id

        if comment is None:
            comment = 'Update shard: change {} release from {} to {}'.format(task_type, old_task_id, task_id)
        data = self.update_service_resources(service_id, {
            'content': resources['content'],
            'comment': comment,
            'snapshot_id': snapshot_id,
        })
        if deploy:
            snapshot_to_deploy_id = data['runtime_attrs']['_id']
            return self._deploy(service_id, snapshot_to_deploy_id, recipe=recipe, comment=deploy_comment,
                                prepare_recipe=prepare_recipe)

    def update_service_sandbox_bsc_shard(self, service_id, task_type, task_id, deploy=False, recipe='common',
                                         comment=None, deploy_comment=None):
        return self._update_service_sandbox_bsc_shard_resource(service_id, task_type, task_id, deploy, recipe,
                                                               comment, deploy_comment, 'SANDBOX_SHARD')

    def update_service_sandbox_bsc_shardmap(self, service_id, task_type, task_id, deploy=False, recipe='common',
                                            comment=None, deploy_comment=None):
        return self._update_service_sandbox_bsc_shard_resource(service_id, task_type, task_id, deploy, recipe,
                                                               comment, deploy_comment, 'SANDBOX_SHARDMAP')

    def update_service_sandbox_file(self, service_id, task_type, task_id, deploy=False,
                                    recipe='common', comment=None, deploy_comment=None,
                                    skip_not_existing_resources=False, prepare_recipe=None,
                                    allow_empty_changes=False, resource_types=()):
        """
        Обновляет все sandbox file ресурсы сервиса, обновляя task_id у ресурсов с типом task_type.

        :param service_id: сервис, который нужно обновить
        :param task_type: тип таска для обновления. Все ресурсы в конфигурации сервиса, ссылающиеся на этот тип таска
                          будут обновлены.
        :param task_id: новый ID таска, который нужно подставить в подходящие по типу таска ресурсы.
        :param deploy: выкатывать ли сервис после обновления
        :param recipe: рецепт, с которым выкатываться.
        :param comment: комментарий к обновлению конфигурации ресурса
        :param deploy_comment: комментарий к задаче выкладки
        :param raise_if_not_changed: бросать исключение если нечего обновлять
        :param prepare_recipe: id рецепта, c которым делать принесение данных.
        :param resource_types: типы ресурсов, которые необходимо обновить. Пустое значение интерпретируется как "все ресурсы".

        :returns: dict -- deploy result or resource update result depending on deploy param
        :rtype: dict

        В комментариях доступны следующие подстановки (str.format(<name>=<val)):
            {updates}   список "типов" обновленных ресурсов сервиса и их количество через запятую:
                            sandbox file
                            sandbox shard
                        Пример:
                            sandbox file: 2, sandbox shard: 1

            {files}     список имен обновленных файлов sandbox через запятую
                        Пример:
                            code-core.tar.gz, code-config.conf, additional-data.tar.gz

            {shard}     идентификатор обновленного шарда: тип sandbox-ресурса для SANDBOX_SHARD

            {task_id}   новый ID таска

            {task_type} тип таска, по которому производится обновление
        """
        resources = self.get_service_resources(service_id)
        snapshot_id = resources['snapshot_id']

        sb_rest_client = rest.Client()
        task_resources = sb_rest_client.resource.read(
            task_id=task_id,
            limit=1000,  # more than thousand resources in one task is nonsense
        )["items"]
        task_resource_map = {resource["type"]: resource for resource in task_resources}

        if resource_types:
            task_resource_map = {res_type: res for res_type, res in task_resource_map.items() if res_type in resource_types}

        old_task_id = None
        old_resource_type = []
        for sandbox_file in resources['content']['sandbox_files']:
            if sandbox_file['task_type'] != task_type:
                continue

            if sandbox_file['resource_type'] in task_resource_map:
                old_task_id = sandbox_file['task_id']
                sandbox_file['task_id'] = task_id
                sandbox_file['resource_id'] = str(task_resource_map[sandbox_file['resource_type']]["id"])
                old_resource_type.append(sandbox_file['resource_type'])
                continue

            if not skip_not_existing_resources:
                raise NannyApiException('Task does not have {} resource'.format(sandbox_file['resource_type']))

            logging.warning('Skip update {} resource of {} task (id {}) in service {}'.format(
                sandbox_file['resource_type'], task_type, task_id, service_id)
            )

        if old_task_id is None:
            if allow_empty_changes:
                return
            raise NannyApiException('Service {} does not have sandbox file created by task of "{}" type'.format(service_id, task_type))

        if comment is None:
            comment = 'Change {} for {} release from {} to {}'.format(
                task_type, ', '.join(old_resource_type), old_task_id, task_id
            )
        data = self.update_service_resources(service_id, {
            'content': resources['content'],
            'comment': comment,
            'snapshot_id': snapshot_id,
        })
        if deploy:
            snapshot_to_deploy_id = data['runtime_attrs']['_id']
            return self._deploy(service_id, snapshot_to_deploy_id, recipe=recipe, comment=deploy_comment,
                                prepare_recipe=prepare_recipe)
        else:
            return data

    def update_service_sandbox_url_file(self, service_id, resource_type, task_id, deploy=False,
                                        recipe='common', comment=None, deploy_comment=None, prepare_recipe=None):
        """ Update http://proxy.sandbox.yandex-team.ru/last/<resource_type>/<file_path>&task_id=<task_id> """

        resources = self.get_service_resources(service_id)
        snapshot_id = resources['snapshot_id']

        old_task_id = None
        for url_file in resources['content']['url_files']:
            if url_file['url'].find('proxy.sandbox.yandex-team.ru/last/{}'.format(resource_type)) != -1:
                match = re.search(r'task_id=([0-9]+)', url_file['url'])
                if not match:
                    raise NannyApiException(
                        'No task_id found in url {} for resource type {}'.format(url_file['url'], resource_type)
                    )
                old_task_id = match.group(1)
                url_file['url'] = url_file['url'].replace(old_task_id, str(task_id))

        if comment is None:
            comment = 'Change {} release from {} to {}'.format(resource_type, old_task_id, task_id)
        data = self.update_service_resources(service_id, {
            'content': resources['content'],
            'comment': comment,
            'snapshot_id': snapshot_id,
        })
        if deploy:
            snapshot_to_deploy_id = data['runtime_attrs']['_id']
            return self._deploy(service_id, snapshot_to_deploy_id, recipe=recipe, comment=deploy_comment,
                                prepare_recipe=prepare_recipe)

    def update_service_common_tags(self, service_id, tags, replace=False, deploy=False,
                                   recipe='common', comment='', deploy_comment=None, prepare_recipe=None):
        instances = self.get_service_instances(service_id)
        # One can use common tags only with EXTENDED_GENCFG_GROUPS now
        # Will be fixed in SWAT-2358
        if instances['content']['chosen_type'] != 'EXTENDED_GENCFG_GROUPS':
            raise NannyApiException("One can use common tags only with EXTENDED_GENCFG_GROUPS (SWAT-2358)")

        common_tags = instances['content']['extended_gencfg_groups']['tags']

        logging.debug("Old tags: %s", common_tags)
        if replace:
            new_tags = tags
        else:
            new_tags = list(set(common_tags) | set(tags))
        instances['content']['extended_gencfg_groups']['tags'] = new_tags

        logging.debug("New tags: %s", new_tags)
        data = self.update_service_instances(service_id, {
            'content': instances['content'],
            'comment': comment,
            'snapshot_id': instances['snapshot_id'],
        })

        if deploy:
            snapshot_to_deploy_id = data['runtime_attrs']['_id']
            return self._deploy(service_id, snapshot_to_deploy_id, recipe=recipe, comment=deploy_comment,
                                prepare_recipe=prepare_recipe)

    def filter_tickets(self, query):
        url = '{}/api/tickets/FindTickets/'.format(self._api_url)
        return self._post_url(url, data=json.dumps(query))["value"]

    def find_releases(self, request):
        url = '{}/api/tickets/FindReleases/'.format(self._api_url)
        return self._post_url(url, data=json.dumps(request))["value"]

    def get_release(self, release_id):
        url = '{}/api/tickets/GetRelease/'.format(self._api_url)
        return self._post_url(url, data=json.dumps({'releaseId': release_id}))["value"]

    def get_ticket(self, ticket_id):
        url = '{}/api/tickets/GetTicket/'.format(self._api_url)
        return self._post_url(url, data=json.dumps({'id': ticket_id}))["value"]

    def create_ticket(self, queue_id, title, description, responsible, copy_to, urgent=False):
        url = '{}/api/tickets/CreateTicket/'.format(self._api_url)
        post_data = {
            "spec": {
                "queueId": queue_id,
                "cc": copy_to,
                "title": title,
                "desc": description,
                "isUrgent": urgent,
                "responsible": responsible,
            }
        }
        return self._post_url(url, json.dumps(post_data))

    def update_ticket_status(self, ticket_id, status, comment):
        url = '{}/api/tickets/UpdateTicketStatus/'.format(self._api_url)
        post_data = {
            "id": ticket_id,
            "status": {"status": status},
            "comment": comment,
        }
        return self._post_url(url, json.dumps(post_data))

    @staticmethod
    def _retrieve_instancectl_release_version(resp):
        """
        :rtype: str | types.NoneType
        """
        # It may be retry, in this case we must not raise exception
        data = resp.json()
        if resp.status_code == http_client.CONFLICT:
            msg = data.get('message', '')
            occ = re.findall(r'(INSTANCECTL_RELEASE[^ ]+)', msg)
            if occ:
                return occ[0]
        elif 200 <= resp.status_code < 300:
            return data['value']['id']

        # If its not 409 and not 2xx, than possibly something went wrong
        resp.raise_for_status()

    def create_instancectl_release(self, data):
        """
        Creates InstanceCtl release via Nanny RPC API, returns created release id.

        :type data: dict
        :rtype: str | types.NoneType
        """
        url = '{}/api/tickets/CreateRelease/'.format(self._api_url)
        dump = json.dumps(data)

        try:
            r = self._session.post(url=url, data=dump)
        except Exception as e:
            raise NannyApiException('Failed to post url "{}"\nError: {}'.format(url, e))

        try:
            v = self._retrieve_instancectl_release_version(r)
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise NannyApiException('Failed to post url "{}"\nError: {}'.format(url, e))
        return v
