# -*- coding: utf-8 -*-
import datetime
import json
import logging
import six

from sandbox import sdk2

from sandbox.common import config
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox.projects.common.nanny import client
from sandbox.projects.common.nanny import const
from sandbox.sandboxsdk import parameters as sp

from .client import NannyClient  # noqa


STARTREK_TICKET_IDS_KEY = const.STARTREK_TICKET_IDS_KEY

CONTENT_KEY = 'content'
RESOURCES_KEY = 'resources'
BSC_SHARD_KEY = 'sandbox_bsc_shard'
SHARDMAP_KEY = 'sandbox_shardmap'
RELEASE_TO_NANNY_KEY = 'release_to_nanny'
RELEASE_TO_ADM_NANNY_KEY = 'release_to_adm_nanny'


class StartrekTicketIdsParameter(sp.SandboxStringParameter):
    name = STARTREK_TICKET_IDS_KEY
    description = 'Startrek ticket ids'
    default_value = ''
    required = False

    @classmethod
    def cast(cls, value):
        if value is None:
            return []
        elif isinstance(value, list):
            return value
        elif isinstance(value, six.string_types):
            value = value.strip()
            if not value:
                return []
            return value.split(',')
        else:
            raise ValueError('Must be either a list of strings or a comma-separated string, '
                             'not {!r} of type {}'.format(value, type(value)))


class StartrekTicketIdsParameter2(sdk2.parameters.List):
    name = STARTREK_TICKET_IDS_KEY
    description = 'Startrek ticket ids'
    required = False
    value_type = sdk2.parameters.String


class WebHookUrlParameter(sp.ListRepeater, sp.SandboxStringParameter):
    name = const.WEBHOOK_URLS_KEY
    description = 'Webhook on release closing'
    default_value = ''
    required = False


class WebHookUrlParameter2(sdk2.parameters.List):
    name = const.WEBHOOK_URLS_KEY
    description = 'Webhook on release closing'
    default = []
    value_type = sdk2.parameters.Url
    required = False


class WebHookTypeParameter(sp.SandboxSelectParameter):
    name = const.WEBHOOK_TYPE_KEY
    description = 'Webhook type'
    choices = [
        ('RELEASE_ONLY', 'RELEASE_ONLY'),
        ('RELEASE_WITH_TICKETS', 'RELEASE_WITH_TICKETS'),
    ]
    default_value = 'RELEASE_ONLY'
    required = False


class WebHookTypeParameter2(sdk2.parameters.String):
    name = const.WEBHOOK_TYPE_KEY
    choices = [
        ('RELEASE_ONLY', 'RELEASE_ONLY'),
        ('RELEASE_WITH_TICKETS', 'RELEASE_WITH_TICKETS'),
    ]
    default = 'RELEASE_ONLY'
    required = False


class LabelsParameter(sp.DictRepeater, sp.SandboxStringParameter):
    name = const.LABELS
    description = 'Release labels'
    required = False


class LabelsParameter2(sdk2.parameters.Dict):
    name = const.LABELS
    description = 'Release labels'
    required = False


class ComponentParameter(sp.SandboxStringParameter):
    name = const.COMPONENT_KEY
    description = 'Release component'
    default_value = ''
    required = False


class ComponentParameter2(sdk2.parameters.String):
    name = const.COMPONENT_KEY
    description = 'Release component'
    required = False


class DuplicatePolicyTypeParameter(sp.SandboxStringParameter):
    name = const.DUPLICATE_POLICY_TYPE_KEY
    description = 'Release duplicate policy type'
    required = False


class DuplicatePolicyTypeParameter2(sdk2.parameters.String):
    name = const.DUPLICATE_POLICY_TYPE_KEY
    description = 'Release duplicate policy type (https://nda.ya.ru/3UVnKk)'
    required = False


class ReleaseToNannyParameter(sp.SandboxBoolParameter):
    name = RELEASE_TO_NANNY_KEY
    description = 'Enable release to Nanny'
    required = False
    default_value = False


class ReleaseToAdmNannyParameter(sp.SandboxBoolParameter):
    name = RELEASE_TO_ADM_NANNY_KEY
    description = 'Enable release to Adm-Nanny'
    required = False
    default_value = False


class ReleaseToNannyTask:
    """
    Класс-mixin для поддержки передачи релизов в Nanny. Переопределяет метод on_release.
    """

    def get_nanny_release_info(self, additional_parameters):
        release_info = self.get_release_info()

        email_notifications = additional_parameters.get('email_notifications', {})
        to = email_notifications.get('to', [])
        cc = email_notifications.get('cc', [])
        # если нам передали строку, превращаем её в список из одного элемента
        if not isinstance(to, list):
            to = [to]
        if not isinstance(cc, list):
            cc = [cc]

        cc = list(self.filter_cc(cc))

        startrek_ticket_ids = (
            self.ctx.get(STARTREK_TICKET_IDS_KEY) or
            additional_parameters.get(STARTREK_TICKET_IDS_KEY, [])
        )
        webhook_urls = (
            self.ctx.get(const.WEBHOOK_URLS_KEY) or
            additional_parameters.get(const.WEBHOOK_URLS_KEY, [])
        )
        webhook_type = (
            self.ctx.get(const.WEBHOOK_TYPE_KEY) or
            additional_parameters.get(const.WEBHOOK_TYPE_KEY, WebHookTypeParameter.default_value)
        )
        labels = (
            self.ctx.get(const.LABELS) or
            additional_parameters.get(const.LABELS, {})
        )
        now = datetime.datetime.utcnow().isoformat()
        spec = {
            'title': additional_parameters.get(const.RELEASE_SUBJECT_KEY),
            'desc': additional_parameters.get(const.RELEASE_COMMENTS),
            'type': 'SANDBOX_RELEASE',
            'component': self.ctx.get('component') or additional_parameters.get('component') or '',
            'sandboxRelease': {
                # Must be '2017-05-11T07:20:31.691742Z'
                'creationTime': now + 'Z',
                'taskType': str(self.type),
                'taskId': str(self.id),
                'releaseAuthor': str(additional_parameters.get('releaser', 'anonymous')),
                'desc': additional_parameters.get(const.RELEASE_COMMENTS, ''),
                'title': additional_parameters.get(const.RELEASE_SUBJECT_KEY, ''),
                'releaseType': additional_parameters['release_status'],
                'resourceTypes': [i['type'] for i in release_info['release_resources']],
                'emailNotifySettings': {
                    'to': to,
                    'cc': cc,
                },
                'resources': [
                    {
                        'id': r['id'],
                        'type': r['type'],
                        'description': r['description'],
                        'skynetId': r['skynet_id'],
                        'httpUrl': r['http_url'],
                        'arch': r['arch'],
                        'fileMd5': r['file_md5'],
                        'releasers': r['releasers'],
                    } for r in release_info['release_resources']
                ],
            },
            'startrekTicketIds': startrek_ticket_ids,
            'webhook': [
                {
                    'url': i,
                    'type': webhook_type,
                } for i in webhook_urls],
        }
        duplicate_policy_type = (
            self.ctx.get('duplicate_policy_type') or
            additional_parameters.get('duplicate_policy_type')
        )
        if duplicate_policy_type:
            spec['duplicate_policy_type'] = duplicate_policy_type
        return {
            'spec': spec,
            'meta': {
                'labels': labels,
            },
        }

    def filter_cc(self, cc):
        """
        Filter cc list skipping any non-string elements

        :param cc: an iterable of notification receivers
        :return: A generator of the filtered cc elements
        """

        invalid_elements = set([])

        for item in cc:
            if not isinstance(item, six.string_types):
                logging.error("A non-string element found in the cc list: %s. Skipping it", item)
                invalid_elements.add(item)
                continue
            yield item

        if invalid_elements:
            self.set_info(
                "These cc elements are invalid and were skipped during cc processing: {}".format(
                    ", ".join(invalid_elements),
                ),
            )

    def get_nanny_oauth_token(self):
        return None

    @property
    def nanny_client(self):
        oauth_token = self.get_nanny_oauth_token()
        return client.NannyClient(api_url=const.NANNY_API_URL, oauth_token=oauth_token)

    def on_release(self, additional_parameters):
        self.mark_released_resources(additional_parameters["release_status"])
        registry = config.Registry()
        if registry.common.installation != ctm.Installation.PRODUCTION:
            logging.info('Current installation is not PRODUCTION but %s, will not send release to Nanny',
                         registry.common.installation)
            return

        logging.info(
            'Gathering information about release of task %s with release parameters %r to send it to Nanny',
            self.id, additional_parameters
        )
        data = self.get_nanny_release_info(additional_parameters)
        logging.info('Sending release of task %s to Nanny', self.id)
        logging.info('Release payload: %s', json.dumps(data, indent=4))
        result = self.nanny_client.create_release2(data)
        logging.info('Release of task %s has been sent to Nanny', self.id)
        release_id = result['value']['id']
        self.ctx.setdefault("nanny_release_requests", []).append(release_id)
        release_link = const.RELEASE_REQUEST_TEMPLATE.format(
            nanny_api_url=const.NANNY_API_URL,
            release_request_id=release_id,
        )
        self.set_info(
            'Nanny release <a href="{}">{}</a> was created.'.format(release_link, release_id),
            do_escape=False
        )

    def on_gencfg_release(self, release_description, additional_parameters):
        release_info = self.get_release_info()
        release_payload = release_description.copy()
        release_payload['sandbox_release'] = {
            'task_id': str(self.id),
            'task_type': str(self.type),
            'release_author': str(additional_parameters.get('releaser', 'anonymous')),
            'release_type': additional_parameters.get('release_status', 'unknown'),
            'resource_types': [r['type'] for r in release_info['release_resources'] if 'type' in r],
        }
        logging.info('Sending release of task %s to Nanny', self.id)
        logging.info('Gencfg release payload: %s', json.dumps(release_payload, indent=4))
        self.nanny_client.create_gencfg_release(release_payload)
        logging.info('Gencfg r of task %s has been sent to Nanny', self.id)


class ReleaseToNannyTask2():
    """
    Mixin class for support sending release requests to Nanny. Override method on_release.
    """

    def get_release_resources(self):
        all_res = sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0)
        resources = [
            {
                'id': str(r.id),
                'type': r.type.name,
                'description': r.description,
                'skynetId': r.skynet_id,
                'httpUrl': r.http_proxy or '',
                'arch': r.arch or '',
                'fileMd5': r.md5 or '',
                'releasers': r.releasers or [],
            } for r in all_res if r.type.releasable
        ]
        logging.info('List of resources to release:\n{}'.format(resources))
        return resources

    def get_nanny_release_info(self, additional_parameters):
        email_notifications = additional_parameters.get('email_notifications', {})
        to = email_notifications.get('to', [])
        cc = email_notifications.get('cc', [])
        # если нам передали строку, превращаем её в список из одного элемента
        if not isinstance(to, list):
            to = [to]
        if not isinstance(cc, list):
            cc = [cc]

        component = self.get_nanny_task_component(additional_parameters)
        webhook_urls = self.get_nanny_webhook_urls(additional_parameters)
        webhook_type = self.get_nanny_webhook_type(additional_parameters)
        startrek_ticket_ids = self.get_nanny_startrek_ticket_ids(additional_parameters)
        duplicate_policy_type = self.get_nanny_duplicate_policy_type(additional_parameters)
        labels = self.get_nanny_labels()

        resources = self.get_release_resources()
        now = datetime.datetime.utcnow().isoformat()
        spec = {
            'title': additional_parameters.get(const.RELEASE_SUBJECT_KEY),
            'desc': additional_parameters.get(const.RELEASE_COMMENTS),
            'type': 'SANDBOX_RELEASE',
            'component': component,
            'sandboxRelease': {
                # Must be '2017-05-11T07:20:31.691742Z'
                'creationTime': now + 'Z',
                'taskType': str(self.type),
                'taskId': str(self.id),
                'releaseAuthor': str(additional_parameters.get('releaser', 'anonymous')),
                'desc': additional_parameters.get(const.RELEASE_COMMENTS, ''),
                'title': additional_parameters.get(const.RELEASE_SUBJECT_KEY, ''),
                'releaseType': additional_parameters['release_status'],
                'resourceTypes': [i['type'] for i in resources],
                'emailNotifySettings': {
                    'to': to,
                    'cc': cc,
                },
                'resources': resources,
            },
            'startrekTicketIds': startrek_ticket_ids,
            'webhook': [
                {
                    'url': url,
                    'type': webhook_type,
                } for url in webhook_urls
            ],
        }

        if duplicate_policy_type:
            spec['duplicate_policy_type'] = duplicate_policy_type

        return {
            'spec': spec,
            'meta': {
                'labels': labels,
            },
        }

    def get_nanny_task_component(self, additional_parameters):
        return self._get_parameter_or_context_value(name=const.COMPONENT_KEY, default=ComponentParameter.default_value)

    def get_nanny_webhook_type(self, additional_parameters):
        return self._get_parameter_or_context_value(name=const.WEBHOOK_TYPE_KEY, default=WebHookTypeParameter.default_value)

    def get_nanny_webhook_urls(self, additional_parameters):
        return self._get_parameter_or_context_value(name=const.WEBHOOK_URLS_KEY, default=[])

    def get_nanny_startrek_ticket_ids(self, additional_parameters):
        return (
            self._get_parameter_or_context_value(name=STARTREK_TICKET_IDS_KEY, default=None) or
            additional_parameters.get(STARTREK_TICKET_IDS_KEY, [])
        )

    def get_nanny_duplicate_policy_type(self, additional_parameters):
        return self._get_parameter_or_context_value(name=const.DUPLICATE_POLICY_TYPE_KEY, default=None)

    def get_nanny_labels(self):
        return self._get_parameter_or_context_value(name=const.LABELS, default={})

    def _get_parameter_or_context_value(self, name, default):
        value = getattr(self.Context, name)
        if value is ctm.NotExists:
            value = default

        return getattr(self.Parameters, name, value)

    def get_nanny_oauth_token(self):
        return None

    @property
    def nanny_client(self):
        oauth_token = self.get_nanny_oauth_token()
        return client.NannyClient(api_url=const.NANNY_API_URL, oauth_token=oauth_token)

    def on_release(self, additional_parameters):
        self.mark_released_resources(additional_parameters["release_status"])
        registry = config.Registry()
        if registry.common.installation != ctm.Installation.PRODUCTION:
            logging.info('Current installation is not PRODUCTION but %s, will not send release to Nanny',
                         registry.common.installation)
            return

        logging.info(
            'Gathering information about release of task %s with release parameters %r to send it to Nanny',
            self.id, additional_parameters
        )
        data = self.get_nanny_release_info(additional_parameters)
        logging.info('Sending release of task %s to Nanny', self.id)
        logging.info('Release payload: %s', json.dumps(data, indent=4))
        result = self.nanny_client.create_release2(data)['value']
        logging.info('Release of task %s has been sent to Nanny', self.id)
        release_id = result['id']
        if self.Context.nanny_release_requests is ctm.NotExists:
            self.Context.nanny_release_requests = []
        self.Context.nanny_release_requests.append(release_id)
        release_link = const.RELEASE_REQUEST_TEMPLATE.format(
            nanny_api_url=const.NANNY_API_URL,
            release_request_id=release_id,
        )
        self.set_info(
            'Nanny release request <a href="{}">{}</a> was created.'.format(release_link, release_id),
            do_escape=False
        )


def get_nanny_runtime_attrs(service_id, oauth_token, api_url=const.NANNY_API_URL):
    return client.NannyClient(api_url, oauth_token).get_service_active_runtime_attrs(service_id)


def get_nanny_runtime_shardmap_info(service_id, oauth_token):
    return get_nanny_runtime_attrs(service_id, oauth_token)[CONTENT_KEY][RESOURCES_KEY][BSC_SHARD_KEY][SHARDMAP_KEY]
