# -*- coding: utf-8 -*-

import os
import shutil
import traceback
import logging
import urllib
import re

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.common.errors import TaskError


class NannyReleaseResource(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Releases your resource to Nanny using @yandex-int/trendbox-ci.nanny-release"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Resource') as resource_block:
            release_resource = sdk2.parameters.Resource(
                'Release resource',
                description=u'Ресурс для релиза',
                required=True,
            )

        with sdk2.parameters.Group('Release') as release_block:
            release_identifier = sdk2.parameters.String(
                'Release identifier',
                description=u'Идентификатор релиза (ветка, тег и т.п.)',
                required=False,
            )

        with sdk2.parameters.Group('Nanny') as nanny_block:
            release_subject = sdk2.parameters.String(
                'Nanny title',
                description=u'Название релиза',
                default='Released by Sandbox task (NANNY_RELEASE_RESOURCE)',
                required=True,
            )
            release_comments = sdk2.parameters.String(
                'Nanny description',
                description=u'Описание релиза (обычно, changelog)',
                multiline=True,
            )
            webhook_type = nanny.WebHookTypeParameter2(
                'Webhook Type',
                description=u'Позволяет выбрать формат payload (https://nda.ya.ru/3UW7sc)',
            )
            webhook_urls = nanny.WebHookUrlParameter2(
                'Webhook URLs',
                description=u'Список URL\'ов куда Няня сделает POST-запрос после закрытия релиза',
            )
            duplicate_policy_type = nanny.DuplicatePolicyTypeParameter2(
                'Duplication Policy Type',
                description=u'Политика закрытия дубликатов в Няне (https://nda.ya.ru/3UVnKk)',
            )
            startrek_ticket_ids = nanny.StartrekTicketIdsParameter2(
                'Tracker Tickers IDs',
                description=u'Список тикетов в StartTrek в которые Няня оставит комментарии о состоянии релиза',
            )
            component = nanny.ComponentParameter2(
                'Component',
                description=u'Компонент релиза. Используется для поиска дубликатов.',
            )
            release_labels = nanny.LabelsParameter2(
                'Release Labels',
                description=u'Метки, которые будут добавлены к релизу (для фильтрации в Няне)',
            )
            email_notifications_to = sdk2.parameters.List('Users to notify by email (to)', sdk2.parameters.Staff)
            email_notifications_cc = sdk2.parameters.List('Users to notify by email (cc)', sdk2.parameters.Staff)

        with sdk2.parameters.Group('Webhook-handler microservice integration') as whh_integration:
            project_github_owner = sdk2.parameters.String(
                'GitHub owner',
                default='search-interfaces',
                description='Логин владельца репозитория или название организации',
            )
            project_github_repo = sdk2.parameters.String(
                'GitHub repo',
                default='frontend',
                description='Название репозитория',
            )
            project_git_base_ref = sdk2.parameters.String(
                'Project git base ref',
                description='Релизная ветка',
            )
            project_git_base_commit = sdk2.parameters.String(
                'Project git base commit',
                description='HEAD релизной ветки',
                default='0000000000000000000000000000000000000000'
            )
            use_arc = sdk2.parameters.Bool(
                'Use arc',
                description='Использовать сборки на arc',
                default=False
            )

    def _sync_resource(self):
        """
        Publishes resource as a child of this task
        """
        resource_type = str(type(self.Parameters.release_resource))

        sdk2.Resource[resource_type](
            task=self,
            description=self.Parameters.release_resource.description,
            path=self._copy_resource(self.Parameters.release_resource),
        )

    def _copy_resource(self, resource):
        """
        Downloads resource to the current task folder
        """
        self.set_info('Copy resource {}'.format(resource))

        source_file = str(sdk2.ResourceData(resource).path)
        target_file = str(self.path(os.path.basename(source_file)))

        try:
            self.set_info('Download from {} to {}'.format(source_file, target_file))
            copy_file_or_folder(source_file, target_file)
        except IOError:
            self.set_info(traceback.format_exc())
            raise TaskError('Error on copy resource {}'.format(resource))

        return target_file

    def on_save(self):
        super(NannyReleaseResource, self).on_save()

        # For microservices integration
        identifier = self.Parameters.release_identifier

        if identifier:
            regex = r"report(?:-images|-video|-news)?-templates/(r\d+/)?(.+)/(.+)@(.+)?"
            matches = re.findall(regex, identifier)

            if matches:
                groups = matches.pop()
                _, owner, repo, branch = groups

                self.Parameters.project_github_owner = owner
                self.Parameters.project_github_repo = repo
                self.Parameters.project_git_base_ref = 'release/{}'.format(branch)

            if identifier.startswith('releases/'):
                self.Parameters.use_arc = True
                self.Parameters.project_git_base_ref = identifier

    def on_execute(self):
        self._sync_resource()

    def on_release(self, additional_parameters):
        """
        В additional_parameters прилетают параметры из тела запроса в ручку /release. https://nda.ya.ru/3UWCVW

        :param additional_parameters: введенные данные из формы релизов
        :type additional_parameters: dict
        """

        release_type = additional_parameters['release_status'].lower()

        tags = set(map(lambda tag: tag.lower(), self.Parameters.tags) + [release_type])

        logging.debug('Releasing task in {release_type} with parameters: {parameters} and tags: {tags}'.format(
            release_type=release_type,
            parameters=additional_parameters,
            tags=tags,
        ))

        self.Parameters.tags = list(tags)

        sdk2.Task.on_release(self, additional_parameters)
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

    def get_nanny_webhook_urls(self, additional_parameters):
        release_type = additional_parameters['release_status'].lower()

        if release_type == 'stable':
            return ['https://webhook-handler.si.yandex-team.ru/v1/nanny/release-request-tickets-status-change']

        webhook_urls = self.Parameters.webhook_urls

        if not webhook_urls:
            # Webhook urls are not defined, sending default urls
            return ['https://webhook-handler.si.yandex-team.ru/v1/sandbox/release-check-static?{}'.format(
                urllib.urlencode({
                    'genisys-section': 'search-interfaces.releases.empty',
                    'beta-host': 'https://hamster.yandex.ru',
                    'stable-host': 'https://yandex.ru',
                })
            )]

        return webhook_urls

    def get_nanny_webhook_type(self, additional_parameters):
        return 'RELEASE_WITH_TICKETS'


def copy_file_or_folder(source_file, target_file):
    if os.path.isdir(source_file):
        shutil.copytree(source_file, target_file)
    else:
        shutil.copy(source_file, target_file)
