# coding: utf-8

"""
Useful network routines.

See also: sandbox.sandboxsdk.network module
"""

import socket
import logging
import random
from sandbox import common

from sandbox.projects.common import decorators


def get_my_ipv6(raise_class=None):
    """
    Obtain IPv6 address of myself (in particular, an LXC container address)
    Described in https://nda.ya.ru/3UXdvF (Sandbox cookbook)
    :return: ipv6 address
    """
    @decorators.retries(max_tries=3, delay=20, backoff=2, exceptions=(socket.error,), raise_class=raise_class)
    def _connect_socket_to_sandbox(sock):
        sock.connect(("sandbox.yandex-team.ru", 443))

    s = socket.socket(socket.AF_INET6)
    s.settimeout(5)
    _connect_socket_to_sandbox(s)
    ip_address = s.getsockname()[0]
    s.close()
    return ip_address


def get_free_port(max_tries=25):
    """
    See https://wiki.yandex-team.ru/sandbox/cookbook/#kakpoluchitdiapazongarantirovannosvobodnyxportov
    for magic constants references.
    :param max_tries: Number of attempts to pick free port.
    """
    for _ in range(max_tries):
        port = random.randint(15000, 25000)
        logging.debug('Checking if %s port is free', port)
        if common.utils.is_port_free(port):
            logging.debug('Port %s is free', port)
            return port
        else:
            logging.debug('Port %s is busy', port)

    raise common.errors.TemporaryError(
        'Failed to find free port after {} retries'.format(max_tries)
    )
