# -*- coding: utf-8 -*-

from sandbox.projects.common import utils
from sandbox.sandboxsdk import task as st
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import environments

import os
import shutil


class YtProxy(sp.SandboxStringParameter):
    name = 'proxy'
    description = 'YT proxy'
    required = True
    default_value = "hahn.yt.yandex.net"


class VaultEnv(sp.SandboxStringParameter):
    name = 'vault_env'
    description = 'YT token from Vault, format: vault_owner:vault_key'
    required = False


class Pool(sp.SandboxStringParameter):
    name = 'pool'
    description = 'Pool'
    required = False
    default_value = None


class ScriptUrl(sp.SandboxSvnUrlParameter):
    name = 'script_url'
    description = 'Script directory svn url'
    required = True


class ScriptCmdline(sp.SandboxStringParameter):
    name = 'cmdline'
    description = 'Script cmdline'
    required = True


class PythonBinary(sp.ResourceSelector):
    name = 'news_python'
    description = 'Resource with News python, leave empty if you want last released resource'
    required = False
    resource_type = 'NEWS_PYTHON'


def get_base_params():
    return [YtProxy, VaultEnv, Pool, ScriptUrl, ScriptCmdline, PythonBinary]


class YtScript(st.SandboxTask):
    '''
    Base class to run python scripts on YT
    '''

    environment = (environments.SvnEnvironment(), environments.PipEnvironment('yandex-yt'))
    cores = 1
    required_ram = 1024  # 1GB

    def get_scripts(self):
        scriptdir = self.path('script')
        if os.path.exists(scriptdir):
            shutil.rmtree(scriptdir)
        os.makedirs(scriptdir)
        svn_url = self.ctx[ScriptUrl.name]
        process.run_process(['svn', 'co', svn_url, scriptdir], log_prefix='checkout')
        return scriptdir

    def get_python(self):
        python_binary = 'python'
        python_resource_id = self.ctx.get(PythonBinary.name)
        if not python_resource_id:
            python_resource_id = utils.get_and_check_last_released_resource_id(
                PythonBinary.resource_type,
                arch=self.client_info['arch']
            )
        if python_resource_id:
            pytar = self.sync_resource(python_resource_id)
            python_binary = pytar
        return python_binary

    def get_token(self):
        token = None
        vault_env = self.ctx.get(VaultEnv.name)
        if vault_env:
            (owner, key) = vault_env.split(':')
            token = self.get_vault_data(owner, key)
        else:
            token = self.get_vault_data(self.owner, 'yt_token')
        return token

    def get_cmdline(self):
        return self.ctx.get(ScriptCmdline.name)

    def get_yt_proxy(self):
        return self.ctx.get(YtProxy.name)

    def get_pool(self):
        return self.ctx.get(Pool.name)
