# -*- coding: utf-8 -*-

from sandbox import sdk2, common
from sandbox.projects.common import utils
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.projects import resource_types
from YtScriptTask import PythonBinary

import os
import shutil
import logging


class YtScriptV2(sdk2.Task):
    '''
    Base class to run python scripts on YT (SDK2 version)
    '''

    class Requirements(sdk2.Task.Requirements):
        environment = (environments.SvnEnvironment(), environments.PipEnvironment('yandex-yt'))
        cores = 1
        ram = 1024  # 1GB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        yt_proxy = sdk2.parameters.String("YT proxy", required=True, default_value="hahn.yt.yandex.net")
        yt_token = sdk2.parameters.YavSecret("YT token from YAV")
        yt_token_field = sdk2.parameters.String("Field to extract from YAV for yt_token. Default: `YT_TOKEN`", default_value="YT_TOKEN")
        yt_pool = sdk2.parameters.String("Pool", default_value=None)
        script_url = sdk2.parameters.String("Script directory svn url", required=True)
        script_cmdline = sdk2.parameters.String("Script cmdline", required=True)
        script_patch = sdk2.parameters.String("Arcadia patch (arc patches not supported)", default_value="")
        news_python = sdk2.parameters.Resource("Resource with News python, leave emtpy if you want last released resource", resource_type=resource_types.NEWS_PYTHON)


    def get_scripts(self):
        scriptdir = str(self.path('script'))
        if os.path.exists(scriptdir):
            shutil.rmtree(scriptdir)
        os.makedirs(scriptdir)
        svn_url = self.Parameters.script_url

        sdk2.svn.Arcadia.check(svn_url)
        sdk2.svn.Arcadia.export(svn_url, scriptdir)

        arcadia_patch = common.encoding.force_unicode_safe(self.Parameters.script_patch)
        if arcadia_patch:
            arcadia_patch_path, is_zipatch = sdk2.svn.Arcadia.fetch_patch(arcadia_patch, str(self.path()))
            if arcadia_patch_path:
                sdk2.svn.Arcadia.apply_patch_file(scriptdir, arcadia_patch_path, is_zipatch, strip_depth=3)

        return scriptdir

    def get_python(self):
        python_binary = 'python'
        python_resource_id = self.Parameters.news_python
        if not python_resource_id:
            # XXX: Check, may not work in SDK2 task
            python_resource_id = utils.get_and_check_last_released_resource_id(
                PythonBinary.resource_type,
                # arch=self.arch
            )
        if python_resource_id:
            pytar = sdk2.ResourceData(sdk2.Resource[python_resource_id]).path
            python_binary = pytar
        return str(python_binary)

    def get_token(self):
        return self.Parameters.yt_token.data()[self.Parameters.yt_token_field]

    def get_cmdline(self):
        return self.Parameters.script_cmdline

    def get_yt_proxy(self):
        return self.Parameters.yt_proxy

    def get_pool(self):
        return self.Parameters.yt_pool

    def run_process_with_output_link(self, *args, **kwargs):
        process_name = kwargs.pop('process_name', 'Process')
        kwargs['wait'] = False
        proc = process.run_process(*args, **kwargs)
        logs_rid = getattr(getattr(channel.task, '_log_resource', None), 'id', None)
        if logs_rid is not None:
            res = channel.sandbox.get_resource(logs_rid)
            if res:
                url = '/'.join([res.proxy_url, proc.stdout_path_filename])
                self.set_info('{0} started. <a href="{1}" target="_blank">output</a>'.format(process_name, url), do_escape=False)
        proc.wait()
        process.check_process_return_code(proc)

