# -*- coding: utf-8 -*-

import os
import logging
import time
import tempfile
import json

import requests

from sandbox import sdk2

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths

from sandbox.projects import resource_types
from sandbox.projects.common.search import components as sc
from sandbox.projects.common import error_handlers as eh

import sandbox.projects.news.resources as resources


class ArrangedParameters(sdk2.Task.Parameters):
    arranged_executable = sdk2.parameters.Resource(
        'Executable', resource_type=resources.NEWS_APPHOST_ARRANGED_EXECUTABLE, required=True
    )
    state = sdk2.parameters.Resource(
        'state dump', resource_type=resource_types.SLAVE_NEWSD_STATE, required=False
    )
    recommender_config = sdk2.parameters.Resource(
        'config for recommender',
        # resource_types=resources.NEWS_RECOMMENDER_SYSTEM_CONFIG,  # FIXME: invalid argument (SANDBOX-6404)
        required=False
    )
    default_flags = sdk2.parameters.Resource(
        'config with default flags', resource_type=resources.NEWS_APPHOST_ARRANGED_DEFAULT_FLAGS, required=False
    )
    recommender_models = sdk2.parameters.Resource(
        'recommender models', resource_type=resources.NEWS_RECOMMENDER_MODELS_PACKAGE, required=False
    )


class Arranged(sc.SearchComponent):
    name = 'arranged'
    http_collection = ''

    def __init__(
        self, binary, port, recommender_config,
        default_flags_config, workdir,
        recommender_models, state=None
    ):
        super(Arranged, self).__init__(self)

        self.binary = binary
        self.state_port = port
        self.app_host_port = port + 1
        self.recommender_config = recommender_config
        self.default_flags_config = default_flags_config
        self.workdir = workdir
        self.recommender_models = recommender_models
        self.state = None
        self.dumpdir = None

        resource_config = {}
        if self.recommender_models:
            resource_config["recommender_models"] = []
            arr = resource_config["recommender_models"]
            for model in os.listdir(self.recommender_models):
                arr.append({})
                arr[-1]["name"] = model
                arr[-1]["path"] = os.path.join(self.recommender_models, model)
        self.resource_config = tempfile.mkstemp()[1]
        with open(self.resource_config, "w") as f:
            f.write(json.dumps(resource_config))

        data_server_config = {"inMemory": False, "numStates": 3}
        if state:
            self.state = state
            self.dumpdir = os.path.join(self.workdir, 'snapshots')
            os.mkdir(self.dumpdir)
            dumpfile = os.path.join(self.dumpdir, str(int(time.time() * 1e6)))
            logging.debug("Creating symlink %s => %s" % (self.state, dumpfile))
            paths.copy_path(self.state, dumpfile)
            data_server_config["snapshotDir"] = self.dumpdir
        self.data_server_config = tempfile.mkstemp()[1]
        with open(self.data_server_config, "w") as f:
            f.write(json.dumps(data_server_config))

    def start(self, isTestMode, now_time):
        arranged_log = paths.get_unique_file_name(paths.get_logs_folder(), "arranged.log")
        cmd = [
            self.binary,
            '--data-server-config', self.data_server_config,
            '--resource-config', self.resource_config,
            '--recommender-config', self.recommender_config,
            '--apphost-port', str(self.app_host_port),
            '--state-port', str(self.state_port),
            '-L', arranged_log,
            '--flags-config', self.default_flags_config
        ]
        if isTestMode:
            cmd.append('--test')
        if now_time:
            cmd += ['--now-time', str(now_time)]
        self.process_parameters = cmd
        self.process = process.run_process(cmd, log_prefix='news_arranged', wait=False)

    def get_apphost_port(self):
        return self.app_host_port

    def wait(self, timeout=180):
        logging.info("Waiting for arranged to become ready for {} seconds".format(timeout))
        start = time.time()
        finish = start + timeout
        while time.time() < finish:
            if not self.is_running():
                logging.info("%s died", self.name)
                self._wait_coredump()
                self._process_post_mortem()
            try:
                r_state = requests.get('http://localhost:%d/is_alive' % self.state_port)
                r_app_host = requests.get('http://localhost:%d' % self.app_host_port)

                if r_state.status_code == 200 and r_app_host.status_code == 200:
                    time.sleep(20)
                    logging.info('Arranged is ready')
                    return
            except:
                pass
            time.sleep(1)
        if self.is_running():
            logging.info("Kill process %s", self.process.pid)
            os.kill(self.process.pid, 6)  # generate coredump
            self._wait_coredump()
        eh.check_failed("Cannot connect to {}, port {} in {} seconds".format(
            self.name, self.state_port, timeout
        ))

    def use_component(self, func):
        return func()

    def warmup_request(self):
        pass
