# -*- coding: utf-8 -*-

import os
import json
import difflib
from multiprocessing import Pool
from itertools import izip_longest
from base64 import b64decode

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.util import system_info
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils


def prettify(response_str, filter_debug=None):
    if response_str != "":
        response = json.loads(response_str)

        if response.get('answers'):
            answers = []
            for answer in response['answers']:
                if isinstance(answer, unicode):
                    answer = answer.encode('utf-8')

                if isinstance(answer, str):
                    try:
                        answer = json.loads(b64decode(answer))
                    except:
                        answer = None

                if isinstance(answer, dict) and "type" in answer and answer["type"] == "service_version":
                    answer = None

                if answer is not None:
                    answers.append(answer)

            response['answers'] = answers

        if filter_debug is not None:
            response = filter_debug(response)

        if "answers" in response and isinstance(response["answers"], list):
            response["answers"] = sorted(response["answers"])
        response_str = json.dumps(response, indent=4, sort_keys=True, ensure_ascii=False)
    return response_str


def compare(diff_path, query_number, query, l1, l2, save_html=False, filter_debug=None):
    r1 = prettify(l1, filter_debug)
    r2 = prettify(l2, filter_debug)

    d = difflib.unified_diff(r1.split('\n'), r2.split('\n'))
    diff = list(d)
    if len(diff) == 0:
        return 0

    with open(os.path.join(diff_path, "query_" + str(query_number) + ".txt"), 'w') as fd:
        fd.write("query #" + str(query_number) + "\n")
        fd.write(query + "\n")
        fd.write('\n'.join(diff).encode('utf-8'))

    if save_html:
        d = difflib.HtmlDiff()
        with open(os.path.join(diff_path, "query_" + str(query_number) + ".html"), 'w') as fd:
            fd.write(d.make_file(r1.split('\n'), r2.split('\n')).encode('utf-8'))
    return 1


class CompareApphostResponses(SandboxTask):
    '''
        Базовый класс, реализующий сравнение AppHost'овых ответов для различных демонов
    '''
    required_ram = 40 * 1024

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx['out_resource_id'] = self.init_resource()

    def init_resource(self):
        pass

    def get_first_responses(self):
        pass

    def get_second_responses(self):
        pass

    def get_queries(self):
        pass

    def save_html(self):
        return False

    def get_filter_debug_func(self):
        return None

    def on_execute(self):
        path_to_diffs = self._get_diff_path()
        make_folder(path_to_diffs)

        resp1_resourse_id = self.get_first_responses()
        resp2_resourse_id = self.get_second_responses()

        queries = self.load_queries(self.get_queries())

        # True - no diff
        self.ctx['compare_result'] = self.compare_responses(queries, resp1_resourse_id, resp2_resourse_id)
        if self.ctx['compare_result']:
            fu.write_file(os.path.join(path_to_diffs, 'no_diff_detected.txt'), 'no_diff_detected')

        self.mark_resource_ready(self.ctx['out_resource_id'])

    def load_queries(self, resource_id):
        self.sync_resource(resource_id)
        queries_resource = channel.sandbox.get_resource(resource_id)
        queries = []
        with open(queries_resource.path) as f:
            for line in f:
                queries.append(line.strip())
        self.ctx['num_of_queries'] = len(queries)
        return queries

    def read_file(self, file_name):
        with open(file_name) as fd:
            for line in fd:
                line = line.rstrip('\n')
                if line:
                    fields = line.split('\t')
                    response_str = fields[2]
                    response_str = unicode(response_str, errors='replace')
                    yield response_str

    def compare_responses(self, queries, resp1_resourse_id, resp2_resourse_id):
        diff_path = self._get_diff_path()

        self.sync_resource(resp1_resourse_id)
        self.sync_resource(resp2_resourse_id)
        resp1_resource = channel.sandbox.get_resource(resp1_resourse_id)
        resp2_resource = channel.sandbox.get_resource(resp2_resourse_id)

        if utils.check_resources_by_md5(resp1_resource, resp2_resource, self._get_diff_path()):
            return True

        results = []
        pool = Pool(processes=system_info()['ncpu'])
        range_num_q = xrange(0, self.ctx['num_of_queries'])
        for query_number, r1, r2 in izip_longest(range_num_q,
                                                 self.read_file(resp1_resource.path),
                                                 self.read_file(resp2_resource.path)):
            eh.verify(r1 is not None, "Size of Responses1 is smaller than size of Queries")
            eh.verify(r2 is not None, "Size of Responses2 is smaller than size of Queries")
            eh.verify(query_number is not None, "Responses are bigger than Queries")

            results.append(
                pool.apply_async(
                    compare,
                    (
                        diff_path,
                        query_number,
                        queries[query_number],
                        r1,
                        r2,
                        self.save_html(),
                        self.get_filter_debug_func()
                    )
                )
            )

        pool.close()
        pool.join()

        num_of_diffs = 0
        for result in results:
            num_of_diffs += result.get()
        self.ctx['num_of_diffs'] = num_of_diffs

        is_equal = not bool(num_of_diffs)
        return is_equal

    def _get_diff_path(self):
        return channel.sandbox.get_resource(self.ctx['out_resource_id']).path
