# -*- coding: utf-8 -*-

import os
import logging
import time
import tempfile
import json
import tarfile

from urllib2 import urlopen
from contextlib import closing

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths

from sandbox.projects.common.search import components as sc
from sandbox.projects.common import error_handlers as eh


def create_newsd_params():
    group_name = "Newsd params"

    class Params(object):
        class Binary(sp.ResourceSelector):
            name = "newsd_executable"
            description = "Executable"
            resource_type = [
                'SLAVE_NEWSD_EXECUTABLE'
            ]
            group = group_name
            requred = True

        class Config(sp.ResourceSelector):
            name = 'newsd_config'
            description = 'Config'
            resource_type = [
                'SLAVE_NEWSD_CONFIG',
            ]
            group = group_name
            required = True

        class StateDump(sp.ResourceSelector):
            name = 'newsd_state'
            description = 'Newsd state dump'
            resource_type = [
                'SLAVE_NEWSD_STATE',
            ]
            group = group_name
            required = True

        class Geobase(sp.ResourceSelector):
            name = 'geobase'
            description = 'geodata4.bin'
            resource_type = [
                'GEODATA_TREE_LING_STABLE',
            ]
            group = group_name
            required = True

        class IndexConfig(sp.ResourceSelector):
            name = 'index_config'
            description = 'index config path'
            resource_type = [
                'NEWS_DEFAULT_INDEX_CONFIG',
            ]
            group = group_name
            required = True

        class NowTime(sp.SandboxIntegerParameter):
            name = 'now_time'
            description = "Now timestamp: use it if you want to redefine Now time in all requests"
            required = False

        class PersonalRubric(sp.ResourceSelector):
            name = 'personal_rubric'
            description = 'Personal rubric shard'
            resource_type = [
                'NEWS_PERSONAL_RUBRIC_TABLE_SHARDS',
            ]
            group = group_name
            required = False

        class FavoriteHosts(sp.ResourceSelector):
            name = 'favorite_hosts'
            description = 'Favorite hosts shard'
            resource_type = [
                'NEWS_FAVORITE_HOSTS_TABLE_SHARDS',
            ]
            group = group_name
            required = False

        class TZdata(sp.ResourceSelector):
            name = 'tzdata'
            description = 'geobase-tzdata'
            resource_type = [
                'GEODATATZDATA_STABLE',
            ]
            group = group_name
            required = False

        params = (
            Binary, Config, StateDump, Geobase, IndexConfig,
            NowTime, PersonalRubric, FavoriteHosts, TZdata
        )

    return Params


class SlaveNewsd(sc.SearchComponent):
    name = 'slave_newsd'
    http_collection = 'yandsearch'

    def __init__(self, binary, cfg, port, geobase, index_config_path,
                 state=None, workdir=None, app_host_mode=False,
                 now=None, personal_rubric=None, favorite_hosts=None,
                 app_host_queue_size=None, tzdata=None):
        super(SlaveNewsd, self).__init__(self)

        self.workdir = workdir
        self.binary = binary
        self.cfg = cfg
        self.state = state
        self.geobase = geobase
        self.index_config_path = index_config_path
        self.now = now
        self.personal_rubric = personal_rubric
        self.favorite_hosts = favorite_hosts
        self.app_host_queue_size = app_host_queue_size
        self.tzdata = tzdata

        self.__http_port = port
        self.__state_port = port + 1
        self.__apphost_port = port + 2

        data_config = {}

        if app_host_mode:
            self.port = self.__apphost_port
        else:
            self.port = self.__http_port

        if self.personal_rubric is None:
            self.personal_rubric = tempfile.mkdtemp()
            data_config["personal_rubric_table"] = {}
            data_config["personal_rubric_table"]["path"] = self.personal_rubric
            data_config["personal_rubric_table"]["reload_interval"] = "360m"

        if self.favorite_hosts is None:
            self.favorite_hosts = tempfile.mkdtemp()
            with open(os.path.join(self.favorite_hosts, "content"), "w"):
                pass
            data_config["favorite_hosts_table"] = {}
            data_config["favorite_hosts_table"]["path"] = self.favorite_hosts
            data_config["favorite_hosts_table"]["reload_interval"] = "360m"

        data_config["default_index_config"] = self.index_config_path
        data_config["geodata4.bin"] = {}
        data_config["geodata4.bin"]["path"] = self.geobase
        data_config["geodata4.bin"]["reload_interval"] = "360m"

        data_config["bans.json"] = tempfile.mkstemp()[1]
        with open(data_config["bans.json"], "w") as f:
            f.write("{}")

        self.data_config = tempfile.mkstemp()[1]
        with open(self.data_config, "w") as f:
            f.write(json.dumps(data_config))

    def start(self):
        if self.state is not None:
            self.dumpdir = os.path.join(self.workdir, 'dump')
            paths.make_folder(self.dumpdir, True)
            dumpfile = os.path.join(self.dumpdir, str(int(time.time() * 1e6)))
            logging.debug("Creating symlink %s => %s" % (self.state, dumpfile))
            paths.copy_path(self.state, dumpfile)

        slave_log = paths.get_unique_file_name(paths.get_logs_folder(), "slave.log")

        environment = {}
        if self.tzdata is not None:
            try:
                tzdata_dir = os.path.join(self.workdir, 'tzdata')
                paths.make_folder(tzdata_dir, True)
                tzdata_dir = os.path.abspath(tzdata_dir)
                tar = tarfile.open(self.tzdata)
                tar.extractall(tzdata_dir)
                environment['GEOBASE_TZ_PATH'] = tzdata_dir + "/zones_bin"
                logging.info("Set environment: GEOBASE_TZ_PATH = {}".format(tzdata_dir + "/zones_bin"))
            except Exception as e:
                logging.error("Something went wrong: " + str(e))

        cmd = [
            self.binary,
            '--http-port', str(self.__http_port),
            '--state-port', str(self.__state_port),
            '--apphost-port', str(self.__apphost_port),
            '-c', self.cfg,
            '-L', slave_log,
            '--data-config-path', self.data_config,
        ]

        if self.state is not None:
            cmd.append('-s')
            cmd.append(self.dumpdir)

        if self.now is not None:
            cmd.append('--now-time')
            cmd.append(str(self.now))

        if self.app_host_queue_size is not None:
            cmd.append('--apphost-queue-length')
            cmd.append(str(self.app_host_queue_size))

        self.process_parameters = cmd

        self.process = process.run_process(cmd, log_prefix='slave_newsd', wait=False, environment=environment)

    def get_http_port(self):
        return self.__http_port

    def get_state_port(self):
        return self.__state_port

    def get_apphost_port(self):
        return self.__apphost_port

    def wait(self, timeout=180):
        logging.info("Waiting for newsd to become ready for {} seconds".format(timeout))
        start = time.time()
        finish = start + timeout
        while time.time() < finish:
            if not self.is_running():
                logging.info("%s died", self.name)
                self._wait_coredump()
                self._process_post_mortem()
            try:
                with closing(urlopen('http://localhost:%d/state' % self.__http_port)) as f:
                    reply = f.read()
                    if reply.find(':') >= 0:
                        logging.info('Newsd is ready')
                        return
            except:
                pass
            time.sleep(1)
        if self.is_running():
            logging.info("Kill process %s", self.process.pid)
            os.kill(self.process.pid, 6)  # generate coredump
            self._wait_coredump()
        eh.check_failed("Cannot connect to {}, port {} in {} seconds".format(
            self.name, self.__http_port, timeout
        ))

    def use_component(self, func):
        return func()

    def warmup_request(self):
        pass
