# -*- coding: utf-8 -*-

import os
import logging
import tempfile
import time
from urllib2 import urlopen, URLError

from sandbox.projects.common.search import components as sc
from sandbox.projects.common import error_handlers as eh

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.paths import make_folder


def create_routerd_params():
    group_name = "Routerd params"

    class Params(object):
        class Binary(sp.ResourceSelector):
            name = "routerd_executable"
            description = "Executable"
            resource_type = [
                'NEWS_APPHOST_ROUTERD_EXECUTABLE'
            ]
            group = group_name
            requred = True

        class DeviceData(sp.ResourceSelector):
            name = 'device_data'
            description = 'Device data'
            resource_type = [
                'NEWS_UATRAITS_BROWSER_XML',
            ]
            group = group_name
            required = True

        class NewsData(sp.ResourceSelector):
            name = 'news_data'
            description = 'News data'
            resource_type = [
                'NEWS_REPORT_DATA_PROD',
            ]
            group = group_name
            required = True

        class NewsDataExp(sp.ResourceSelector):
            name = 'news_data_exp'
            description = 'News data exp'
            resource_type = [
                'NEWS_REPORT_DATA_EXP',
            ]
            group = group_name
            required = True

        class Geobase(sp.ResourceSelector):
            name = 'geobase'
            description = 'geodata4.bin'
            resource_type = [
                'GEODATA_TREE_LING_STABLE',
            ]
            group = group_name
            required = True

        class DynamicRobotsTxtConfig(sp.ResourceSelector):
            name = "dynamic_robots_txt"
            description = "Dynamic robots txt config"
            resource_type = ['NEWS_ROUTERD_DYNAMIC_ROBOTS_TXT_CONFIG']
            group = group_name
            requred = True

        class AllowedOrigins(sp.ResourceSelector):
            name = "allowed_origins"
            description = "Allowed origins"
            resource_type = ['ROUTERD_ALLOWED_ORIGINS']
            group = group_name
            requred = True

        class NowTime(sp.SandboxIntegerParameter):
            name = 'now_time'
            description = "Now timestamp: use it if you want to redefine Now time in all requests"
            required = False

        params = (Binary, DeviceData, NewsData, NewsDataExp, Geobase, DynamicRobotsTxtConfig, AllowedOrigins, NowTime)

    return Params


class Routerd(sc.SearchComponent):
    name = 'routerd'
    http_collection = ''

    def __init__(self, binary, workdir, port, geobase, newsdata, newsdata_exp, device_data, allowed_origins, dynamic_robots_txt, now=None):
        super(Routerd, self).__init__(self)

        self.binary = binary
        self.workdir = workdir
        self.port = port
        self.geobase = geobase
        self.newsdata = newsdata
        self.newsdata_exp = newsdata_exp
        self.device_data = device_data
        self.dynamic_robots_txt_config = dynamic_robots_txt
        self.allowed_origins = allowed_origins
        self.now = now

    def start(self):
        flags_dir = os.path.join(self.workdir, "flags")
        make_folder(flags_dir)
        for flags_file_name in ('flags.json', 'sport_flags.json', 'she_flags.json', 'news_test_domain_flags.json', 'staff_login_flags.json'):
            with open(os.path.join(flags_dir, flags_file_name), 'w') as fd:
                fd.write("{}")

        if not os.path.isfile(self.newsdata):
            self.newsdata = os.path.join(self.newsdata, 'newsdata2.json')
        if not os.path.isfile(self.newsdata_exp):
            self.newsdata_exp = os.path.join(self.newsdata_exp, 'newsdata2.json')

        agencies_data_file = os.path.join(self.workdir, "agencies_data")
        with open(agencies_data_file, 'w') as fd:
            fd.write("{}")

        persistent_flags_file = os.path.join(self.workdir, "persistent_flags")
        with open(persistent_flags_file, 'w') as fd:
            fd.write("{}")

        district_data_file = os.path.join(self.workdir, "district_data")
        with open(district_data_file, 'w') as fd:
            fd.write("{}")

        cmd = [
            self.binary,
            '--port', str(self.port),
            '--log_path', os.path.join(self.workdir, "log"),
            '--geobase_path', self.geobase,
            '--production_newsdata_path', self.newsdata,
            '--experimental_newsdata_path', self.newsdata_exp,
            '--device_data_path', self.device_data,
            '--its_flags_path', flags_dir,
            '--allowed_origins_path', self.allowed_origins,
            '--dynamic-robots-txt-path', self.dynamic_robots_txt_config,
            '--templates_config_path', tempfile.mkstemp()[1],
            '--agencies_json_path', agencies_data_file,
            '--permanent-flags-file-path', persistent_flags_file,
            '--district_data_path', district_data_file
        ]

        if self.now is not None:
            cmd.append('--now-time')
            cmd.append(str(self.now))

        self.process_parameters = cmd

        environment = {
            'Y_NEWS_API_KEYS': "234",
        }

        self.process = process.run_process(cmd, log_prefix='routerd_', wait=False, environment=environment)

    def wait(self, timeout=30):
        logging.info("Waiting for routerd to become ready for {} seconds".format(timeout))
        start = time.time()
        finish = start + timeout
        while time.time() < finish:
            if not self.is_running():
                logging.error("%s died", self.name)
                self._wait_coredump()
                self._process_post_mortem()
            try:
                urlopen('http://localhost:%d/admin?action=ping' % self.port)
                logging.info('routerd is ready')
                return
            except URLError:
                # request to routerd failed
                pass
            except:
                # ???
                pass
            time.sleep(1)
        if self.is_running():
            logging.info("Kill process %s", self.process.pid)
            os.kill(self.process.pid, 6)  # generate coredump
            self._wait_coredump()
        eh.check_failed("Cannot connect to {}, port {} in {} seconds".format(
            self.name, self.port, timeout
        ))

    def use_component(self, func):
        return func()

    def warmup_request(self):
        pass
