# coding: utf-8
import logging
import time
import requests

from sandbox import sdk2
from sandbox.projects.yappy.api import YappyApi


MODEL_URL = 'http://yappy.z.yandex-team.ru/api/yappy.services.Model'


class BetaConsistanceTimeoutError(RuntimeError):
    pass


def do_with_retries(retries, wait_time):
    def decorator(func):
        def wrapper(*args, **kwargs):
            i = 0
            while True:
                try:
                    value = func(*args, **kwargs)
                    print("Got value in decorator {}".format(value))
                    return value
                except Exception as e:
                    logging.error("Retry number: {}.\nFailed with error: {}".format(i, e))
                    i += 1
                    if i > retries:
                        raise e
                    time.sleep(wait_time)
        return wrapper
    return decorator


@do_with_retries(5, 5)
def deallocate_beta(api, payload):
    return api.deallocate_beta(payload=payload)


@do_with_retries(5, 5)
def delete_beta(api, payload):
    return api.delete_beta(payload=payload)


@do_with_retries(5, 5)
def beta_exists(api, payload):
    return api.beta_exists(payload=payload)


@do_with_retries(5, 5)
def create_beta_from_template_request(api, payload):
    return api.create_beta_from_template(payload=payload)


@do_with_retries(5, 5)
def get_beta_status(api, payload):
    return api.get_beta_status(payload=payload)


@do_with_retries(5, 5)
def post_request(url, payload, headers, timeout):
    response = requests.post(url, json=payload, headers=headers, timeout=timeout)
    if not response.ok:
        raise ValueError('request {} failed:\n{}\n{}\n{}'.format(url, response.content, response.headers, response.status_code))
    return response


def stop_and_delete_beta(service_config, beta_suffix, yappy_token):
    api = YappyApi(oauth_token=yappy_token)
    beta_name = '{}-{}'.format(service_config['beta_template'], beta_suffix)
    if not beta_exists(api, {'name': beta_name}).get('exists'):
        raise ValueError('beta {} does not exist'.format(beta_name))

    payload = {'name': beta_name}
    deallocate_beta(api, payload)
    return delete_beta(api, payload)


def get_beta_backends(service_config, beta_suffix, yappy_token):
    headers = {'Content-type': 'application/json'}
    if yappy_token:
        headers['Authorization'] = 'OAuth ' + yappy_token

    api = YappyApi(oauth_token=yappy_token)
    beta_name = '{}-{}'.format(service_config['beta_template'], beta_suffix)
    if not beta_exists(api, {'name': beta_name}).get('exists'):
        raise ValueError('beta {} does not exist'.format(beta_name))

    payload = {'betas_regexps': [beta_name]}
    url = '{}/retrieveBetasSlots'.format(MODEL_URL)
    response = post_request(url, payload, headers, 10)
    return response.json()["betaSlots"][0]["typeToSlots"][service_config["component_id"]]["hosts"]


def create_beta_from_template(service_config, beta_suffix, resources, yappy_token, timeout):
    api = YappyApi(oauth_token=yappy_token)
    beta_name = '{}-{}'.format(service_config['beta_template'], beta_suffix)
    if beta_exists(api, {'name': beta_name}).get('exists'):
        raise ValueError('beta {} is already exist'.format(beta_name))
    patches = []
    for resource_id in resources:
        resource = sdk2.Resource.find(
            id=resource_id
        ).limit(1).first()
        resource_type = resource.type
        if resource_type in service_config['patched_resources']:
            resource_info = service_config['patched_resources'][resource_type]
            patches.append({'res_id': resource_id, 'local_path': resource_info['local_path']})

    payload = {
        'template_name': service_config['beta_template'],
        'suffix': beta_suffix,
        'patches': [{
            'componentId': service_config['component_id'],
            'patch': {
                'parentExternalId': service_config['base_service'],
                'resources': []
            }
        }]
    }
    for patch in patches:
        payload['patches'][0]['patch']['resources'].append({
            'manageType': 'SANDBOX_RESOURCE',
            'localPath': patch['local_path'],
            'sandboxResourceId': str(patch['res_id'])
        })

    create_beta_from_template_request(api, payload)

    wait_start_time = time.time()
    while time.time() - wait_start_time < timeout:
        time.sleep(120)
        try:
            beta_status = get_beta_status(api, {'name': beta_name})
            logging.debug('get_beta_status response: %s', str(beta_status))
            if beta_status.get('state', {}).get('status') == 'CONSISTENT':
                return
        except ValueError:  # Ignore 5xx from yappy response
            pass

    raise BetaConsistanceTimeoutError('Beta consistance timeout')
