# -*- coding: utf-8 -*-

import logging
import re
import urllib2
import urlparse

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.betas.beta_api import BetaApi
from sandbox.projects.common.search import instance_resolver as ir
from sandbox.projects.websearch.upper import resources as upper_resources

from sandbox.sandboxsdk.channel import channel

BETA_SERVICES = {
    "noapache.priemka.yandex.ru": "priemka__noapache__web_noapache",
    "apphost.priemka.yandex.ru": "priemka__apphost__web_noapache",
    "rearr.priemka.yandex.ru": "priemka__rearr__web_noapache",
    "stable.priemka.yandex.ru": "priemka__stable__web_noapache",
    "noapache.priemka.yandex.ru/images": "priemka__noapache__imgs_noapache",
    "stable.priemka.yandex.ru/images": "priemka__stable__imgs_noapache",
    "hamster.yandex.ru": "hamster_noapache_vla_web_yp",
    "hamster.yandex.ru/images": "hamster_noapache_vla_imgs_yp",
}


# FIXME: remove this trash
@decorators.retries(max_tries=3, delay=30)
def read_url(url, ssl_context=None):
    logging.debug('read url: ' + url)
    if ssl_context:
        conn = urllib2.urlopen(url, context=ssl_context)
    else:
        conn = urllib2.urlopen(url)
    try:
        return conn.read()
    finally:
        conn.close()


def name_from_url(beta_url):
    """
        convert beta_url to canonical beta name (which can be used as argument for noapache_list())
    """
    purl = urlparse.urlparse(beta_url)
    netloc = purl.netloc
    tr_suffix = '.com.tr'
    if netloc.endswith(tr_suffix):
        netloc = netloc[:-len(tr_suffix)] + '.ru'
    return (netloc + '/images') if purl.path.startswith('/images') else netloc


def noapache_list(beta_name, nanny_token=None):
    """
        :return endpoints list (list of "host:port" strings)
        for all noapache instances under this beta
    """
    beta_suffix = '.hamster.yandex.'
    if beta_name in BETA_SERVICES:
        instances = ir.get_joined_instances_by_config(BETA_SERVICES[beta_name], nanny_oauth_token=nanny_token)
    elif beta_suffix in beta_name:
        short_beta_name = beta_name.split(beta_suffix)[0]
        instances = BetaApi.fromurl().get_instances(short_beta_name, 'noapache')
    else:
        raise Exception('Input parameter contain unsupported beta: "{}"'.format(beta_name))
    return instances


def get_noapache_version(instances):
    """
        Take noapache major_version from first host in ok state
    """
    verinfo = ''
    for ep in instances:
        try:
            logging.debug('get noapache instance on [{}] info'.format(ep))
            conn = urllib2.urlopen('http://{}/yandsearch?info=getversion'.format(ep))
            verinfo = conn.read()
            conn.close()
            break
        except Exception as e:
            logging.warn(str(e))
            continue
    for line in verinfo.split('\n'):
        line = line.lstrip(' ')
        if line.startswith('URL: '):
            verinfo = line[5:]
            break
    if not verinfo:
        eh.check_failed('can not determine current _priemka_ noapache version')
    # SAMPLE: verinfo = 'svn+ssh://zomb-sandbox-ro@arcadia.yandex.ru/arc/tags/upple/stable-36/r22/arcadia'
    # now its more like 'svn://arcadia.yandex.ru/arc/tags/upple/stable-219-24/arcadia' and next 3 lines prblby outdated
    # p = re.compile('(^.*)arc/tags/upple/stable-(.*)/r[0-9]*/arcadia(.*$)')
    # major = p.sub(r'\2', verinfo)
    # if major == verinfo:
    p = re.compile('(^.*)arc/tags/upple/stable-(.*)-(.*)/arcadia(.*$)')
    major = p.sub(r'\2', verinfo)
    minor = p.sub(r'\3', verinfo)
    if major == verinfo:
        eh.check_failed('can not extract major version from ' + verinfo)
    ver_major = int(major)
    ver_minor = int(minor)
    logging.info('detect noapache-%s-%s', ver_major, ver_minor)
    return ver_major, ver_minor


class NoapacheConfigInfo:
    def __init__(self, endpoint=None, beta=None):
        self.beta = beta
        self.endpoint = endpoint
        self.exe_ver = None
        self.exe_taskid = None
        self.cfg_ver = None
        self.rearrange_ver = None
        self.rearrange_taskid = None
        self.rearrange_dynamic_ver = None
        self.rearrange_dynamic_taskid = None
        self.rearrange_data_fast_ver = None
        self.rearrange_data_fast_taskid = None

    def ask_instance(
        self,
        getconfig=False,
        getversion=False,
        getrearrangeversion=False,
        sandboxtaskid=False,
        nothrow=False,
    ):
        if getconfig:
            self._getconfig(nothrow=nothrow)
        if getversion:
            self._getversion(nothrow=nothrow)
        if getrearrangeversion:
            self._getrearrangeversion(nothrow=nothrow)
        if sandboxtaskid:
            self._sandboxtaskid(nothrow=nothrow)
        return self

    def _info_request(self, req, nothrow=False):
        rv = ''
        try:
            rv = read_url('http://{}/yandsearch?info={}'.format(self.endpoint, req))
        except Exception as err:
            if nothrow:
                logging.error("fail info={} for noapache: {}".format(req, err))
            else:
                raise
        return rv

    def _getconfig(self, nothrow=False):
        for line in self._info_request('getconfig').split('\n'):
            line = line.strip()
            if line.startswith("ConfigVersion "):
                tk = line.split(' ')
                if len(tk) >= 2:
                    self.cfg_ver = ' '.join(tk[1:])
        return self

    def _getversion(self, nothrow=False):
        for line in self._info_request('getversion').split('\n'):
            line = line.strip()
            if line.startswith("URL:"):
                self.exe_ver = re.sub('/arcadia.*', '', re.sub('.*/stable-', '', line))
        return self

    def _getrearrangeversion(self, nothrow=False):
        r = False
        rd = False
        rfast = False
        for line in self._info_request('getrearrangeversion').split('\n'):
            line = line.strip()
            if line.startswith("RearrangeData:"):
                r = True
            elif line.startswith("RearrangeDynamicData:"):
                rd = True
            elif line.startswith("RearrangeDataFast:"):
                rfast = True
            task_prefix = tuple(('Sandbox task: ', 'sandbox_task: '))
            task_prefix_len = len(task_prefix[0])
            assert len(set(map(len, task_prefix))) == 1
            if line.startswith("URL:"):
                if r:
                    self.rearrange_ver = re.sub('/arcadia.*', '', re.sub('.*/stable-', '', line))
                elif rd:
                    self.rearrange_dynamic_ver = re.sub('/arcadia.*', '', re.sub('.*/rearrange.dynamic/', '', line))
            elif line.startswith('fast_data_version'):
                if rfast:
                    self.rearrange_data_fast_ver = line[len("fast_data_version: "):]
            elif line.startswith(task_prefix):
                if r:
                    self.rearrange_taskid = int(line[task_prefix_len:])
                    r = False
                elif rd:
                    self.rearrange_dynamic_taskid = int(line[task_prefix_len:])
                    rd = False
                elif rfast:
                    self.rearrange_data_fast_taskid = int(line[task_prefix_len:])
                    rfast = False
        return self

    def _sandboxtaskid(self, nothrow=False):
        self.exe_taskid = int(self._info_request('sandboxtaskid'))
        return self

    def to_string(self):
        return "{} {} exe:{} cfg:{} rearrange:{} r.d:{} rearrange.fast:{}".format(
            self.beta, self.endpoint, self.exe_ver, self.cfg_ver, self.rearrange_ver,
            self.rearrange_dynamic_ver, self.rearrange_data_fast_ver
        )


def get_resources_id(instances, use_sdk2=False):
    """
        Return dict(resource_type=resource_id) described rearrange(.dynamic) resources
        from first responded noapache instance
    """
    config_info = None
    for instance in instances:
        try:
            config_info = NoapacheConfigInfo(instance).ask_instance(
                sandboxtaskid=True,
                getrearrangeversion=True,
            )
        except Exception as err:
            eh.log_exception("Instance %s query failed, skip it", err)
            continue
        break
    eh.ensure(
        config_info,
        "Nobody among %s instances responded properly, please check your service configuration "
        "(group names, service availability, etc). ",
    )
    result = {}

    def set_resource_id(task_id, rt):
        if not task_id or not rt:
            logging.error('skip get resource {} from sandbox taskid:{}'.format(rt, task_id))
            return
        logging.debug('get resource {} from sandbox taskid:{}'.format(rt, task_id))
        if use_sdk2:
            lr = [sdk2.Resource.find(rt, task=sdk2.Task[task_id]).first()]
        else:
            lr = channel.sandbox.list_resources(resource_type=rt, task_id=task_id)
        eh.ensure(len(lr), 'finding resource in sandbox failed (has assigned channel.task = self ?)')
        sid = lr[0].id
        result[str(rt)] = sid
        logging.info('use {}({}) from sandbox taskid:{}'.format(rt, sid, task_id))

    if use_sdk2:
        set_resource_id(config_info.exe_taskid, upper_resources.NoapacheUpper)
        set_resource_id(config_info.exe_taskid, resource_types.EVLOGDUMP_EXECUTABLE)
        set_resource_id(config_info.rearrange_taskid, resource_types.REARRANGE_DATA)
        set_resource_id(config_info.rearrange_dynamic_taskid, resource_types.REARRANGE_DYNAMIC_DATA)
        set_resource_id(config_info.rearrange_data_fast_taskid, upper_resources.RearrangeDataFast)
    else:
        set_resource_id(config_info.exe_taskid, 'NOAPACHE_UPPER')
        set_resource_id(config_info.exe_taskid, 'EVLOGDUMP_EXECUTABLE')
        set_resource_id(config_info.rearrange_taskid, 'REARRANGE_DATA')
        set_resource_id(config_info.rearrange_dynamic_taskid, 'REARRANGE_DYNAMIC_DATA')
        set_resource_id(config_info.rearrange_data_fast_taskid, 'REARRANGE_DATA_FAST')
    return result
