import re

from sandbox.projects.common.search import response_patcher as rp


UNSTABLE_SEARCH_PROPS = map(lambda prop_name: re.sub(r'\W', '_', prop_name), [
    "Acc",
    "BsTouched",
    "CalcMetaRelevanceMcs",  # SEARCH-9579
    "HostName",
    "Ig",
    "Nctx",
    "SourceTimestamp",
    "WaitInfo2.debug",

    "SerpFact.CandidateFacts",
    "SerpFact.Fact.debug",
    "SerpFact.FactJson.debug",
    "SerpFact.FactShown",
    "SerpFact.Timedout",
    "SerpObject.ErrorMessage.debug",

    "ApplyAdsBlender.NoGroupingBlenderTime",
    "ApplyAdsBlender.PureGroupingTime",
    "ApplyBlender.NoGroupingBlenderTime",
    "ApplyBlender.PureGroupingTime",
    "ApplyImagesBlender.NoGroupingBlenderTime",
    "ApplyImagesBlender.PureGroupingTime",
    "ApplyImagesBlender.dynamic_factors",
    "ApplyImagesBlender.factors",
    "ApplyVideoBlender.NoGroupingBlenderTime",
    "ApplyVideoBlender.PureGroupingTime",
    "ApplyVideoBlender.dynamic_factors",
    "ApplyVideoBlender.factors",

    "scheme.json.nodump",

    "DocumentsStats",  # SEARCH-11681
])

UNSTABLE_SEARCH_PROPS_REGEXES = [
    re.compile(r'_TimeDistr$'),

    # Source configuration data (https://wiki.yandex-team.ru/users/and42/configinfo)
    re.compile(r'^ConfigInfo_'),

    # Debug info about cache keys (SEARCH-4139)
    re.compile(r'^ClientGroup_'),
]

UNSTABLE_MARKERS = set([
    'RobotDaterFreshAge',
])

UNSTABLE_SAVED_COPY_URL_DATA_ARGS = [
    'tm',
]


def match_any_regex(value, regexes):
    return any(regex.search(value) for regex in regexes)


def sort_search_prop_values(props, key, sep1, sep2=None):
    value = props.GetPropValue(key)
    if not value:
        return
    sorted_value = rp.sort_val(value, sep1, sep2)
    if value != sorted_value:
        raw_value = props.GetPropValue(key, parse=False)
        if raw_value[0] == '"' and raw_value[-1] == '"':
            sorted_value = '"{}"'.format(sorted_value)
        props.SetPropValue(sorted_value)


def unanswers_state(btd):
    # TODO
    return None


def same_unanswers(btd1, btd2):
    return unanswers_state(btd1) == unanswers_state(btd2)


def remove_unstable_search_props(search_props):
    for source, props in search_props.nodes().items():
        for props_elem in props:
            props_elem.nodes().pop('property_count', None)
            properties = props_elem.get_node('properties', required=False)
            if properties is None:
                continue
            for key in properties.props().keys():
                if key in UNSTABLE_SEARCH_PROPS or match_any_regex(key, UNSTABLE_SEARCH_PROPS_REGEXES):
                    properties.props().pop(key)


def make_upper_props_stable(search_props):
    if 'UPPER' not in search_props.nodes():
        return

    for upper_props in search_props.nodes()['UPPER']:
        properties = upper_props.get_node('properties', required=False)
        if properties is None:
            continue
        for key in properties.props().keys():
            if key in ['ApplyBlender.#fmls', 'ApplyBlender.fmls']:
                sort_search_prop_values(properties, key, ' ', '|')
            elif key == 'ApplyBlender.slices':
                sort_search_prop_values(properties, key, ' ')
            elif key.startswith('ApplyBlender.') and key.endswith('.debug'):
                if key.endswith('.#inserted.debug'):
                    sort_search_prop_values(properties, key, '; ')
                else:
                    sort_search_prop_values(properties, key, '. ')
            elif key == "Vertical.EmptyIntent":
                sort_search_prop_values(properties, key, '|')
            elif key == "Personalization.ag":
                sort_search_prop_values(properties, key, '|')
            elif key == "EntitySearch.ListOntoids":
                sort_search_prop_values(properties, key, '|')


def sort_search_props(search_props):
    pass


def try_get_by_path(node, path):
    ret = node
    for id in path:
        ret = ret.get_node(id, required=False)
        if ret is None:
            break
    return ret


def make_search_props_stable(btd):
    search_props = try_get_by_path(btd, ['tmpl_data', 'search', 'search_props'])

    if not search_props:
        search_props = try_get_by_path(btd, ['data', 'search_props'])

    if not search_props:
        return

    remove_unstable_search_props(search_props)
    make_upper_props_stable(search_props)
    sort_search_props(search_props)


def remove_unstable_doc_markers(docs):
    if not docs:
        return

    for doc in docs:
        markers = doc.get_node('markers', required=False)
        if markers:
            for key in UNSTABLE_MARKERS:
                markers.nodes().pop(key, None)

        _markers = doc.nodes().get('_markers')
        if _markers:
            for marker in _markers[:]:
                key, value = marker.split('=')
                for key in UNSTABLE_MARKERS:
                    _markers.remove(marker)

        saved_copy_url = doc.get_node('saved_copy_url', required=False)
        if saved_copy_url:
            data = saved_copy_url.get_node('data')
            args = data.get_node('args')
            for key in UNSTABLE_SAVED_COPY_URL_DATA_ARGS:
                args.nodes().pop(key, None)


def make_docs_stable(search_data):
    remove_unstable_doc_markers(search_data.nodes().get('docs'))
    remove_unstable_doc_markers(search_data.nodes().get('docs_right'))


def make_searchdata_stable(btd):
    search_data = try_get_by_path(btd, ['tmpl_data', 'search', 'searchdata'])
    if not search_data:
        search_data = try_get_by_path(btd, ['data', 'searchdata'])
    make_docs_stable(search_data)


def prepare_blender_tmpl_data_for_comparison_impl(node, name):
    btd_num = 0
    while name in node.nodes():
        btd = node.get_node(name)

        make_search_props_stable(btd)
        make_searchdata_stable(btd)

        btd_num += 1
        name = 'answers_blender_tmpl_data#{}'.format(btd_num)


def prepare_blender_tmpl_data_for_comparison(node):
    prepare_blender_tmpl_data_for_comparison_impl(node, 'answers_blender_tmpl_data')
    prepare_blender_tmpl_data_for_comparison_impl(node, 'answers_template_data')


def can_compare_blender_tmpl_data(node1, node2):
    nodes = [node1.nodes(), node2.nodes()]
    btds = [[], []]

    for idx, node in enumerate(nodes):
        name = 'answers_blender_tmpl_data'
        while name in nodes[idx]:
            btds[idx].append(nodes[idx][name])
            name = 'answers_blender_tmpl_data#{}'.format(len(btds[idx]))

    # number of blender_tmpl_data is unequal
    if len(btds[0]) != len(btds[1]):
        return False

    for btd1, btd2 in zip(btds[0], btds[1]):
        if not same_unanswers(btd1, btd2):
            return False
    return True
