#!/usr/bin/env python
# encoding: utf8

import os
import logging
import tarfile
import distutils

from sandbox.projects.common import utils

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters


class Params:
    """class BuildTgzBundle(parameters.SandboxBoolParameter):
        name = 'build_tgz_bundle'  # WARNING: YaMakeTask has 'build_bundle' param
        description = 'Pack data to tar.gz bundle'
        default_value = True"""

    class UseArchiver(parameters.SandboxBoolParameter):
        name = 'use_archiver'
        description = 'Use tools/archiver to gather blender models in one file'
        default_value = False

    class ExtensionsToArchive(parameters.SandboxStringParameter):
        name = 'extensions_to_archive'
        description = "Pack following extensions to archive (comma-separated), see BLNDR-1699"
        default_value = 'info,xtd'


def use_archiver(task):
    return utils.get_or_default(task.ctx, Params.UseArchiver)


def calculate_targets(task, static=True, dynamic=True, fast=True):
    targets = []

    if static:
        targets.append('search/web/rearrs_upper/rearrange')

    if dynamic:
        targets.append('search/web/rearrs_upper/rearrange.dynamic')

    if fast:
        targets.append('search/web/rearrs_upper/rearrange.fast')

    if use_archiver(task):
        targets.append('tools/archiver')

    task.ctx['targets'] = ';'.join(targets)


def export_ya_make_result(src_dir, dst_dir):
    logging.info("export_ya_make_result %s %s", src_dir, dst_dir)
    distutils.dir_util.copy_tree(src_dir, dst_dir)  # use replace copytree version


def get_directory_size(path, mb=True):
    du_proc = process.run_process(['du', '-bs', path], outs_to_pipe=True)
    du_output = du_proc.communicate()[0].strip()
    size = int(du_output.split('\t')[0])
    return (size / 1048576.0) if mb else size


def tar(bundle_name, path, name):
    with tarfile.open(bundle_name, 'w:gz') as tar:
        tar.add(path, name)


def archive(task, rearrange_dynamic_dir, output_dir):
    if not use_archiver(task):
        return

    archiver = os.path.join(output_dir, 'tools/archiver/archiver')
    target_dir = os.path.join(rearrange_dynamic_dir, 'blender')
    extensions = utils.get_or_default(task.ctx, Params.ExtensionsToArchive).split(",")
    extensions = ["." + extension for extension in extensions]
    logging.info("Extensions to archive: %s", extensions)
    to_archive = [
        os.path.join(target_dir, fn)
        for fn in os.listdir(target_dir) if os.path.splitext(fn)[1] in extensions
    ]
    process.run_process(
        [
            "/bin/ls", "-la",
        ] + to_archive,
        log_prefix="archiver_SEARCH_5859_bug_control",
    )
    process.run_process(
        [
            archiver, '--deduplicate', '--plain', '--output', os.path.join(target_dir, 'models.archive')
        ] + to_archive,
        log_prefix="archiver",
    )

    for f in to_archive:
        os.remove(f)
