# -*- coding: utf-8 -*-

import base64
import json
import subprocess
import logging
import collections

from sandbox.projects.common.base_search_quality.tree import htmldiff
from sandbox.projects.common.noapacheupper.diff import diff_maker


class FractionApphostResponsesDataComparator(diff_maker.FractionDataComparator):
    def __init__(self, converter, proto2hr, data_patchers, can_compare, app_host_ops=None):
        super(FractionApphostResponsesDataComparator, self).__init__(data_patchers, can_compare)
        self.converter = converter
        self.proto2hr = proto2hr
        self.app_host_ops = app_host_ops

    def data_to_node(self, data):
        """
            конвертируем raw app_host данные в формат для передачи компаратору (Node)
        """
        proc = subprocess.Popen([self.converter, '--input-format', 'service_response'],
                                 stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = proc.communicate(data)
        if proc.returncode:
            raise Exception("Fail converter: {}]\nconverted_data:{}\n".format(err, base64.b64encode(data)))
        j = json.loads(out.decode('latin1'))
        ans = j
        if "answers" in j:
            ans = {}
            numbers = collections.defaultdict(int)
            for i, answer in enumerate(j["answers"]):
                answer_type = answer.get("type")
                if answer_type in ['eventlog', 'internal_source_profile', 'debug_info']:
                    continue
                if not answer_type:
                    name = i
                else:
                    name = answer_type
                    if numbers[answer_type]:
                        name += "#{}".format(numbers[answer_type])
                    numbers[answer_type] += 1
                if 'binary' in answer:
                    ans["answers_{}".format(name)] = answer['binary']
                else:
                    ans["answers_{}".format(name)] = answer

        if 'balancing_hints' in j:
            ans['balancing_hints'] = j['balancing_hints']

        if not isinstance(ans, dict):
            logging.debug("'ans' is not a dictionary. Converter output was: %s", out)

        if 'type' in ans:
            if ans['type'] == 'eventlog':
                logging.debug('Eventlog in type, ignoring (SEARCH-5251)')
                ans = {}

        node = htmldiff._convert_json_dict_to_tree_impl(ans)
        try:
            self.patch(node)
        except Exception as ex:
            error_msg = "Patch node failed: {}.\nNode was: {}".format(ex, ans)
            logging.debug(error_msg)

        return node

    def on_has_diff(self, num, node1, node2, descr):
        super(FractionApphostResponsesDataComparator, self).on_has_diff(num, node1, node2, descr)
        for r in zip(node1._nodes.get('response', []), node2._nodes.get('response', [])):
            self.sbs_info.add_diff(num, r[0], r[1], descr)

    def on_has_not_diff(self, num):
        super(FractionApphostResponsesDataComparator, self).on_has_not_diff(num)
        self.sbs_info.add_not_diff()


class DiffApphostResponses(diff_maker.DiffMaker):
    """
        Сравниваем ответы app_host-овых поисковых сервисов (noapache)
    """

    def __init__(self, converter, proto2hr, data_patchers, can_compare=None, app_host_ops=None):
        self.converter = converter
        self.proto2hr = proto2hr
        self.data_patchers = data_patchers
        self.can_compare = can_compare
        self.app_host_ops = app_host_ops

    def create_comparator(self):
        return FractionApphostResponsesDataComparator(self.converter, self.proto2hr, self.data_patchers, self.can_compare, self.app_host_ops)
