# -*- coding: utf-8 -*-

import six
import cgi
import cStringIO
import logging
import os
import struct
import traceback
import time
import itertools
from multiprocessing import Pool

from sandbox.projects.common import utils
from sandbox.projects.common.base_search_quality import node_types
from sandbox.projects.common.base_search_quality import response_diff_task
from sandbox.projects.common.base_search_quality.tree import htmldiff
from sandbox.projects.common.search.requester import sequence_binary_data_from_stream
from sandbox.projects.common.search.response.diff.side_by_side import SbsMaker, SbsInfo

DIFF_FILE_PREFIX = 'diff_'
VALUE_OF_KEY = 'Value of key '


class Comparator(object):
    """
        Базовая реализация сравнения 2-х последовательностей данных, конвертируемых в node_tree
        Пишет на диск один html файл с diff-ом (если diff есть)
    """

    def __init__(self, data_patchers, can_compare=None):
        self.changed_props = htmldiff.ChangedProps()
        self.sbs_info = SbsInfo()
        self.data_patchers = data_patchers
        self._can_compare = can_compare
        self.skip_recs = 0
        self.diff_recs = 0
        self.equal_recs = 0

    @staticmethod
    def add_key_names_for_searcher_props(root):
        try:
            searcher_props = root.nodes().get("response")[0].nodes().get("SearcherProp")
            for sp in searcher_props:
                try:
                    key = sp.GetPropValue("Key")
                    value_prop = sp.GetPropValue("Value", required=False)
                    if value_prop is not None:
                        sp.SetPropValue("{}{}".format(VALUE_OF_KEY, key), value_prop)
                        sp.remove_prop("Value")
                    else:
                        value_node = sp.get_node("Value", required=False)
                        if value_node is not None:
                            sp._nodes["{}{}".format(VALUE_OF_KEY, key)] = [value_node]
                            sp._nodes.pop("Value", None)
                except Exception as e:
                    logging.debug(e)
        except Exception as e:
            logging.debug(e)

    @staticmethod
    def remove_key_names_for_searcher_props(root):
        try:
            searcher_props = root.nodes().get("response")[0].nodes().get("SearcherProp")
            for sp in searcher_props:
                try:
                    for p, v in sp._props.iteritems():
                        try:
                            if p.startswith(VALUE_OF_KEY):
                                sp.SetPropValue("Value", v[0])
                                sp.remove_prop(p)
                                break
                        except Exception as e:
                            logging.debug(e)
                    else:
                        for p, v in sp._nodes.iteritems():
                            try:
                                if p.startswith(VALUE_OF_KEY):
                                    sp._nodes["Value"] = v
                                    sp._nodes.pop(p, None)
                                    break
                            except Exception as e:
                                logging.debug(e)

                except Exception:
                    pass
        except Exception as e:
            logging.debug(e)

    def compare(self, num, rec1, rec2, f, descr):
        if descr is None:
            descr = ''
        node1 = self.data_to_node(rec1)
        node2 = self.data_to_node(rec2)
        logging.debug("data_to_node's done %s", num)
        if not self.can_compare(node1, node2):
            return None
        # self.patch(node1)  # patch is called in data_to_node func, we don't need to patch second time SEARCH-4846
        # self.patch(node2)
        logging.debug("patching done %s", num)

        self.add_key_names_for_searcher_props(node1)
        self.add_key_names_for_searcher_props(node2)
        logging.debug("add_key_names_for_searcher_props done %s", num)

        ss = cStringIO.StringIO()
        if htmldiff.write_tree_diff(
            ss,
            num,
            node1,
            node2,
            node_types.make_node_types_dict(),
            self.changed_props,
        ):
            logging.debug("write_tree_diff done %s", num)
            self.remove_key_names_for_searcher_props(node1)
            self.remove_key_names_for_searcher_props(node2)
            logging.debug("remove_key_names_for_searcher_props done %s", num)

            # write to file only real diff-s
            f.write(htmldiff.StartBlock('<a name="rec{}"><b>{}:</b></a> {}'.format(num, num, descr)))
            f.write(ss.getvalue())
            f.write(htmldiff.EndBlock())
            f.write("\n<hr>\n")
            logging.debug("html done %s", num)
            self.on_has_diff(num, node1, node2, descr)
            logging.debug("on_has_diff done %s", num)
            return True
        self.on_has_not_diff(num)
        return False

    def can_compare(self, node1, node2):
        if self._can_compare and not self._can_compare(node1, node2):
            self.skip_recs += 1
            return False
        return True

    def on_has_diff(self, num, node1, node2, descr):
        """
            Здесь в наследниках можно добавить данных в sbs_info (если используется)
        """
        logging.debug("Has diff")
        self.diff_recs += 1

    def on_has_not_diff(self, num):
        logging.debug("Has no diff")
        self.equal_recs += 1

    def compare_sequence(self, diff_filename, num, it1, it2, it_descr):
        with open(diff_filename, 'w') as df:
            self.write_diff_header(df)
            has_diff = False
            for r in itertools.izip(it1, it2, it_descr):
                has_diff = self.compare(num, r[0], r[1], df, r[2]) or has_diff
                num += 1
            self.write_diff_footer(df)
        if not has_diff:
            os.remove(diff_filename)
        return has_diff

    def write_diff_header(self, f):
        f.write(htmldiff._HTML_HEADER % "app_service")
        f.write(htmldiff._LEGEND)
        f.write(htmldiff._BUTTONS)

    def write_diff_footer(self, f):
        f.write(htmldiff._HTML_FOOTER)

    def data_to_node(self, data):
        """
            convertor data to node_tree
        """
        raise Exception('MUST be implemented by inheritor')

    def patch(self, node):
        """
            apply node_tree patchers (after data_to_node)
            (remove unstable fields, convert field to sub node_tree-s)
        """
        for patcher in self.data_patchers:
            patcher(node)


def data_iterator(data_filename, index_filename, recs_begin, recs_limit, use_index=True):
    with open(data_filename) as fd:
        with open(index_filename) as fi:
            if use_index:
                fi.seek(recs_begin * 8, os.SEEK_SET)
                while True:
                    data_pos = fi.read(8)
                    if not data_pos:
                        break
                    fd.seek(struct.unpack('Q', data_pos)[0], os.SEEK_SET)
                    head = fd.read(4)
                    if len(head) != 4:
                        raise Exception("Unable to read size: broken file content")
                    size = struct.unpack("i", head)[0]
                    body = fd.read(size)
                    if len(body) != size:
                        raise Exception("Unable to read data: broken file content")
                    yield body
            else:
                data_pos = fi.read(8)
                fd.seek(struct.unpack('Q', data_pos)[0], os.SEEK_SET)
                for i, data in enumerate(sequence_binary_data_from_stream(fd)):
                    if i < recs_begin:
                        continue
                    if recs_limit is not None:
                        if recs_limit <= 0:
                            break
                        recs_limit -= 1
                    yield data


def build_index(data_filename, index_filename):
    logging.info('build index {} for {}'.format(index_filename, data_filename))
    with open(data_filename) as f, open(index_filename, 'w') as fi:
        shift = 0
        while True:
            data = f.read(4)
            if not data:
                break
            if len(data) != 4:
                raise Exception('bad data')
            dsize = struct.unpack('i', data)[0]
            f.seek(dsize, os.SEEK_CUR)
            fi.write(struct.pack('Q', shift))
            shift += 4 + dsize
    return index_filename


class FractionDataComparator(Comparator):
    """
        Базовая реализация сравнения указанной части данных (из данных файлов)
        (сравниваем reqs_limit записей, начиная с recs_begin)
    """

    def process(
        self,
        diff_filename,
        recs_begin,
        recs_limit,
        data_filename1,
        index_filename1,
        data_filename2,
        index_filename2,
        descr,
        use_index=True,
    ):
        if descr is None:
            descr_gen = ('' for d in six.moves.range(0, recs_limit))
        else:
            descr_gen = (d for d in descr)
        return self.compare_sequence(
            diff_filename,
            recs_begin,
            data_iterator(data_filename1, index_filename1, recs_begin, recs_limit, use_index),
            data_iterator(data_filename2, index_filename2, recs_begin, recs_limit, use_index),
            descr_gen,
        )


class ChangedProps(response_diff_task.ChangedProps):
    def find_file_for_query(self, q_num, list_of_files):
        for name in list_of_files:
            if name.startswith(DIFF_FILE_PREFIX):
                borders = name[len(DIFF_FILE_PREFIX):-5].split("-")
                if (q_num <= int(borders[1])) and (q_num >= int(borders[0])):
                    return name + '#rec' + str(q_num)


class CompareJob:
    def __init__(
        self,
        comparator,
        diff_filename,
        num,
        recs_per_file,
        data_filename1,
        index_filename1,
        data_filename2,
        index_filename2,
        descr,
        use_index=False,
    ):
        self.comparator = comparator
        self.diff_filename = diff_filename
        self.num = num
        self.recs_per_file = recs_per_file
        self.data_filename1 = data_filename1
        self.index_filename1 = index_filename1
        self.data_filename2 = data_filename2
        self.index_filename2 = index_filename2
        self.descr = descr
        self.use_index = use_index

    def run(self):
        try:
            self.run_unsafe()
        except Exception as e:
            logging.error(traceback.format_exc())
            raise e
        return self

    def run_unsafe(self):
        self.has_diff = self.comparator.process(
            self.diff_filename,
            self.num,
            self.recs_per_file,
            self.data_filename1,
            self.index_filename1,
            self.data_filename2,
            self.index_filename2,
            self.descr,
            self.use_index,
        )


class DiffMaker(object):
    """
        Базовая реализация сравнения 2-х наборов данных (файлы данных + индексов)
        Результаты сравнения сохраняются в указанную директорию
        Необходимо наследование от данного класса для реализации метода-создателя компаратора
    """

    def process(
        self,
        data_filename1,
        index_filename1,
        data_filename2,
        index_filename2,
        diff_dir,
        descr=None,
        recs_per_file=100,
        recs_limit=0,
        process_limit=None,
        soft_timelimit=60 * 60 * 3,  # 3 hours
        use_index=True,
    ):
        if not index_filename1 or not os.path.isfile(index_filename1):
            index_filename1 = build_index(data_filename1, 'data_index_1')
        if not index_filename2 or not os.path.isfile(index_filename2):
            index_filename2 = build_index(data_filename2, 'data_index_2')

        def count_data(index_file):
            return os.path.getsize(index_file) / 8

        cnt = count_data(index_filename1)
        cnt2 = count_data(index_filename2)

        if recs_limit:
            cnt = min(cnt, recs_limit)
            cnt2 = min(cnt2, recs_limit)
        if cnt != cnt2:
            raise Exception('different input data size {} != {}'.format(cnt, cnt2))
        num = 0
        i = 0
        jobs = []
        diff_filenames = []
        while num < cnt:
            if (num + recs_per_file) > cnt:
                recs_per_file = cnt - num
            diff_filename = "{}{:05d}-{:05d}.html".format(DIFF_FILE_PREFIX, num, num + recs_per_file)
            diff_path = os.path.join(diff_dir, diff_filename)
            diff_filenames.append(diff_filename)
            jobs.append(CompareJob(
                self.create_comparator(),
                diff_path,
                num,
                recs_per_file,
                data_filename1,
                index_filename1,
                data_filename2,
                index_filename2,
                descr[num:num + recs_per_file] if descr else None,
                use_index=use_index,
            ))
            i += 1
            num += recs_per_file

        self.has_diff = False
        changed_props = ChangedProps()

        class LinkedSbsMaker(SbsMaker):
            def format_query(self, num, q):
                f = changed_props.find_file_for_query(num, diff_filenames)
                return '<div class="cgi"><a href="{}">{}</a>: {}</div>'.format(f, num, cgi.escape(q))

        self.sbs_maker = LinkedSbsMaker()
        self.skip_recs = 0
        self.diff_recs = 0
        self.equal_recs = 0
        if process_limit != 1:
            # execute comparators in separate processes
            if process_limit == 0:
                process_limit = None
            pool = Pool(process_limit)
            cut_jobs_tl = time.time() + soft_timelimit
            results = [pool.apply_async(job.run, ()) for job in jobs if time.time() < cut_jobs_tl]
            pool.close()
            pool.join()
            for r in results:
                self.on_end_job(r.get(), changed_props)
        else:
            # single process (for debug purpose)
            for job in jobs:
                job.run_unsafe()
                self.on_end_job(job, changed_props)

        if self.has_diff:
            with utils.TimeCounter('generating changed props'):
                with open(os.path.join(diff_dir, 'changed_props.html'), 'w') as out_file:
                    changed_props.write(out_file, diff_dir)
            with utils.TimeCounter('write side_by_side'):
                self.sbs_maker.write(os.path.join(diff_dir, 'side_by_side.html'))

        return self.has_diff

    def on_end_job(self, job, changed_props):
        if job.has_diff:
            self.has_diff = True
            changed_props.combine(job.comparator.changed_props)
            self.sbs_maker.merge_info(job.comparator.sbs_info)
        self.skip_recs += job.comparator.skip_recs
        self.diff_recs += job.comparator.diff_recs
        self.equal_recs += job.comparator.equal_recs

    def create_comparator(self):
        raise Exception('MUST be implemented by inheritor')
