# -*- coding: utf-8 -*-

import sandbox.common.types.client as ctc
import sandbox.sandboxsdk.parameters as sp
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.websearch import release_setup
from sandbox.projects.common.search.requester import Requester

from .search_component import create_noapacheupper_params
from .search_component import get_noapacheupper
from . import request


class Noapacheupper(SandboxTask, object):
    """
        Базовый класс для задач обстрела noapacheupper
    """
    client_tags = ctc.Tag.LINUX_PRECISE
    required_ram = 250 << 10  # 250Gb ram for noapache

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self

    def init_search_component(self, search_component):
        """
            override in subclasses (use valgrind, etc)
        """
        pass


_requester_param_group = 'Requester params'


class Params:
    class Requests(sp.ResourceSelector):
        group = _requester_param_group
        name = 'noapacheupper_requests_resource_id'
        description = 'Requests for noapacheupper'
        resource_type = resource_types.BINARY_REQUESTS_FOR_NOAPACHEUPPER
        required = True

    class RequestsLimit(sp.SandboxIntegerParameter):
        group = _requester_param_group
        name = 'requests_limit'
        description = 'Limit number of used requests (0 = all)'
        default_value = 0

    class UseHttpGet(sp.SandboxBoolParameter):
        # used for sending info request (meminfo test)
        group = _requester_param_group
        name = 'use_http_get'
        description = 'Use http GET method'
        default_value = False

    class WorkersCount(sp.SandboxIntegerParameter):
        group = _requester_param_group
        name = 'workers_count'
        description = 'Use workers (processes) for sending requests'
        default_value = 8

    params = (
        Requests,
        RequestsLimit,
        UseHttpGet,
        WorkersCount,
    )


class WorkaloadNoapacheupper(Noapacheupper):
    """
        Реализация обстрела noapache запросами
        (полученные ответы игнорирутся - отслеживается только наличие ответа)
    """

    client_tags = release_setup.WEBSEARCH_TAGS_P1
    execution_space = 12 * 1024  # 12 Gb

    def __init__(self, task_id=0):
        super(WorkaloadNoapacheupper, self).__init__(task_id)
        self.ctx['kill_timeout'] = 12 * 60 * 60

    @staticmethod
    def get_input_parameters():
        params = (
            create_noapacheupper_params().params +
            Params.params
        )
        return params

    def on_execute(self):
        requests_path = channel.task.sync_resource(self.ctx[Params.Requests.name])
        search_component = get_noapacheupper()
        self.init_search_component(search_component)
        rr = Requester()
        rr.use(
            request.binary_requests_iterator(
                requests_path,
                use_get=self.ctx[Params.UseHttpGet.name],
                use_apphost=self.ctx['noapacheupper_apphost_mode'],
            ),
            search_component,
            process_count=self.ctx[Params.WorkersCount.name],
            requests_limit=self.ctx[Params.RequestsLimit.name]
        )

        proc_info = {}
        if search_component.rss_mem_after_start is not None:
            proc_info['rss_mem_after_start'] = search_component.rss_mem_after_start
        if search_component.rss_mem_before_stop is not None:
            proc_info['rss_mem_before_stop'] = search_component.rss_mem_before_stop
        if len(proc_info):
            self.ctx['proc_info'] = proc_info

        if rr.error:
            raise SandboxTaskFailureError(rr.error)
