# -*- coding: utf-8 -*-

import six
import base64
import json
import logging
import os
import time
import urllib2

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common.noapacheupper.request_common import Request
from sandbox.projects.common.search import requester
from sandbox.projects.common.search import requester_core
from sandbox.projects.websearch.upper import resources as upper_resources


class ResourceParam(sp.ResourceSelector):
    name = 'noapache_requests_resource_id'
    description = 'Noapache requests (size-separated) or base64-encoded text lines'
    resource_type = [
        resource_types.BINARY_REQUESTS_FOR_NOAPACHEUPPER,
        # на самом деле тут не plaintext-запросы,
        # а base64-закодированные патроны, но нам пофиг, потом поправим
        upper_resources.BinaryRequestsForNoapacheAdjuster,
        upper_resources.BinaryRequestsForBlender,
        upper_resources.BlenderGrpcClientPlan,
        resource_types.PLAIN_TEXT_QUERIES,
    ]
    required = True


class AddGlobalCtxParams(sp.SandboxStringParameter):
    name = 'add_global_ctx_params'
    description = 'Add global ctx params (use cgi-format here)'
    default_value = None


def make_hr(q):
    r = Request(q)
    if r.global_ctx.get('hr', '') != 'da':
        r.add_global_ctx_param('hr', 'da')
    return r.serialize()


def remove_dump_params(s):
    s = s.replace("json_dump%3Deventlog%26", "")
    s = s.replace("%26dump%3Deventlog", "")
    s = s.replace(',\"dump\":[\"eventlog\"]', '')
    return s


class CollectRequestNotFound(Exception):
    def __init__(self, message):
        # Call the base class constructor with the parameters it needs
        super(CollectRequestNotFound, self).__init__(message)


def get_post_body(eventlog, suppress_dump_params=False):
    post_body = None
    for logline in eventlog.split('\n'):
        t = logline.rstrip('\n').split('\t')
        if len(t) == 0:
            continue
        if post_body is not None:
            if t[0].isdigit():
                if suppress_dump_params:
                    post_body = remove_dump_params(post_body)
                return post_body.encode('utf-8')
            post_body += '\n' + logline.rstrip('\n')
            continue
        elif len(t) >= 3 and t[2] == "PostBody":
            cgi = '\t'.join(t[3:])
            post_body = cgi
        elif len(t) >= 3 and t[2] == "Base64PostBody":
            post_body = base64.decodestring(t[3])
            if suppress_dump_params:
                post_body = remove_dump_params(post_body)
            return post_body
        elif len(t) >= 6 and t[2] == "AppHostRequest":
            post_body = base64.decodestring(t[5])
            # TODO:
            # if suppress_dump_params:
            #    post_body = remove_dump_params(post_body)
            return post_body
    if post_body is None:
        raise CollectRequestNotFound(eventlog)


def collect_request(qurl, suppress_dump_params=True):
    """
        Получаем запрос, пришедший на noapache
        Формат возвращаемого запроса: str, - не unicode!
        suppress_dump_params - remove &dump=eventlog flag from request
    """
    cnt = 1
    try_count = 0
    qurl += "&json_dump=eventlog"
    while True:  # 'zelo.yandex.ru' often reject(or return empty response) requests, so make few try
        logging.info("ask[{}/{}]: {}".format(cnt, try_count, qurl))
        try_count += 1
        try:
            u = None
            req = urllib2.Request(
                qurl,
                headers={
                    'User-Agent': (
                        'curl/7.22.0 (x86_64-pc-linux-gnu) libcurl/7.22.0 '
                        'OpenSSL/1.0.1 zlib/1.2.3.4 libidn/1.23 librtmp/2.3'
                    ),
                },
            )
            u = urllib2.urlopen(req)
            resp = json.loads(u.read())
            return get_post_body(resp['eventlog'], suppress_dump_params)
        except Exception:
            if try_count > 7:
                raise
            time.sleep(1 * try_count)
        finally:
            if u is not None:
                u.close()


def save_requests(report_url, users_queries, queries_limit, output_file):
    """
        Переколдовываем пользовательские запросы к репорту в запросы к noapache
        (сохраняем в файл в бинарном формате (req-size separate))
    """
    with open(users_queries) as fuq, open(output_file, 'w') as f:
        if not queries_limit:
            queries_limit = None

        for line in fuq:
            qurl = report_url + requester.format_users_queries_line(line)
            req = collect_request(qurl)
            requester.write_binary_data(f, req)
            if queries_limit is not None:
                if not queries_limit:
                    break
                queries_limit -= 1


def create_binary_iterator(filename, gzipped_file=False):
    base64_encoded = False
    if os.path.isdir(filename):
        # apphost reqs stored in folder with 'data' + 'index' files
        filename = os.path.join(filename, 'data')
    else:
        logging.debug("%s is not a dir", filename)
        # it's useful sometimes to store queries as base64-encoded strings (PLAIN_TEXT_QUERIES)
        if os.path.getsize(filename) > 512:
            try:
                logging.debug("Try to decode base64")
                base64_test = open(filename).read(512)
                # https://mvel.at.yandex-team.ru/6360
                if base64_test == base64.b64encode(base64.decodestring(base64_test)):
                    base64_encoded = True
            except Exception as err:
                logging.info("This file seems not to be base64 encoded: %s", str(err))

    iterator = requester.sequence_binary_data
    if gzipped_file:
        iterator = requester.sequence_gzipped_binary_data

    if base64_encoded:
        logging.info("base64 detected, wtf")
        iterator = requester_core.apphost_requests_iter

    return iterator(filename)


def binary_requests_iterator(
    filename,
    add_global_ctx_params=None,
    gzipped_file=False,
    use_get=False,
    use_apphost=False,
    adjuster=False,
):
    for r in create_binary_iterator(filename, gzipped_file):
        if use_apphost:
            if adjuster:
                yield ('/adjust_params', r)
            else:
                yield ('/', r)
        else:
            if add_global_ctx_params:
                r = Request(r).add_global_ctx_cgi_params(add_global_ctx_params).serialize()
            if '\t' in r:
                yield ('/yandsearch', r, {'Content-Type': 'application/ya-multi-json'})
            else:
                if use_get:
                    yield '/yandsearch?' + r
                else:
                    yield ('/yandsearch?', r)


def get_users_queries_for(queries_resource_id):
    # requests to noapacheupper are too huge and complex, so we try find users queries
    # and display original user queries, if we can (or None on failure)
    queries_resource = channel.sandbox.get_resource(queries_resource_id)
    if queries_resource:
        users_queries_resource_id = channel.sandbox.get_resource_attribute(
            queries_resource_id, 'text_queries_resource_id'
        )
        if users_queries_resource_id is None:
            users_queries_resource_id = channel.sandbox.get_resource_attribute(
                queries_resource_id, 'misc_text_queries_resource_id'
            )
        logging.info("User queries resource id: %s", str(users_queries_resource_id))
        return users_queries_resource_id
    return None


##################################################
# fuzzy mutation request methods

def _walk_tree(tree, f):
    if tree is None:
        return tree
    if isinstance(tree, list) or isinstance(tree, tuple):
        lst = []
        for it in tree:
            lst.append(_walk_tree(it, f))
        return lst
    if isinstance(tree, dict):
        dct = {}
        for k, v in tree.iteritems():
            dct[k] = _walk_tree(v, f)
        return dct
    return f(tree)


def _apply_f(r, f):
    r.global_ctx = _walk_tree(r.global_ctx, f)
    r.client_ctx = _walk_tree(r.client_ctx, f)
    r.raw_ctx = _walk_tree(r.raw_ctx, f)


def _fuzzy_num(r, mult):
    def mult_int_if_can(v):
        try:
            if isinstance(v, str):
                return str(int(v) * mult)
            if isinstance(v, six.text_type):
                return six.text_type(int(v) * mult)
            if isinstance(v, int):
                return v * mult
            return v
        except ValueError:
            return v

    _apply_f(r, mult_int_if_can)
    return r


def _fuzzy_cut_in_half(r):
    def cut_half(v):
        if isinstance(v, six.string_types) and len(v) > 1:
            try:
                float(v)
            except ValueError:
                return v[:len(v) / 2]
        return v

    _apply_f(r, cut_half)
    return r


def _fuzzy_duplex(r):
    def duplex(v):
        if isinstance(v, six.string_types):
            return v + v
        return v

    _apply_f(r, duplex)
    return r


class fuzzy_type(object):
    # multiply all digit argument to 10^1, 10^3, 10^6
    mult_num_10 = 1
    mult_num_1k = 2
    mult_num_1m = 3
    # cut half of all values (as string)
    cut_in_half = 4
    # duplicat all values (as string)
    duplex = 5


fuzzy_methods = {
    fuzzy_type.mult_num_10: lambda r: _fuzzy_num(r, 10),
    fuzzy_type.mult_num_1k: lambda r: _fuzzy_num(r, 1000),
    fuzzy_type.mult_num_1m: lambda r: _fuzzy_num(r, 1000000),
    fuzzy_type.cut_in_half: _fuzzy_cut_in_half,
    fuzzy_type.duplex: _fuzzy_duplex,
}


def fuzzy(r, ft):
    """
        main method:
        input - request, fuzzy_type
        sample:
        #for ft in fuzzy_type:
        #    rf = fuzzy(r, ft)
    """
    return fuzzy_methods[ft](r)
