# -*- coding: utf-8 -*-

import logging

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import decorators
from . import beta
from sandbox.sandboxsdk.task import SandboxTask

from sandbox import sdk2


class ImportNoapacheupperResourcesBase(object):
    noapache_instances = 'noapache_instances'
    _nanny_token = None

    @decorators.retries(2, delay=300, backoff=2)
    def _get_config(self, instances):
        """
            Бета имеет право лежать какое-то время, поэтому если не получилось, попробуем
            выбрать другой инстанс или подождать немножко. Перебираем весь список инстансов,
            т.е. пар (host, port)
            :param instances: список строк с инстансами вида "host:port"
        """
        for instance in instances:
            host, port = instance.split(':')
            logging.info("Try to get noapache config from %s:%s", host, port)
            try:
                return utils.get_config_by_info_request(host, port)
            except Exception:
                pass
        logging.error("All instances walked, no valid config downloaded")
        raise Exception("All instances walked, no valid config downloaded")

    @decorators.retries(3, delay=180, backoff=2)
    def _get_noapache_list(self, beta_url):
        """
            Безопасно получить список инстансов беты
            :param beta_url: beta url
            :return: список строк с инстансами вида "host:port"
        """
        return beta.noapache_list(beta.name_from_url(beta_url), nanny_token=self._nanny_token)

    def _get_instances(self, beta_url):
        result = []
        try:
            result = self._get_noapache_list(beta_url)
            if not result:
                raise Exception("Got empty instances list from beta {}".format(beta_url))
            logging.info("Got instances list of size %s", len(result))
        except Exception as exc:
            eh.log_exception("Problem obtaining noapache instances list", exc)
            eh.check_failed("Cannot detect noapacheupper hostname. ")
        return result


class ImportNoapacheupperResources(SandboxTask, ImportNoapacheupperResourcesBase):
    """
        Класс реализует импорт ресурсов noapache с указанного инстанса noapache
        (набирает ресурсы выясняя их sandbox-id или выкачивая и помещая в sandbox)
        используем наследованием-подмешиванием в нужные sandbox таски - наследники от SandboxTask.
    """

    def import_resources(self, beta_url):
        """
            Получить список инстансов верхнего метапоиска с беты
        """
        logging.info("Obtain resources for tests...")
        instances_list = self._get_instances(beta_url)

        fu.write_file('noapache.cfg', self._get_config(instances_list))
        config_resource = self.create_resource(
            'Config resource from beta {} for {}'.format(beta_url, self.descr),
            'noapache.cfg',
            resource_types.NOAPACHEUPPER_CONFIG,
        )
        # task can spawn children, so we need to save results
        self.mark_resource_ready(config_resource)
        resources = beta.get_resources_id(instances_list)
        resources['NOAPACHEUPPER_CONFIG'] = config_resource.id

        return resources


class ImportNoapacheupperResources2(sdk2.Task, ImportNoapacheupperResourcesBase):
    """
        Класс реализует импорт ресурсов noapache с указанного инстанса noapache
        (набирает ресурсы выясняя их sandbox-id или выкачивая и помещая в sandbox)
        используем наследованием-подмешиванием в нужные sandbox таски - наследники от sdk2.Task.
    """

    def import_resources(self, beta_url):
        """
            Получить список инстансов верхнего метапоиска с беты
        """
        logging.info("Obtain resources for tests...")
        instances_list = self._get_instances(beta_url)

        config_resource = resource_types.NOAPACHEUPPER_CONFIG(
            self, 'Config resource from beta {} for {}'.format(beta_url, self.Parameters.description), 'noapache.cfg'
        )
        file_resource_data = sdk2.ResourceData(config_resource)
        fu.write_file(file_resource_data.path, self._get_config(instances_list))
        file_resource_data.ready()

        resources = beta.get_resources_id(instances_list, use_sdk2=True)
        resources['NOAPACHEUPPER_CONFIG'] = config_resource.id

        return resources
