# -*- coding: utf-8 -*-

import six
from sandbox.projects.common.base_search_quality import basesearch_response_parser as brp
from sandbox.projects.common.base_search_quality import response_saver
from sandbox.projects.common.base_search_quality.tree.node import WalkPath
from sandbox.projects.common.base_search_quality.tree.htmldiff import _append_list_item, _convert_json_dict_to_tree_impl
from sandbox.projects.common.search import response_patcher as rp
from sandbox.projects.common.search.response import state as rs

import logging

not_stable_search_props = (
    'Nctx',

    'ApplyAdsBlender.PureGroupingTime',
    'ApplyAdsBlender.NoGroupingBlenderTime',
    'ApplyBlender.PureGroupingTime',
    'ApplyBlender.NoGroupingBlenderTime',
    'ApplyVideoBlender.PureGroupingTime',
    'ApplyVideoBlender.NoGroupingBlenderTime',

    # these two should be ignored on WEB, see SEARCH-5330
    "ApplyVideoBlender.factors",  # SEARCH-5330
    "ApplyVideoBlender.dynamic_factors",  # SEARCH-5330

    # these two should be ignored on WEB, see SEARCH-5506
    "ApplyImagesBlender.factors",  # SEARCH-5506
    "ApplyImagesBlender.dynamic_factors",  # SEARCH-5506

    'ApplyImagesBlender.PureGroupingTime',
    'ApplyImagesBlender.NoGroupingBlenderTime',
    'Acc',
    'HostName',
    'Ig',
    'SourceTimestamp',
    'WaitInfo.debug',
    'WaitInfo2.debug',
    'ReqError.debug',
    'SerpFact.Timedout',
    'SerpFact.FactShown',
    'SerpFact.FactJson.debug',
    'SerpFact.Fact.debug',
    'SerpFact.CandidateFacts',
    'SerpObject.ErrorMessage.debug',
)

not_stable_search_properties = (
    'BsTouched',
    'Nctx',
    'CalcMetaRelevanceMcs',  # SEARCH-9579
    'DocumentsStats',  # SEARCH-11681
)


def iterate_answers(node, answer_type):
    name = 'answers_{}'.format(answer_type)
    num = 0
    while name in node.nodes():
        response = node.get_node(name)
        yield response

        num += 1
        name = 'answers_{}#{}'.format(answer_type, num)


def _sort_lines(lines):
    return '\n'.join(sorted(lines.split('\n')))


def convert_searcher_prop_protobuf_hr_to_tree(response, prop_key):
    searcher_props = response.nodes().get("SearcherProp")
    tree_key = 'SearcherProp-' + prop_key
    if searcher_props:
        tmp_searcher_props = []
        tmp_tries = []
        for sp in searcher_props:
            key = sp.GetPropValue("Key")
            if key == prop_key:
                tmp_tries.append(brp._parse_single_response(sp.GetPropValue("Value")))
            else:
                tmp_searcher_props.append(sp)
        searcher_props = tmp_searcher_props
        response.nodes()["SearcherProp"] = searcher_props
        if tmp_tries:
            if tree_key in response.nodes():
                response.nodes()[tree_key].extend(tmp_tries)
            else:
                response.nodes()[tree_key] = tmp_tries


default_query_data = 'QueryData.json.debug'
default_scheme_key = 'scheme.json.nodump'


def common_response_patcher(response):
    response_saver.remove_config_info_props(response)
    brp.remove_unstable_properties(response)


def remove_search_errors_debug(node):
    """
        Used until SEARCH-5251 is fixed (avitella@, elshiko@, mvel@, dima-zakharov@)
    """
    for response in iterate_answers(node, 'blender'):
        report = response.get_node('report')
        searcher_props = report.nodes().get("SearcherProp")
        if not searcher_props:
            continue

        searcher_props = [prop for prop in searcher_props if not prop.GetPropValue("Key") == 'SearchErrors.debug']
        report.nodes()["SearcherProp"] = searcher_props


def patch_dynamic_factors(node):
    """
        Do better diff for dynamic_factors SEARCH-4846
    """
    for response in iterate_answers(node, 'blender'):
        try:
            searcher_props = response.get_node('report').nodes().get("SearcherProp")
            for sp in searcher_props:
                key = sp.GetPropValue("Key")
                value_prop = sp.GetPropValue("Value", required=False)
                if key == "ApplyBlender.dynamic_factors":
                    parsed_factors = _convert_dynamic_factors(value_prop)
                    _append_list_item(
                        sp._nodes,
                        key + "_parsed",
                        _convert_json_dict_to_tree_impl(parsed_factors)
                    )
        except Exception as e:
            logging.debug(e)


def patch_factors(node):
    """
        Do better diff for dynamic_factors SEARCH-4846
    """
    for response in iterate_answers(node, 'blender'):
        try:
            searcher_props = response.get_node('report').nodes().get("SearcherProp")
            for sp in searcher_props:
                key = sp.GetPropValue("Key")
                value_prop = sp.GetPropValue("Value", required=False)
                if key == "ApplyBlender.factors" or key == "ApplyBlender.factors_d.debug":
                    parsed_factors = _convert_factors(value_prop)
                    _append_list_item(
                        sp._nodes,
                        key + "_parsed",
                        _convert_json_dict_to_tree_impl(parsed_factors)
                    )

        except Exception as e:
            logging.debug(e)


def patch_compressed_factors(node):
    """
    NOAPACHE-256
    """
    for response in iterate_answers(node, 'blender'):
        try:
            searcher_props = response.get_node('report').nodes().get("SearcherProp")
            for sp in searcher_props:
                key = sp.GetPropValue("Key")
                value_prop = sp.GetPropValue("Value", required=False)
                if key in (
                    "ApplyBlender.compressed_factors",
                    "ApplyImagesBlender.compressed_factors",
                    "ApplyVideoBlender.compressed_factors"
                ):
                    parsed_factors = _convert_compressed_factors(value_prop)
                    _append_list_item(
                        sp._nodes,
                        key + "_parsed",
                        _convert_json_dict_to_tree_impl(parsed_factors)
                    )

        except Exception as e:
            logging.debug(e)


def _convert_dynamic_factors(factors):
    split_factors = factors.split()
    result = dict()
    index = 0
    while index < len(split_factors):
        key = split_factors[index]
        count = int(split_factors[index + 1])
        result[key] = " ".join(split_factors[index + 1:index + 2 + count])
        index += count + 2
    return result


def _convert_factors(factors):
    split_factors = factors.split()
    result = dict()
    for i, factor in enumerate(split_factors):
        result[str(i)] = factor
    return result


def _convert_compressed_factors(factors):
    import kernel.blender.factor_storage.pylib.serialization as fs

    error, static, dynamic = fs.decompress(factors)
    if error is not None:
        raise Exception(error)

    result = dict()
    for i, factor_value in enumerate(static):
        result['static.{}'.format(i)] = factor_value

    for key, value in dynamic.items():
        result['dynamic.{}'.format(key)] = value

    return result


def _patch_searcher_props(response):
    """
    Remove/Sort unstable SearcherProp-s
    """
    searcher_props = response.nodes().get("SearcherProp")
    if not searcher_props:
        return

    filtered_searcher_props = []

    for sp in searcher_props:
        key = sp.GetPropValue("Key")
        if key in not_stable_search_props:
            continue
        if key.startswith('ConfigInfo_'):
            continue  # remove unstable configuration info (exe/cfg info)
        if key.startswith('DebugConfigInfo_'):
            continue
        if key == 'SearchErrors.debug':
            val = sp.GetPropValue("Value")
            if not val:
                continue
            sp.SetPropValue("Value", _sort_lines(val))
        filtered_searcher_props.append(sp)
    searcher_props = filtered_searcher_props
    response._nodes["SearcherProp"] = searcher_props

    for sp in searcher_props:
        # try reoreder unstable serilaziation ApplyBlender.*.debug data (result hash serialization)
        key = sp.GetPropValue("Key")
        ktdi = _key_after_tdi_id(key)
        if ktdi:
            key = ktdi
        if key == "ApplyBlender.#fmls" or key == "ApplyBlender.fmls":
            rp.sort_search_prop_values(sp, ' ', '|')
        elif key == "ApplyBlender.slices":
            rp.sort_search_prop_values(sp, ' ')
        elif key.startswith("ApplyBlender.") and key.endswith(".debug"):
            if key.endswith(".#inserted.debug"):
                rp.sort_search_prop_values(sp, "; ")
            else:
                rp.sort_search_prop_values(sp, ". ")
        elif key == "Vertical.EmptyIntent":
            rp.sort_search_prop_values(sp, '|')
        elif key == "Personalization.ag":
            rp.sort_search_prop_values(sp, '|')
        elif key == "EntitySearch.ListOntoids":
            rp.sort_search_prop_values(sp, '|')


def response_patcher(response):
    """
        Convert json to tree (to remove unstable json parts),
        remove or reorder unstable upper-specific parameters
    """
    _patch_scheme_tree(response, default_scheme_key)
    _patch_scheme_local_standard_tree(response)
    _patch_query_data(response, default_query_data)
    # rp.convert_serp_data_to_tree(response)
    _patch_searcher_props(response)

    # reason rare unstability "BsTouched" now unknown yet
    def ProcessProperties(p):
        if p.GetPropValue("Key") in not_stable_search_properties:
            p.SetPropValue("Value", "##UNSTABLE##")
    WalkPath(response, ("SearchProperties", "Properties"), ProcessProperties)

    # sort unoredered doc attribute value (result hash serialization)
    attr = "Slices="

    def ProcessGtaRelatedAttribute(n):
        key = n.GetPropValue("Key")
        if key == "_Markers":
            val = n.GetPropValue("Value")
            if val.startswith(attr):
                ordered_val = '"{}{}"'.format(attr, rp.sort_val(val[len(attr):-1], '|'))
                n.SetPropValue("Value", ordered_val)

    WalkPath(
        response,
        ("Grouping", "Group", "Document", "ArchiveInfo", "GtaRelatedAttribute"),
        ProcessGtaRelatedAttribute,
    )

    # props from multicontext tdi experiments prefixed by exp. number
    _patch_tdi_props(response)


def _patch_scheme_tree(response, key_scheme):
    key_scheme_tree = rp.convert_scheme_to_tree(response, key_scheme)
    _patch_old_query_data(response, key_scheme_tree)
    _patch_msc_scheme(response, key_scheme_tree)


def _patch_scheme_local_standard_tree(response, key_scheme=rp.KEY_SCHEME):
    key_scheme_tree = rp.convert_scheme_to_tree(response, key_scheme=key_scheme)
    if key_scheme_tree is None:
        key_scheme_tree = "SearcherProp-{}".format(key_scheme)

    _patch_old_query_data(response, key_scheme_tree)
    _patch_msc_scheme(response, key_scheme_tree)


def _patch_msc_scheme(response, key_scheme_tree):
    # remove {scheme-root}/Porno/only_porno_fix_pl, - it is recevied from few sources (IMAGESP, IMAGESP_MISSPELL),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree,),
        lambda node:
            node._props.pop('ImgPornoPl', None))

    # remove {scheme-root}/Porno/only_porno_fix_pl, - it is recevied from few sources (WEB, WEB_MISSPELL, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "Porno"),
        lambda node:
            node._props.pop('only_porno_fix_pl', None))

    # remove {scheme-root}/blender/vk_nav/docs/vkontakte.ru/Title - depends on response order (WEB, WEB_MISSPELL, ...)
    WalkPath(
        response,
        (key_scheme_tree, "blender", "vk_nav", "docs", "vkontakte.ru"),
        lambda node:
            node._props.pop('Title', None))

    # stange not stable field from alex-sh@
    WalkPath(
        response,
        (key_scheme_tree, "blender", "TIMELINE", "Rules"),
        lambda node:
            node._props.pop('Timeline', None))

    # remove {scheme-root}/Local/Vertical/ when have empty name intents they values mixed in same place (GEOV, TV, ...),
    # so value depend on response order (not stable, - based on mixed value 'weight')
    # WalkPath(response, (key_scheme_tree, "Local"),
    #    lambda node:
    #        node._nodes.pop('Vertical', None))


def _patch_old_query_data(response, key_scheme_tree):
    # remove {scheme-root}/OldQueryData/pess_owner_ban_exp, - it is recevied from few sources (WEB, WEB_MISSPELL, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData"),
        lambda node:
            node._nodes.pop('pess_owner_ban_exp', None))

    # remove {scheme-root}/OldQueryData/porno_rotation_experiment,
    # - it is recevied from few sources (WEB, WEB_MISSPELL, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData"),
        lambda node:
            node._nodes.pop('porno_rotation_experiment', None))

    # remove {scheme-root}/OldQueryData/categmaskmarkup_malware/timestamp,
    # - it is recevied from few sources (WEB, QUICK, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "categmaskmarkup_malware"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/categmaskmarkup_malware/timestamp,
    # - it is recevied from few sources (WEB, QUICK, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "categmaskban_malware_current"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/categmaskban_malware/timestamp,
    # - it is recevied from few sources (WEB, QUICK, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "categmaskban_malware"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/categflag_totalban2/timestamp,
    # - it is recevied from few sources (WEB, QUICK, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "categflag_totalban2"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/docid_setprops/timestamp, - it is recevied from few sources (WEB, QUICK, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "docid_setprops"),
        lambda node:
            node._props.pop('timestamp', None))
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "docid_setprops_content_plugins"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/docid_setprops/social_annotations/timestamp
    # - depends on response order (WEB, QUICK, ...)
    WalkPath(
        response,
        (key_scheme_tree, 'OldQueryData', 'docid_setprops/social_annotations'),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/docid_setprops_social_annotations - depends on response order (WEB, QUICK, ...)
    WalkPath(
        response,
        (key_scheme_tree, 'OldQueryData'),
        lambda node:
            node._nodes.pop('docid_setprops_social_annotations', None))

    # remove {scheme-root}/OldQueryData/fastsnips/timestamp
    WalkPath(
        response,
        (key_scheme_tree, 'OldQueryData', 'fastsnips'),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/masksflag_safesearch/timestamp
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "masksflag_safesearch"),
        lambda node:
            node._props.pop('timestamp', None))

    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "porno_fix_url"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/masksflag_safesearch/timestamp
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "scheme_fix_middle"),
        lambda node:
            node._props.pop('timestamp', None))

    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "porno_fix_host"),
        lambda node:
            node._props.pop('timestamp', None))

    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "porn_adv"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/trash_masks/timestamp, - it is recevied from few sources (WEB, QUICK, ...),
    # so value depend on response order (not stable, - stored last received)
    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "trash_masks"),
        lambda node:
            node._props.pop('timestamp', None))

    WalkPath(
        response,
        (key_scheme_tree, "OldQueryData", "test_trash_masks"),
        lambda node:
            node._props.pop('timestamp', None))

    # remove {scheme-root}/OldQueryData/short_beak/values/* - depends on response order (MISSPELL, IMAGES_MISSPELL)
    WalkPath(
        response,
        (key_scheme_tree, 'OldQueryData', 'short_beak'),
        lambda node:
            node._nodes.pop('values', None))


def _patch_query_data(response, query_data=default_query_data):
    # convert QueryData.json.debug from huge string (protobuf hr) to tree
    convert_searcher_prop_protobuf_hr_to_tree(response, query_data)
    # sort SearcherProp-QueryData.json.debug/SourceFactors nodes
    query_data_hr = response._nodes.get('SearcherProp-' + query_data)
    if query_data_hr:
        # QueryData SourceFactors Node key
        def sf_key(sf):
            return (
                sf._props.get('SourceName', ''),
                sf._props.get('SourceKeyType', ''),
                sf._props.get('Version', ''),
            )

        for qd in query_data_hr:
            sfs = qd._nodes.get('SourceFactors')
            if sfs:
                qd._nodes['SourceFactors'] = sorted(sfs, key=sf_key)


def _key_after_tdi_id(key):
    # try remove digits + '_' prefix from key and return it or return None
    # (use format tdi props stated by epar@ in SEARCH-983)
    ksize = len(key)
    if ksize and key[0].isdigit():
        for i in six.moves.range(1, ksize):
            if not key[i].isdigit():
                break
        if i < ksize and key[i] == '_':
            return key[i + 1:]
    return None


def _patch_tdi_props(response):
    # if props: starts with from digit it is can be tdi version props (SEARCH-983)
    scheme_keys = []
    scheme_local_standard_keys = []
    query_data_keys = []
    searcherProps = response.nodes().get("SearcherProp")
    if searcherProps:
        filteredSearcherProps = []
        for sp in searcherProps:
            tdi_key = sp.GetPropValue("Key")
            key = _key_after_tdi_id(tdi_key)
            if key:
                if key in not_stable_search_props:
                    continue
                if key.startswith('ConfigInfo'):
                    continue
                if key.startswith('DebugConfigInfo'):
                    continue
                if key.startswith('SearchErrors.debug'):
                    val = sp.GetPropValue("Value")
                    if val:
                        sp.SetPropValue("Value", _sort_lines(val))
                    else:
                        continue
                if key == rp.KEY_SCHEME:
                    scheme_local_standard_keys.append(tdi_key)
                elif key == 'scheme.json.nodump':
                    scheme_keys.append(tdi_key)
                elif key == default_query_data:
                    query_data_keys.append(tdi_key)

            filteredSearcherProps.append(sp)
        response._nodes["SearcherProp"] = filteredSearcherProps

    for k in scheme_keys:
        _patch_scheme_tree(response, k)
    for k in scheme_local_standard_keys:
        _patch_scheme_local_standard_tree(response, k)
    for k in query_data_keys:
        _patch_query_data(response, k)


# apphost method
def remove_noapache_specific_noise(node):
    """
        Удалить из ответа поиска известные шумные поля
    """
    for response in iterate_answers(node, 'blender'):
        common_response_patcher(response.get_node('report'))
        response_patcher(response.get_node('report'))


def can_compare_noapache_apphost(node1, node2):
    responses = [list(iterate_answers(node1, 'blender')), list(iterate_answers(node2, 'blender'))]

    if len(responses[0]) != len(responses[1]):
        return False

    for response1, response2 in zip(responses[0], responses[1]):
        if not rs.can_compare(response1.get_node('report'), response2.get_node('report')):
            return False
    return True
