# -*- coding: utf-8 -*-

import httplib
import logging
import time

from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.noapacheupper.request import binary_requests_iterator
from sandbox.projects.common.noapacheupper.request import ResourceParam
from sandbox.projects.common.noapacheupper.search_component import Params as NoapacheParams
from sandbox.projects.common.search.requester import Params as RequesterParams
from sandbox.projects.common.search import requester_core
from sandbox.projects.common import error_handlers as eh


def get_url_avg_response_time(url, requests_filename, requests_limit, use_apphost=False):
    """
        :return: среднее время ответа на запросы в миллисекундах
    """
    u = None
    reqs_cnt = 0
    reqs_time = 0
    logging.info('Start asking %s (calc avg. resp. time) (limit=%s)', url, requests_limit)
    for req in binary_requests_iterator(requests_filename, use_apphost=use_apphost):
        if requests_limit > 0 and requests_limit == reqs_cnt:
            break

        req = requester_core.build_request(url, req)
        begin = time.time()
        try:
            u = requester_core.url_open(req, 20)
            u.read()
        except httplib.BadStatusLine as err:
            eh.log_exception("Bad noapache response (bad status line), seems service crashed", err)
            eh.check_failed("Bad noapache response (bad status line), seems service crashed")
        finally:
            if u:
                u.close()
        end = time.time()
        if end > begin:
            reqs_time += end - begin
        reqs_cnt += 1
        if (reqs_cnt % 100) == 0:
            avg_resp_time = 1000. * reqs_time / reqs_cnt
            logging.debug("Processed %s requests (avg_rt=%s)", reqs_cnt, avg_resp_time)
    if reqs_cnt == 0:
        raise Exception(
            "Queries list is empty. "
            "Please check your input parameters or plan resources. "
        )
    return 1000. * reqs_time / reqs_cnt


def get_avg_response_time(search_component, ctx):
    """
        используя ресурс с запросами из ctx, опросить поисковый компонент,
        вернуть среднее время запроса (в миллисекундах)
    """

    logging.info('get_avg_response_time')
    search_component.start()
    search_component.wait()
    avg_resp_time = search_component.use_component(
        lambda: get_url_avg_response_time(
            "http://localhost:{}".format(search_component.get_port()),
            channel.task.sync_resource(ctx[ResourceParam.name]),
            ctx.get(RequesterParams.RequestsLimit.name, 0),
            use_apphost=ctx[NoapacheParams.AppHostMode.name],
        )
    )
    search_component.stop()

    return avg_resp_time
