# -*- coding: utf-8 -*-

import logging
import os
import re
import datetime
import urllib2
import json

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process

from sandbox.projects.release_machine import notify_helper
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import config as sconf
from sandbox.projects.common.noapacheupper import request
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects import resource_types


def create_noapacheupper_params(
    neh_cache_mode=None,
    n=None,
    settings=True,
    add_requests=False,
    with_evlogdump=False,
    config_params=None,
):
    """
        :param neh_cache_mode: if 'read' then add params for choose neh cache resource
        :param n: for multiple components in one task
        :param settings: add settings params pack
        :param add_requests: add requests parameter
    """
    n_suffix = "" if n is None else " {}".format(n)
    n = "" if n is None else "_{}".format(n)
    group_name = 'Noapacheupper{} parameters'.format(n_suffix)

    class ComponentParams(object):
        class Binary(sp.ResourceSelector):
            name = 'noapacheupper{}_executable_resource_id'.format(n)
            description = 'Noapacheupper{} executable'.format(n_suffix)
            group = group_name
            resource_type = upper_resources.NoapacheUpper

        class Config(sp.ResourceSelector):
            name = 'noapacheupper{}_config_resource_id'.format(n)
            description = 'Noapacheupper{} config'.format(n_suffix)
            group = group_name
            resource_type = resource_types.NOAPACHEUPPER_CONFIG
            required = False

        class Data(sp.ResourceSelector):
            name = 'noapacheupper{}_data_resource_id'.format(n)
            description = 'Noapacheupper search data (dirs "rearrange", "rearrange.dynamic", "rearrange.fast")'
            group = group_name
            resource_type = resource_types.NOAPACHEUPPER_DATA
            required = False

        class RearrangeData(sp.ResourceSelector):
            name = 'rearrange_data{}_resource_id'.format(n)
            description = 'Alternate rearrange data'
            group = group_name
            resource_type = resource_types.REARRANGE_DATA
            required = False

        class RearrangeDynamicData(sp.ResourceSelector):
            name = 'rearrange_dynamic_data{}_resource_id'.format(n)
            description = 'Alternate rearrange.dynamic data'
            group = group_name
            resource_type = resource_types.REARRANGE_DYNAMIC_DATA
            required = False

        class RearrangeDataFast(sp.ResourceSelector):
            name = 'rearrange_data_fast{}_resource_id'.format(n)
            description = 'Alternate rearrange.fast data'
            group = group_name
            resource_type = upper_resources.RearrangeDataFast
            required = False

        class Evlogdump(sp.ResourceSelector):
            name = 'noapacheupper{}_evlogdump_resource_id'.format(n)  # for CALC_EVENTLOG_STATS
            description = 'Evlogdump'
            group = group_name
            resource_type = resource_types.EVLOGDUMP_EXECUTABLE
            required = False

        class OldRearrangeStagesOrder(sp.SandboxBoolParameter):
            name = 'noapacheupper{}_old_rearrange_stages_order'.format(n)
            description = 'Use old rearrange stages order (SEARCH-8183)'
            default_value = False

        class MaxCalcRelevQueueSize(sp.SandboxIntegerParameter):
            name = 'noapacheupper{}_max_calc_relev_queue_size'.format(n)
            description = 'MaxCalcRelevQueueSize'
            default_value = None

        class ConfigDebugOptions(sp.SandboxStringParameter):
            name = 'noapacheupper{}_config_debug_options'.format(n)
            description = 'DebugOptions in config'
            default_value = None

    class Settings(object):

        class AppHostMode(sp.SandboxBoolParameter):
            name = 'noapacheupper_apphost_mode'
            description = 'Use apphost mode'
            group = group_name
            default_value = False

        class StartTimeout(sp.SandboxIntegerParameter):
            name = 'noapacheupper_start_timeout'
            description = 'Start timeout (sec)'
            group = group_name
            default_value = sc.DEFAULT_START_TIMEOUT

        class ShutdownTimeout(sp.SandboxIntegerParameter):
            name = 'noapacheupper_shutdown_timeout'
            description = 'Shutdown timeout (sec)'
            group = group_name
            default_value = 120

        class ServerInputDeadline(sp.SandboxStringParameter):
            name = 'noapacheupper_server_input_timeout'
            description = 'DDoS protection (read input data timeout, empty_string == not change)'
            group = group_name
            default_value = ''

        class NehCache(sp.LastReleasedResource):
            name = 'noapacheupper_neh_cache_resource_id'
            description = 'Noapacheupper neh cache (subsources responses for standalone tests)'
            group = group_name
            resource_type = [resource_types.NOAPACHEUPPER_NEH_CACHE, upper_resources.BlenderNehCache]
            required = False

    class RequestOptions(object):
        class Requests(request.ResourceParam):
            name = 'noapache{}_requests_resource_id'.format(n)
            required = False

    class FlexParams(ComponentParams, Settings, RequestOptions):
        params = (
            ComponentParams.Binary,
            ComponentParams.Config,
            ComponentParams.Data,
            ComponentParams.RearrangeData,
            ComponentParams.RearrangeDynamicData,
            ComponentParams.RearrangeDataFast,
        )
        if with_evlogdump:
            params += (
                ComponentParams.Evlogdump,
                ComponentParams.OldRearrangeStagesOrder,
            )

        if settings:
            params += (
                sc.VerifyStderrCommon,
                Settings.AppHostMode,
                Settings.StartTimeout,
                Settings.ShutdownTimeout,
                Settings.ServerInputDeadline,
            )

            if neh_cache_mode == 'read':
                params += (
                    Settings.NehCache,
                )

        if add_requests:
            params += (
                RequestOptions.Requests,
            )

        if config_params:
            params += (
                ComponentParams.MaxCalcRelevQueueSize,
                ComponentParams.ConfigDebugOptions,
            )

    return FlexParams


Params = create_noapacheupper_params()


class Noapacheupper(sc.Middlesearch):
    name = 'noapacheupper'
    default_port = 10150
    need_dump_tass = False
    tass_dump = None
    _LOAD_LOG_FILENAME = 'noapacheupper{}_load.log'
    _SERVER_LOG_FILENAME = 'noapacheupper%s_server.log'
    _EVENT_LOG_FILENAME = 'noapacheupper%s_event.log'

    def __init__(self, **kwargs):
        config_params = kwargs.get('config_params', {})
        config_params['Server/ClientTimeout'] = 2000
        config_params['Server/ScarabReqAnsLog'] = None
        config_params['Server/DirectScarabReqAnsLog'] = None
        config_params['Server/SiteSearchScarabReqAnsLog'] = None
        config_params['Server/ReqAnsLog'] = None
        config_params['Server/PushAgentOptions'] = None
        kwargs.update(config_params=config_params)
        if 'port' not in kwargs:
            kwargs.update(port=self.default_port)
        if 'use_verify_stderr' not in kwargs:
            kwargs.update(use_verify_stderr=True)
        if 'outputs_to_one_file' not in kwargs:
            kwargs.update(outputs_to_one_file=False)
        sc.Middlesearch.__init__(self, **kwargs)

    def warmup_request(self):
        """
            Для верхнего метапоиска прогревочный запрос вреден (см. SEARCH-616 в Middlesearch, r1419472),
            так что вместо этого вычитываем версию (чтобы убедиться, что приложение хоть как-то заработало)
        """
        try:
            logging.info("Warming up...")
            url = "http://127.0.0.1:{}/{}?info=getversion".format(self.port, self.http_collection)
            logging.info("Fetch version info: " + url)
            logging.info("Use noapacheupper version: " + urllib2.urlopen(url, timeout=300).read())
        except Exception as e:
            eh.check_failed("Cannot fetch noapacheupper version. Error: {}".format(e))

    @property
    def _stop_command(self):
        return "/admin?action=shutdown&timeout={}".format(self.shutdown_timeout if self.shutdown_timeout else 300)

    def fetch(self, url, **kwargs):
        if self.shutdown_timeout and 'timeout' not in kwargs:
            kwargs['timeout'] = self.shutdown_timeout  # valgrind require more slow shutdown
        return sc.Middlesearch.fetch(self, url, **kwargs)

    def verify_stderr(self, file_name):
        with open(file_name) as f:
            warnings = []

            for line in f:
                line = line.rstrip()
                if not line:
                    continue

                line_len = len(line)
                if self.sanitize_type is not None and line.startswith("=================="):
                    break  # SEARCH-6733

                log_line = re.sub(r'^[0-9T\-:.]*(?:Z|[+-]\d{2}:?\d{2})\t', '', line)

                # check free-way stderr output
                if 'Out of buffer' in log_line:
                    # UPREL-6014#60ec4c2de5dbf4126436a607
                    continue

                eh.ensure(len(log_line) < line_len, "Unexpected stderr output: '{}'".format(line))

                log_type, log_text = log_line.split('\t', 1)
                if log_type == '[INFO]' or log_type == '[DEBUG]':
                    continue

                if log_type == '[WARNING]':
                    if 'Mappings is not clear' in log_line:
                        logging.debug('Ignore warning: %s', log_line)
                        continue
                    warnings.append(log_line)
                    continue

                if "Rearrange rule " in log_line and "doesn't exist" in log_line:
                    logging.debug("Found stderr log line: '%s' (SEARCH-6030)", log_line)  # TODO remove this SEARCH-6030
                    notify_helper.telegram(
                        self,
                        "Found stderr log line: '{}' (SEARCH-6030) in task {}".format(
                            log_line,
                            lb.task_link(channel.task.id, plain=True),
                        ),
                        people=['mvel']
                    )
                    continue

                # SEARCH-6714
                if "Can't lock self memory" in log_line:
                    logging.debug(
                        "Ignore error: '%s'.\n"
                        "Fail on locking memory is not that awfull. (approved by @avitella)",
                        log_line
                    )
                    continue

                eh.check_failed("Unexpected stderr log line: '{}'".format(log_line))

            if warnings:
                notify_helper.telegram(
                    channel.task,
                    "Found WARNING(s) in noapache log in this task: {}\nHere is some of them:\n{}".format(
                        lb.task_link(channel.task.id, plain=True),
                        '\n'.join(warnings[:3]),
                    ),
                    people=['elshiko', 'mvel']
                )

    def get_port(self):
        # FIXME: Get AppHostPort from config
        if self.apphost_mode:
            return int(self.port) + 1

        return self.port

    def update_src_accessibility(self, src_acc, evlogdump):
        # update src_acc = {} # src_name -> [success_count, error_count]
        srcs_idx = {}  # src_num -> src_name
        srcs_stat = {}  # src_num -> [success_count, error_count]
        cnt_frames = 0

        logging.info('get sources accessibility from eventlog ' + self.get_event_log_path())
        with open(self.get_event_log_path(), 'rb') as inp:
            proc = process.run_process(
                [evlogdump, '-i', 'SubSourceInit,SubSourceRequest,SubSourceOk,SubSourceError'],
                stdin=inp,
                log_prefix="parse_evlog",
                wait=True,
                check=True,
                outputs_to_one_file=False,
            )

        with open(proc.stdout_path) as fev:
            curr_frame = -1
            curr_reqs = {}
            for line in fev:
                tk = line.split('\t')
                if tk[1] != curr_frame:
                    if curr_reqs:
                        raise Exception('not closed reqs: {} in frame {}'.format(curr_reqs, curr_frame))
                    cnt_frames += 1
                curr_frame = tk[1]
                if tk[2] == 'SubSourceInit':
                    src_id = int(tk[3])
                    if src_id in srcs_idx:
                        # paranoid consistency validation
                        if srcs_idx[src_id] != tk[6]:
                            raise Exception(
                                'Conflict for source id to source name mapping: frame={}, src={}'.format(
                                    curr_frame, src_id
                                )
                            )
                    else:  # update index
                        srcs_idx[src_id] = tk[6]
                elif tk[2] == 'SubSourceRequest':
                    src_id = int(tk[3])
                    attempt = int(tk[4])
                    task_num = int(tk[8])
                    curr_reqs[(src_id, attempt, task_num)] = None
                elif tk[2] == 'SubSourceOk':
                    src_id = int(tk[3])
                    attempt = int(tk[4])
                    task_num = int(tk[9])
                    del curr_reqs[(src_id, attempt, task_num)]
                    if src_id not in srcs_stat:
                        srcs_stat[src_id] = [0, 0]
                    srcs_stat[src_id][0] += 1
                elif tk[2] == 'SubSourceError':
                    src_id = int(tk[3])
                    attempt = int(tk[4])
                    task_num = int(tk[11])
                    del curr_reqs[(src_id, attempt, task_num)]
                    if src_id not in srcs_stat:
                        srcs_stat[src_id] = [0, 0]
                    srcs_stat[src_id][1] += 1

        #####################################
        # merge results to src_acc
        for k, v in srcs_stat.iteritems():
            src_name = srcs_idx[k]
            if src_name not in src_acc:
                src_acc[src_name] = [0, 0]
            acc = src_acc[src_name]
            acc[0] += v[0]
            acc[1] += v[1]

    def set_dump_tass(self):
        self.need_dump_tass = True

    def get_tass_output(self):
        return self.tass_dump

    def _dump_tass(self):
        self.tass_dump = json.loads(self.fetch('/tass'))
        logging.debug(json.dumps(self.tass_dump, indent=4))

    def _stop_impl(self):
        if self.need_dump_tass:
            self._dump_tass()
        super(Noapacheupper, self)._stop_impl()


def get_noapacheupper(**kwargs):
    """
        Синхронизировать ресурсы и получить обёртку для noapacheupper
        (https://wiki.yandex-team.ru/poiskovajaplatforma/upple/)
        - объект класса Middlesearch.
        Идентификаторы параметров-ресурсов берутся из контекста таска
        @param neh_cache_mode: 'read' - брать ответы подисточников из neh cache,
                               'write' - записывать ответы подисточников в neh cache,
                               'nodelay_read' - брать ответы подисточников из neh cache и
                                                не эмулировать задержку ответа источником
                               None - не использовать neh cache (Default)
    """

    neh_cache_mode = None
    if 'neh_cache_mode' in kwargs:
        neh_cache_mode = kwargs['neh_cache_mode']
        del kwargs['neh_cache_mode']
    params = None
    if 'params' in kwargs:
        params = kwargs['params']
        del kwargs['params']
    if params is None:
        params = create_noapacheupper_params(neh_cache_mode)

    ctx = channel.task.ctx

    config_file = kwargs.get('config_file')
    if not config_file:
        config_file = channel.task.sync_resource(ctx[params.Config.name])

    pure_dir = None
    rearrange_id = ctx.get(params.RearrangeData.name)
    rearrange_dir = channel.task.sync_resource(rearrange_id) if rearrange_id else None
    rearrange_dynamic_id = ctx.get(params.RearrangeDynamicData.name)
    rearrange_dynamic_dir = channel.task.sync_resource(rearrange_dynamic_id) if rearrange_dynamic_id else None
    rearrange_data_fast_id = ctx.get(params.RearrangeDataFast.name)
    rearrange_data_fast_dir = channel.task.sync_resource(rearrange_data_fast_id) if rearrange_data_fast_id else None

    data_id = ctx.get(params.Data.name)
    if data_id:
        data_path = channel.task.sync_resource(data_id)
        pure_dir = os.path.join(data_path, "pure")
        if not rearrange_dir:
            rearrange_dir = os.path.join(data_path, "rearrange")
        if not rearrange_dynamic_dir:
            rearrange_dynamic_dir = os.path.join(data_path, "rearrange.dynamic")
        if not rearrange_data_fast_dir:
            rearrange_data_fast_dir = os.path.join(data_path, "rearrange.fast")

    neh_cache_destinations = kwargs.pop('neh_cache_destinations', [])

    def destinations():
        yabs_dest = [
            "full2://yabs.yandex.ru:80/code/2->yabs_cgi",
            "full2://yabs.yandex.com.tr:80/code/90->yabs_com_tr2_cgi",
            "full2://yabs.yandex.com.tr:80/code/165->yabs_com_tr_cgi",
            "full2://yabs.yandex.com.tr:80/code/238->yabs_com_tr3_cgi",
            "full2://yabs.yandex.ua:80/code/63->yabs_ua_cgi",
            "full2://yabs.yandex.ru:80/code/90->yabs_ru_cgi",
            "full2://yabs.yandex.kz:80/code/113->yabs_kz_cgi",
            "full2://yabs.yandex.by:80/code/129->yabs_by_cgi",  # TODO:remove this hack?!
            "full2://yabs.yandex.ru:80/code/165->yabs_ru2_cgi",
            "full2://yabs.yandex.ru:80/code/238->yabs_ru3_cgi",
            "full2://yabs.yandex.ru:80/page/244->yabs_ru4_cgi",
            "full2://yabs.yandex.ru:80/code/183->yabs_ru5_cgi",
            "full2://yabs.yandex.ru:80/code/184->yabs_ru6_cgi",
            "full2://yabs.yandex.ru:80/code/186->yabs_ru7_cgi",
            "full2://yabs.yandex.ru:80/code/254->yabs_ru8_cgi",
            "full2://yabs.yandex.ru:80/code/243449->yabs_ru9_cgi",
            "full2://yabs.yandex.ru:80/code/187->yabs_ru10_cgi",  # I think we should fix it
            "full2://yabs.yandex.by:80/code/243449->yabs_by2_cgi",
            "full2://yabs.yandex.ua:80/code/243449->yabs_ua2_cgi",
            "full2://yabs.yandex.kz:80/code/243449->yabs_kz2_cgi",
            "full2://yabs.yandex.com.tr:80/code/243449->yabs_tr4_cgi",
        ]
        dest = []
        if neh_cache_destinations:
            for d in neh_cache_destinations:
                dest.append('->'.join(d))
        return ' '.join(yabs_dest + dest)

    logs_folder = paths.get_logs_folder()
    neh_cache_log = os.path.join(logs_folder, 'neh_cache.log')
    neh_cache_errlog = os.path.join(logs_folder, 'neh_cache_err.log')
    common_neh_cache_opt = (
        ', WeakHash=yes, Format=IndexedFile'
        ', LogFile={}, ErrLogFile={}, Destinations={}'
    ).format(neh_cache_log, neh_cache_errlog, destinations())
    if neh_cache_mode == 'write':
        neh_cache = 'Mode=write' + common_neh_cache_opt
    elif neh_cache_mode == 'read':
        neh_cache_dir = channel.task.sync_resource(ctx[params.NehCache.name])
        neh_cache = 'Mode=read, DelayResponse=no, Folder={}{}'.format(neh_cache_dir, common_neh_cache_opt)
    elif neh_cache_mode == 'nodelay_read':
        neh_cache_dir = channel.task.sync_resource(ctx[params.NehCache.name])
        neh_cache = 'Mode=read, Folder={}{}'.format(neh_cache_dir, common_neh_cache_opt)
    else:
        neh_cache = None
        neh_cache_errlog = None

    kwargs.update(
        is_int=False,
        work_dir=channel.task.abs_path(),
        binary=channel.task.sync_resource(ctx[params.Binary.name]),
        config_class=sconf.NoapacheupperConfig,
        config_file=config_file,
        pure_dir=pure_dir,
        rearrange_dir=rearrange_dir,
        rearrange_dynamic_dir=rearrange_dynamic_dir,
        rearrange_data_fast_dir=rearrange_data_fast_dir,
        start_timeout=utils.get_or_default(ctx, params.StartTimeout),
        neh_cache=neh_cache,
        use_verify_stderr=kwargs.get('use_verify_stderr', True),
        apphost_mode=ctx.get(params.AppHostMode.name, False),
        server_input_deadline=ctx.get(params.ServerInputDeadline.name),
    )

    noapacheupper = Noapacheupper(**kwargs)
    if neh_cache_mode == 'read' or neh_cache_mode == 'nodelay_read':
        noapacheupper.config.add_dns_cache_hosts([
            ('yabs.yandex.ru', 'localhost'),
            ('yabs.yandex.com.tr', 'localhost'),
            ('yabs.yandex.ua', 'localhost'),
            ('yabs.yandex.kz', 'localhost'),
            ('yabs.yandex.by', 'localhost'),
        ])
        noapacheupper.config.set_src_subsources_one_name()
    noapacheupper.shutdown_timeout = ctx.get(params.ShutdownTimeout.name, params.ShutdownTimeout.default_value)
    noapacheupper.neh_cache_errlog = neh_cache_errlog  # for validating success usage cache
    return noapacheupper
