# -*- coding: utf-8 -*-

import six
import json
import logging
import os
import shutil
import re

from sandbox import common
from sandbox.common.types.client import Tag
from sandbox.common.fs import get_dir_size

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.task import SandboxTask

import sandbox.projects.common.search.components.mkl as sc_mkl
from sandbox.projects import resource_types
from sandbox.projects.app_host.resources import AppHostGrpcClientExecutable
from sandbox.projects.websearch import release_setup
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.differ import coloring
from sandbox.projects.common.base_search_quality import basesearch_response_parser as brp
from sandbox.projects.common.noapacheupper import request
from sandbox.projects.common.noapacheupper import response_patcher as noapache_response_patcher
from sandbox.projects.common.noapacheupper.diff.app_host import DiffApphostResponses
from sandbox.projects.common.noapacheupper.diff.hr import DiffProtoHrResponses
from sandbox.projects.common.noapacheupper.response_time import get_avg_response_time as get_avg_response_time2
from sandbox.projects.common.noapacheupper.response_parser import unpack_to_files
from sandbox.projects.common.search import config as sconf
from sandbox.projects.common.search.eventlog import check_eventlog
from sandbox.projects.common.search.rearrange_timings import calc_timings_from_evlog as calc_rearrange_timings
from sandbox.projects.common.search.requester import Requester
from sandbox.projects.common.search.requester import Params as RequesterParams
from sandbox.projects.common.search.requester import BinaryResponseSaver
from sandbox.projects.common.search.response.state import can_compare
from sandbox.projects.websearch.upper import resources as upper_resources
from .blender_tmpl_data_patcher import can_compare_blender_tmpl_data
from .blender_tmpl_data_patcher import prepare_blender_tmpl_data_for_comparison
from .response_patcher import can_compare_noapache_apphost
from .response_patcher import remove_noapache_specific_noise
from .search_component import create_noapacheupper_params
from .search_component import get_noapacheupper
from .search_component import Params as NoapacheParams

_group_neh_cache_params = 'Noapacheupper neh cache parameter'


class Params:
    class EmulateResponseDelay(sp.SandboxBoolParameter):
        name = 'emulate_response_delay'
        description = 'Emulate delay response from source'
        group = _group_neh_cache_params
        default_value = True

    class IgnoreNehCacheErrors(sp.SandboxBoolParameter):
        name = 'ignore_neh_cache_errors'
        description = 'Ignore neh cache errors (misshit obviously)'
        group = _group_neh_cache_params
        default_value = False

    class CheckResponsesStability(sp.SandboxBoolParameter):
        name = 'check_responses_stability'
        description = 'Check responses stability (use double run and compare responses)'
        group = _group_neh_cache_params
        default_value = False

    class IgnoreLoopInMergeErrors(sp.SandboxBoolParameter):
        name = 'ignore_loop_in_merge_errors'
        description = 'Ignore merge in loop errors (can be caused slow responses in memcheck tests)'
        default_value = False

    class EvlogdumpExecutable(check_eventlog.Params.EvlogdumpExecutable):
        pass

    params = (
        EmulateResponseDelay,
        IgnoreNehCacheErrors,
        CheckResponsesStability,
        EvlogdumpExecutable,
        IgnoreLoopInMergeErrors,
    )


class StandaloneNoapacheupper(SandboxTask, check_eventlog.CheckEventlog):
    """
        Базовый класс для задач обстрела noapacheupper с использование neh cache
        (набор кеша/использования кеша в автономном тесте + сбор ответов для сравнения)
    """
    noapacheupper_responses_resource_id = 'noapacheupper_responses_resource_id'
    unpacked_responses_resource_id = 'unpacked_responses_resource_id'
    neh_cache_resource_id = 'neh_cache_resource_id'
    client_tags = Tag.LINUX_PRECISE

    execution_space = 60000

    def __init__(self, task_id=0, neh_cache_mode='read'):
        """
             :param neh_cache_mode: 'write' - формировать кеш ответов от подисточников (НЕ автономный режим)
                                    'read' - получать ответы от подисточников из кеша (автономный режим)
        """
        SandboxTask.__init__(self, task_id)
        check_eventlog.CheckEventlog.__init__(self)
        self.checked_events.append('SubSourceOk')
        self.checked_events.append('SubSourceError')
        self.neh_cache_mode = neh_cache_mode
        # hacks for ignore specific errors
        self.last_misshit_cache_frame = -1
        self.prev_line_is_ok_entity_search = False

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        self.ctx['invalid_reqids'] = self.ctx.get('invalid_reqids', [])
        self.ctx['personalization_errors'] = self.ctx.get('personalization_errors', [])
        self.required_ram = 250 << 10  # 250Gb ram for noapache
        if self.neh_cache_mode == 'write':
            if utils.get_or_default(self.ctx, RunMode) == 'blender':
                self.create_noapacheupper_resource(self.neh_cache_resource_id,
                                                   'blender neh cache with subsources request results',
                                                   'neh_cache',
                                                   upper_resources.BlenderNehCache)
            else:
                self.create_noapacheupper_resource(self.neh_cache_resource_id,
                                                   'noapacheupper neh cache with subsources request results',
                                                   'neh_cache',
                                                   resource_types.NOAPACHEUPPER_NEH_CACHE)

    def init_search_component(self, search_component):
        """
            override in subclasses (use valgrind, etc)
        """
        pass

    def create_noapacheupper_resource(self, ctx_id, descr, filename, resource_type):
        self.ctx[ctx_id] = self.create_resource(
            descr,
            filename,
            resource_type,
            attributes={
                "ttl": 14,
            },
            arch='any',
        ).id

    def check_neh_cache_errors(self, noapacheupper_search):
        if self.ctx.get(Params.IgnoreNehCacheErrors.name, False):
            return

        errlog = os.path.join(self.abs_path(), noapacheupper_search.neh_cache_errlog)
        if os.path.getsize(errlog) > 0:
            unknown_destinations = set()
            ud_regexp = re.compile("neh cache: unknown destination: (.*)")
            yabs_issue = False
            with open(errlog) as f:
                for line in f:
                    line = line.rstrip()
                    matched = ud_regexp.search(line)
                    if matched:
                        dest = matched.group(1)
                        unknown_destinations.add(dest)
                        if "yabs" in dest:
                            yabs_issue = True
            if unknown_destinations:
                channel.task.set_info("detected unknown destinations, here's some of them:\n{}".format(
                    " ".join(list(unknown_destinations)[:10])
                ))
            if yabs_issue:
                channel.task.set_info(
                    "Detected yabs misshit, try to add yabs destination into neh_cache config\n"
                    "You should add it in that file: "
                    "//trunk/arcadia/sandbox/projects/common/noapacheupper/search_component.py"
                )

            raise SandboxTaskFailureError(
                'detected neh cache error (misshit obviously), - for details see {}'.format(
                    noapacheupper_search.neh_cache_errlog
                )
            )

    def _link_neh_cache_from_requests(self):
        requests = channel.sandbox.get_resource(self.ctx[request.ResourceParam.name])
        linked_neh_cache = requests.attributes.get('neh_cache')
        if not utils.get_or_default(self.ctx, NoapacheParams.NehCache) and linked_neh_cache:
            self.ctx[NoapacheParams.NehCache.name] = linked_neh_cache
            logging.info("Linked neh cache resource %s", linked_neh_cache)

    def _get_neh_cache_mode(self):
        if self.neh_cache_mode == 'read' and not self.ctx.get(Params.EmulateResponseDelay.name, False):
            return 'nodelay_read'
        return self.neh_cache_mode

    def on_eventlog_line(self, tk):
        # override handler for handle special guest event: SubSourceError
        frame = int(tk[1])
        event = tk[2]
        ignore = False
        if event == 'SubSourceError':
            if tk[-1].endswith('(yexception) response not found'):
                # store current frame if it was neh cache misshit trouble for ignore induсed errors
                self.last_misshit_cache_frame = frame
            ignore = True
        elif event == 'SubSourceOk':
            ignore = True
        elif event == 'ErrorMessage':
            if utils.get_or_default(self.ctx, RunMode) == RunModes.blender:
                # It's OK to ignore this, because SERP_OBJECT deliberately disabled
                if tk[-1].endswith('(yexception) can not get reply from SERP_OBJECT'):
                    ignore = True
                # TODO @avitella SEARCH-5202
                elif tk[-1] == 'loop in merge':
                    logging.debug("line ignored, but need to be fixed: %s", tk)
                    ignore = True
                # TODO @avitella SEARCH-5202
                elif tk[-1].startswith("can't convert region"):
                    logging.debug("line ignored, but need to be fixed: %s", tk)
                    ignore = True
                # TODO @avitella SEARCH-5202
                elif "Caught exception in MakePage" in tk[-1]:
                    logging.debug("line ignored, but need to be fixed: %s", tk)
                    ignore = True
                # TODO mvel@, dima-zakharov@ SEARCH-7079
                elif "Illegal requests to IMAGESP_MISSPELL" in tk[-1]:
                    logging.debug("line ignored, but need to be fixed: %s", tk)
                    ignore = True
                # TODO mvel@, dima-zakharov@ NOAPACHE-1
                elif "Illegal requests to VIDEOP" in tk[-1]:
                    logging.debug("line ignored, but need to be fixed: %s", tk)
                    ignore = True
                # TODO mvel@, dima-zakharov@ NOAPACHE-1
                elif "Illegal requests to IMAGESP" in tk[-1]:
                    logging.debug("line ignored, but need to be fixed: %s", tk)
                    ignore = True
                elif "Invalid ReqId" in tk[-1]:
                    logging.debug("line ignored, but will be sent to telegram: %s", tk)
                    try:
                        req_id = tk[-1].split("Invalid ReqId")[1].strip()
                    except Exception as e:
                        logging.debug("Tried to get reqid from %s, got exception - %s", tk[-1], e)
                        req_id = "failed to parse"
                    self.ctx['invalid_reqids'] = self.ctx.get('invalid_reqids', [])
                    self.ctx['invalid_reqids'].append(req_id)
                    ignore = True
        elif event == 'RearrangeError':
            pass

        if not ignore:
            check_eventlog.CheckEventlog.on_eventlog_line(self, tk)
        self.prev_line_is_ok_entity_search = (event == 'SubSourceOk' and tk[8] == 'ok-entity_search')

    def on_event_ErrorMessage(self, frame, text):
        if text.startswith('bad rt userdata timestamps'):  # ignore this case in sandbox tests
            self.log_errors.append('ErrorMessage at frame {}: {}'.format(frame, text))
        elif text.startswith('Request timestamp '):  # ignore this case in sandbox tests
            self.log_errors.append('ErrorMessage at frame {}: {}'.format(frame, text))
        elif text.startswith('exception caught while parsing user profile'):  # ignore this case in sandbox tests
            self.log_errors.append('ErrorMessage at frame {}: {}'.format(frame, text))
        elif (
            text.startswith('catch exception from MetaSearchJob') and
            text.endswith('can not get reply from SERP_OBJECT')
        ):
            self.log_errors.append('ErrorMessage at frame {}: {}'.format(frame, text))
        elif text.startswith('loop in merge'):
            # TODO: remove True (temporary ignoring merge loop error - wait finish release noapache-26)
            if True or self.ctx.get('ignore_loop_in_merge_errors', False) or self.last_misshit_cache_frame == frame:
                # ignore errors caused neh cache miss SEARCH-1090
                self.log_errors.append('ErrorMessage at frame {}: {}'.format(frame, text))
            else:
                check_eventlog.CheckEventlog.on_event_ErrorMessage(self, frame, text)
        elif self.prev_line_is_ok_entity_search and text.endswith('Stream is exhausted'):
            # ENTITY_SEARCH source can return empty 200 Ok response - ignore such errors
            self.log_errors.append('ErrorMessage at frame {}: {}'.format(frame, text))
        else:
            check_eventlog.CheckEventlog.on_event_ErrorMessage(self, frame, text)


class ProtoHrResponses(sp.SandboxBoolParameter):
    name = 'response_format_proto_hr'
    description = 'Save hr=da output resource (SEARCH_HR_RESPONSES)'
    default_value = False


class ToolConverter(sp.ResourceSelector):
    name = 'tool_converter'
    description = 'Tool to convert various apphost protocol messages in/out of json'
    resource_type = resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE


class ToolProto2Hr(sp.ResourceSelector):
    name = 'tool_proto2hr'
    description = 'Tool for convert raw proto to proto-hr format'
    resource_type = resource_types.PROTO2HR_EXECUTABLE


class AppHostOps(sp.ResourceSelector):
    name = 'app_host_ops'
    description = 'New tool to convert various apphost protocol messages in/out of json (app_host_ops)'
    resource_type = upper_resources.APP_HOST_OPS_EXECUTABLE


class RunModes(object):
    upper = 'upper'
    adjuster = 'adjuster'
    blender = 'blender'


class RunMode(sp.SandboxRadioParameter):
    name = 'run_mode'
    description = 'Run mode: '
    choices = [(mode_name, mode_name) for mode_name in [RunModes.upper, RunModes.adjuster, RunModes.blender]]
    default_value = RunModes.upper


class GrpcClientExecutable(sp.ResourceSelector):
    name = 'grpc_client'
    description = "Use specific grpc client to shoot (higher priority than next option)"
    resource_type = AppHostGrpcClientExecutable
    required = False


class UseLastStableGrpcClient(sp.SandboxBoolParameter):
    default_value = False
    name = 'use_last_stable_grpc_client'
    description = "Use last stable grpc client to shoot"


class CustomReqNumbers(sp.SandboxStringParameter):
    name = 'custom_req_numbers'
    description = 'Send requests only with specified numbers, i.e. "5, 7, 100" will send only those 3 requests. '\
                  'Each of them will be send 10 times, to help reproduce incorrect behaviour.'
    default_value = ""


class MetricConstraint(sp.SandboxStringParameter):
    name = 'noapacheupper_metric_constraint'
    description = 'Explore noapacheupper output of /tass'
    default_value = ''


class ResponsesStandaloneNoapacheupper(StandaloneNoapacheupper):
    def __init__(self, task_id=0, neh_cache_mode='read'):
        super(ResponsesStandaloneNoapacheupper, self).__init__(task_id, neh_cache_mode)

    @staticmethod
    def get_input_parameters(neh_cache_mode):
        params = (
            create_noapacheupper_params(neh_cache_mode=neh_cache_mode).params +
            (
                request.ResourceParam,
                request.AddGlobalCtxParams,
                ProtoHrResponses,
                ToolConverter,
                ToolProto2Hr,
                AppHostOps,
                RunMode,
                CustomReqNumbers,
                GrpcClientExecutable,
                UseLastStableGrpcClient,
                MetricConstraint,
            ) +
            BinaryResponseSaver.create_params().lst
        )
        if neh_cache_mode == 'read':
            params += Params.params
        return params

    @property
    def footer(self):
        if not self.is_completed() or not self.ctx.get(Params.CheckResponsesStability.name):
            return None
        stb_res = self.list_resources(resource_type='BASESEARCH_RESPONSES_COMPARE_RESULT')
        if len(stb_res):
            diff_count = '<h4><a href="{}">responses stability</a> statistics</h4>'.format(
                stb_res[0].remote_http(),
            )
        else:
            diff_count = '<h4>responses stability statistics</h4>'
        if self.ctx.get('not_stable_responses', False):
            diff_count += "\n<b style='color:{}'>Has not stable responses</b>".format(coloring.DiffColors.bad)
        fields_to_show = ['skip_responses', 'diff_responses', 'equal_responses', 'total_responses']
        return [{
            'helperName': '',
            'content': {
                diff_count: {
                    "header": [
                        {"key": "Key", "title": "Key"},
                        {"key": "Value", "title": "Value"},
                    ],
                    "body": {
                        "Key": fields_to_show,
                        "Value": [self.ctx.get(k) for k in fields_to_show]
                    }
                }
            }
        }]

    def on_enqueue(self):
        super(ResponsesStandaloneNoapacheupper, self).on_enqueue()
        self.create_noapacheupper_resource(
            self.noapacheupper_responses_resource_id,
            'noapacheupper responses',
            'resp.out',
            self.get_response_type(),
        )
        if self.neh_cache_mode != 'write':
            self.create_noapacheupper_resource(
                self.unpacked_responses_resource_id,
                'unpacked noapacheupper responses',
                'unpacked_resp.out',
                self.get_response_type(unpacked=True),
            )
        if self.ctx[ProtoHrResponses.name]:
            if self.ctx[request.AddGlobalCtxParams.name] is None:
                self.ctx[request.AddGlobalCtxParams.name] = 'hr=da'
            if 'hr=da' not in self.ctx[request.AddGlobalCtxParams.name]:
                self.ctx[request.AddGlobalCtxParams.name] += '&hr=da'

    def on_execute(self):
        self._link_neh_cache_from_requests()
        if self.neh_cache_mode == 'read' and (
            utils.get_or_default(self.ctx, RunMode) in [RunModes.adjuster] or
            not utils.get_or_default(self.ctx, NoapacheParams.NehCache)
        ):
            neh_cache_mode = None
        else:
            neh_cache_mode = self._get_neh_cache_mode()
        params = create_noapacheupper_params(neh_cache_mode=neh_cache_mode)
        config_file = channel.task.sync_resource(self.ctx[params.Config.name])
        patched_config_file = self._patch_config_timeouts(config_file)
        patched_config_file = self._patch_config_dump_hr_props(patched_config_file)
        if self.neh_cache_mode != 'write':
            patched_config_file = self._patch_config_search_source_cgi_search_prefix(patched_config_file)
        search_component = get_noapacheupper(
            params=params,
            config_file=patched_config_file,
            neh_cache_mode=neh_cache_mode
        )
        constraints = self.get_metric_constraints()
        if constraints:
            search_component.set_dump_tass()

        sc_mkl.configure_mkl_environment(search_component)
        self.init_search_component(search_component)
        responses_resource = channel.sandbox.get_resource(self.ctx[self.noapacheupper_responses_resource_id])
        if neh_cache_mode != 'write':
            unpacked_responses_resource_path = channel.sandbox.get_resource(
                self.ctx[self.unpacked_responses_resource_id]
            ).path
        else:
            unpacked_responses_resource_path = None
        self.save_responses(search_component, responses_resource.path, unpacked_responses_resource_path)
        self.check_neh_cache_errors(search_component)
        self.check_eventlog_errors(search_component.event_log_path)
        if constraints:
            self.check_stats(search_component, constraints)
        eh.ensure(
            get_dir_size(str(responses_resource.path)) > 100,
            "Responses size is too small",
        )
        # self.mark_resource_ready(responses_resource)
        if self.ctx.get(Params.CheckResponsesStability.name):
            if not self.apphost_mode() and not self.ctx.get(ProtoHrResponses.name, True):
                raise SandboxTaskFailureError('now can verify proto responses stability only using hr format responses')
            self.save_second_run_responses(search_component)

            with utils.TimeCounter('get users queries'):
                queries_res_id = request.get_users_queries_for(self.ctx[request.ResourceParam.name])
                queries = None
                if queries_res_id:
                    res_path = self.sync_resource(queries_res_id)
                    with open(res_path) as f:
                        queries = [line.strip() for line in f]

            # queries should also be filtered in the same manner as requests
            queries = list(self.iter_wrapper(queries))

            diff_dir = 'check_responses_stability_result'
            if self.apphost_mode():
                diff_maker = self.compare_apphost_responses(
                    queries, responses_resource.path, self.second_responses_resource_path, diff_dir
                )
            else:
                diff_maker = self.compare_hr_responses(
                    queries, responses_resource, diff_dir
                )

            self.ctx['skip_responses'] = diff_maker.skip_recs
            self.ctx['diff_responses'] = diff_maker.diff_recs
            self.ctx['equal_responses'] = diff_maker.equal_recs
            self.ctx['total_responses'] = total_recs = (
                diff_maker.skip_recs +
                diff_maker.diff_recs +
                diff_maker.equal_recs
            )
            if (float(diff_maker.skip_recs) / total_recs) > 0.3:  # allowed 30% uncompared responses
                raise common.errors.TemporaryError(
                    'Too many uncompared queries ({} from {}) in responses stability check'.format(
                        diff_maker.skip_recs, total_recs
                    )
                )

            if diff_maker.has_diff:
                self.ctx['not_stable_responses'] = True
                self.create_resource_second_run_responses()
                diff_resource = channel.task.create_resource(
                    description=self.descr,
                    resource_path=diff_dir,
                    resource_type=resource_types.BASESEARCH_RESPONSES_COMPARE_RESULT,
                    attributes={
                        "ttl": 14,
                    },
                )
                # self.mark_resource_ready(diff_resource.id)
                self.set_info("There are <a href={}>diff</a>".format(diff_resource.proxy_url), do_escape=False)

    def _patch_config_timeouts(self, config_filename):
        """
            Multiplies every 'TimeOut=*' by 100
        """
        fu.write_lines(
            'patched_config',
            (re.sub('TimeOut=([0-9]*)', r'TimeOut=\g<1>00', line) for line in fu.read_line_by_line(config_filename))
        )
        self._patch_config_append_to_scheme_options(
            'patched_config',
            'SerpFact/Timeout=5000,SerpObject/Timeout=10000'
        )
        return 'patched_config'

    def _patch_config_append_to_scheme_options(self, config_filename, value):
        scheme_options = ''
        for line in fu.read_line_by_line(config_filename):
            splitted = line.rstrip('\n').lstrip(' ').split()
            if len(splitted) >= 2 and splitted[0] == 'SchemeOptions':
                scheme_options = splitted[1]
                break
        scheme_options = ','.join(filter(None, (scheme_options, value)))

        config = sconf.SearchConfig(open(config_filename).read())
        config.apply_local_patch({'Collection/SchemeOptions': scheme_options})
        config.save_to_file(config_filename)
        return config_filename

    def _patch_config_dump_hr_props(self, config_filename):
        """
            Add DumpHrProps param
        """
        with open(config_filename) as f:
            config = sconf.SearchConfig(f.read())
        config.apply_local_patch({'Collection/DumpHrProps': "true"})
        config.save_to_file(config_filename)
        return config_filename

    def _patch_config_search_source_cgi_search_prefix(
        self,
        config_filename,
        search_source='SERP_OBJECT',
        cgi_search_prefix='http://localhost:17000/yandsearch?',
    ):
        """
            Patch CgiSearchPrefix of some <SearchSource>
        """
        inside_source_section = False
        inside_required_search_source_section = True
        text_lines = []
        for line in fu.read_line_by_line(config_filename):
            sl = line.rstrip('\n').lstrip(' ')
            if sl.startswith("</SearchSource>"):
                inside_source_section = False
                inside_required_search_source_section = False
            elif sl.startswith("<SearchSource>"):
                inside_source_section = True
            elif inside_source_section:
                if sl.startswith('ServerDescr {}'.format(search_source)):
                    inside_required_search_source_section = True
                elif inside_required_search_source_section:
                    if sl.startswith("CgiSearchPrefix"):
                        line = "CgiSearchPrefix {}".format(cgi_search_prefix)
            text_lines.append(line)
        fixed_text = "\n".join(text_lines) + "\n"

        fu.write_file('patched_config', fixed_text)
        logging.debug("Fixed text:\n%s", fixed_text)
        return 'patched_config'

    def get_response_type(self, unpacked=False):
        if utils.get_or_default(self.ctx, RunMode) == RunModes.adjuster:
            return (
                upper_resources.NoapacheAdjusterResponses if not unpacked
                else upper_resources.NoapacheAdjusterUnpackedResponses
            )
        if utils.get_or_default(self.ctx, RunMode) == RunModes.blender:
            return (
                upper_resources.NoapacheBlenderResponses if not unpacked
                else upper_resources.NoapacheBlenderUnpackedResponses
            )
        if self.ctx[ProtoHrResponses.name]:
            return resource_types.SEARCH_HR_RESPONSES
        if self.apphost_mode():
            return (
                resource_types.RESPONSES_SEARCH_APP_SERVICE if not unpacked
                else upper_resources.UpperIntUnpackedResponses
            )
        return resource_types.SEARCH_PROTO_RESPONSES

    def apphost_mode(self):
        return self.ctx.get(NoapacheParams.AppHostMode.name, False)

    def on_start_get_responses(self):
        pass

    def get_req_numbers(self):
        custom_req_numbers = utils.get_or_default(self.ctx, CustomReqNumbers)
        if custom_req_numbers:
            return map(int, custom_req_numbers.split(','))
        return None

    def iter_wrapper(self, iter):
        if iter is None:
            return
        numbers = self.get_req_numbers()
        if numbers:
            for i, elem in enumerate(iter):
                if i in numbers:
                    for _ in six.move.range(10):
                        yield elem
        else:
            for elem in iter:
                yield elem

    def save_responses(self, search_component, responses_resource_path, unpacked_responses_resource_path=None):
        saver = BinaryResponseSaver(self.ctx, search_component)
        requests = self.sync_resource(self.ctx[request.ResourceParam.name])
        data = os.path.join(responses_resource_path, 'data')
        index = os.path.join(responses_resource_path, 'index')

        if (
            utils.get_or_default(self.ctx, UseLastStableGrpcClient)
            or utils.get_or_default(self.ctx, GrpcClientExecutable)
        ):
            if utils.get_or_default(self.ctx, GrpcClientExecutable):
                grpc_client_res_id = utils.get_or_default(self.ctx, GrpcClientExecutable)
                logging.debug("Got specific grpc client: %s", grpc_client_res_id)
            else:
                grpc_client_res_id = utils.get_and_check_last_released_resource_id(
                    AppHostGrpcClientExecutable,
                    use_new_version=True,
                )
                logging.debug("Got last released grpc client: %s", grpc_client_res_id)
            grpc_client_path = self.sync_resource(grpc_client_res_id)
            os.chmod(grpc_client_path, 0o755)
            with search_component as sc:
                try:
                    rps = 10
                    if self.ctx.get(CustomReqNumbers.name):
                        custom_req_path = os.path.join(os.getcwd(), 'custom_req')
                        with open(custom_req_path, 'w') as custom_requests, open(requests) as f:
                            custom_requests.writelines(self.iter_wrapper(f))
                        requests = custom_req_path
                    with open(requests) as f:
                        requests_number = sum(1 for _ in f)
                    requests_limit = utils.get_or_default(self.ctx, RequesterParams.RequestsLimit)
                    if requests_limit:
                        requests_number = min(requests_number, requests_limit)
                    cmd = "{} localhost:{} -P {} -t bin -s {} -r {} -o {}".format(
                        grpc_client_path, int(sc.port) + 2, requests, rps, requests_number, responses_resource_path
                    )
                    logging.debug("Try to run process: %s", cmd)
                    process.run_process(cmd, shell=True, log_prefix="grpc_client_shoot")
                except Exception as err:
                    eh.log_exception("Fail in grpc mode", err)
        else:
            if os.path.isdir(requests):
                requests = os.path.join(requests, 'data')
            req_iter = request.binary_requests_iterator(
                requests,
                self.ctx.get(request.AddGlobalCtxParams.name),
                use_apphost=self.apphost_mode(),
                adjuster=(utils.get_or_default(self.ctx, RunMode) == RunModes.adjuster)
            )
            req_iter = self.iter_wrapper(req_iter)
            if self.apphost_mode():
                make_folder(responses_resource_path)
                saver.run(req_iter, data, index)
            else:
                saver.run(req_iter, responses_resource_path)
        if unpacked_responses_resource_path:
            make_folder(unpacked_responses_resource_path)
            converter = self.sync_resource(self.ctx[ToolConverter.name])
            logging.debug("Start unpacking responses")
            unpack_to_files(
                data,
                index,
                0,
                utils.get_or_default(self.ctx, RequesterParams.RequestsLimit) or None,
                converter,
                unpacked_responses_resource_path
            )
            logging.debug("Finished unpacking responses")
        self.ctx['queries_resource_id'] = self.ctx[request.ResourceParam.name]

    def save_second_run_responses(self, search_component):
        self.second_responses_resource_path = 'tmp_resp2.out'
        self.second_unpocked_responses_resource_path = 'tmp_unpacked_resp2.out'
        self.current_action('save second version responses')
        self.save_responses(
            search_component,
            self.second_responses_resource_path,
            self.second_unpocked_responses_resource_path
        )

    def create_resource_second_run_responses(self):
        second_responses_resource = self.create_resource(
            'second version noapacheupper responses for stability check',
            'resp2.out',
            self.get_response_type(),
            attributes={
                "ttl": 14,
            },
            arch='any',
        )
        shutil.move(self.second_responses_resource_path, second_responses_resource.path)

        second_unpacked_responses_resource = self.create_resource(
            'second version noapacheupper responses for stability check',
            'unpacked_resp2.out',
            self.get_response_type(True),
            attributes={
                "ttl": 14,
            },
            arch='any',
        )
        shutil.move(self.second_unpocked_responses_resource_path, second_unpacked_responses_resource .path)

    def compare_hr_responses(self, queries, responses_resource, diff_dir):
        # compare pair of resp. results
        diff_maker = DiffProtoHrResponses(
            [brp.remove_unstable_properties, noapache_response_patcher.response_patcher],
            can_compare,
        )
        make_folder(diff_dir)
        with utils.TimeCounter('check reponses stability'):
            diff_maker.process(
                responses_resource.path,
                None,
                self.second_responses_resource_path,
                None,
                diff_dir,
                queries,
            )
        return diff_maker

    def compare_apphost_responses(self, queries, data_dir1, data_dir2, compare_result):
        data_filename1 = os.path.join(data_dir1, 'data')
        data_filename2 = os.path.join(data_dir2, 'data')
        index_filename1 = os.path.join(data_dir1, 'index')
        index_filename2 = os.path.join(data_dir2, 'index')
        make_folder(compare_result)
        converter = self.sync_resource(self.ctx[ToolConverter.name])
        proto2hr = self.sync_resource(self.ctx[ToolProto2Hr.name])
        app_host_ops = self.sync_resource(self.ctx[AppHostOps.name])

        diff_maker = DiffApphostResponses(
            converter,
            proto2hr,
            [
                remove_noapache_specific_noise,
                prepare_blender_tmpl_data_for_comparison,
            ],
            can_compare_noapache_apphost or can_compare_blender_tmpl_data,
            app_host_ops,
        )
        diff_maker.process(
            data_filename1,
            index_filename1,
            data_filename2,
            index_filename2,
            compare_result,
            queries,
        )
        return diff_maker

    def get_metric_constraints(self):
        class MetricConstraintOptions:
            delimeter = ';;'
            ops = {
                '<=': lambda x, y: x <= y,
                '>=': lambda x, y: x >= y,
                '<': lambda x, y: x < y,
                '>': lambda x, y: x > y,
                '==': lambda x, y: x == y,
                '!=': lambda x, y: x != y,
            }

            @staticmethod
            def get_ops():
                # reverse to match '<=' earlier than '<'
                return reversed(sorted(MetricConstraintOptions.ops.iteritems()))

        class MetricConstraintChecker:
            def __init__(self, expr):
                self.action = None
                self.expr = expr
                for (op, action) in MetricConstraintOptions.get_ops():
                    res = expr.split(op)
                    if len(res) == 2:
                        res[0] = res[0].strip()
                        res[1] = res[1].strip()
                        try:
                            self.regexp = re.compile(res[0])
                        except Exception:
                            raise SandboxTaskFailureError('Invalid regexp: {}'.format(res[0]))
                        if not res[1].isdigit():
                            raise SandboxTaskFailureError('Expected number. Got: {}'.format(res[1]))
                        self.value = int(res[1])
                        self.action = action
                        break
                eh.ensure(self.action, 'Got invalid constraint: {}'.format(expr))

            def apply(self, x):
                return self.action(x, self.value)

            def check(self, result):
                if isinstance(result[1], int) and self.regexp.match(result[0]) and not self.apply(result[1]):
                    return 'Constraint: {} is not satisfied. {} = {}.'.format(self.expr, result[0], result[1])
                return ""

        metric_constraints = self.ctx.get(MetricConstraint.name, MetricConstraint.default_value)
        return [MetricConstraintChecker(x) for x in metric_constraints.split(MetricConstraintOptions.delimeter) if x]

    def check_stats(self, search_component, constraints):
        errors = []
        for result in search_component.get_tass_output():
            for constraint in constraints:
                res = constraint.check(result)
                if res:
                    errors.append(res)
        eh.ensure(not errors, 'Some signals violate constraints: {}'.format(errors))


class ResponsesStandaloneNoapacheupper2(ResponsesStandaloneNoapacheupper):
    pass


class EventlogStandaloneNoapacheupper(StandaloneNoapacheupper):
    _eventlog_path = 'eventlog'
    eventlog_resource_id = 'eventlog_resource_id'

    client_tags = release_setup.WEBSEARCH_TAGS_P1

    def __init__(self, task_id=0, neh_cache_mode='read'):
        super(EventlogStandaloneNoapacheupper, self).__init__(task_id, neh_cache_mode)

    @staticmethod
    def get_input_parameters(neh_cache_mode):
        params = create_noapacheupper_params(neh_cache_mode=neh_cache_mode).params \
            + (request.ResourceParam, request.AddGlobalCtxParams) \
            + RequesterParams.lst \
            + Params.params
        return params

    def on_enqueue(self):
        super(EventlogStandaloneNoapacheupper, self).on_enqueue()
        # HACK: eventlogs folder already registered as OTHER_RESOURCE, so use eventlog copy
        self.create_noapacheupper_resource(
            self.eventlog_resource_id,
            'noapacheupper eventlog',
            self._eventlog_path,
            resource_types.EVENTLOG_DUMP
        )

    def on_execute(self):
        self._link_neh_cache_from_requests()
        search_component = get_noapacheupper(neh_cache_mode=self._get_neh_cache_mode())
        self.init_search_component(search_component)
        requester = Requester()
        req_iter = request.binary_requests_iterator(
            self.sync_resource(self.ctx[request.ResourceParam.name]),
            self.ctx.get(request.AddGlobalCtxParams.name),
            use_apphost=self.apphost_mode(),
        )
        requester.use(req_iter, search_component, ctx=self.ctx)
        self.check_neh_cache_errors(search_component)
        # HACK: eventlogs folder already registered as OTHER_RESOURCE, so use eventlog copy
        os.link(search_component.get_event_log_path(), self._eventlog_path)

    def apphost_mode(self):
        return self.ctx.get(NoapacheParams.AppHostMode.name, False)


class EventlogStandaloneNoapacheupper2(EventlogStandaloneNoapacheupper):
    pass


def _quantile(arr, q):
    arr.sort()
    qidx = int(len(arr) * q + 1)
    if qidx >= len(arr):  # handle tail overrun
        return (arr[-1] if arr else 0)
    return arr[qidx]


class TimingsStandaloneNoapacheupper(StandaloneNoapacheupper):
    avg_resp_time = "avg_resp_time"
    rearrange_timings = "rearrange_timings"
    quantile95_timings = "quantile95_timings"
    quantile99_timings = "quantile99_timings"
    timings_resource_id = "timings_resource_id"
    timings_filename = "timings.json"

    class ExternalSessionsLimit(sp.SandboxIntegerParameter):
        name = 'external_sessions_limit'
        description = 'Run external sessions (repeat queries)'
        group = 'Requester params'
        default_value = 0

    def __init__(self, task_id=0):
        super(TimingsStandaloneNoapacheupper, self).__init__(task_id)

    @staticmethod
    def get_input_parameters():
        params = create_noapacheupper_params(neh_cache_mode='read').params \
            + (request.ResourceParam, request.AddGlobalCtxParams) \
            + RequesterParams.lst \
            + Params.params
        return params

    def on_enqueue(self):
        super(TimingsStandaloneNoapacheupper, self).on_enqueue()
        if Params.EvlogdumpExecutable.name not in self.ctx or self.ctx[Params.EvlogdumpExecutable.name] == 0:
            return
        self.ctx[self.timings_resource_id] = self.create_resource(
            'requests profile/timings',
            self.timings_filename,
            resource_types.NOAPACHEUPPER_REQUEST_TIMINGS,
            attributes={
                "ttl": 14,
            },
            arch='any',
        ).id

    def on_execute(self):
        self._link_neh_cache_from_requests()
        search_component = get_noapacheupper(neh_cache_mode=self._get_neh_cache_mode())
        sessions_limit = 1 + self.ctx[self.ExternalSessionsLimit.name]

        avg_resp_times = []
        evlog_map = {}
        while sessions_limit:
            sessions_limit -= 1
            avg_resp_times.append(self.get_avg_response_time(search_component))
            self.check_neh_cache_errors(search_component)
            evlog_map[avg_resp_times[-1]] = search_component.event_log_path
        avg_resp_times.sort()
        self.ctx[self.avg_resp_time] = avg_resp_times[0]

        if Params.EvlogdumpExecutable.name not in self.ctx or self.ctx[Params.EvlogdumpExecutable.name] == 0:
            return

        logging.info('begin analyze eventlog')
        # use eventlog from session with best avg resp time result
        eventlog = evlog_map[self.ctx[self.avg_resp_time]]
        logging.info('get rearrange timings from eventlog: %s', eventlog)
        evlogdump_binary = self.sync_resource(self.ctx[Params.EvlogdumpExecutable.name])
        self.parsed_eventlog = self.abs_path(eventlog + '.txt')
        with open(eventlog) as in_file, open(self.parsed_eventlog, "w") as out_file:
            process.run_process([evlogdump_binary, '-i', '328'], stdin=in_file, stdout=out_file)
        timings = calc_rearrange_timings(self.parsed_eventlog)
        t_avg = {}
        t_q95 = {}
        t_q99 = {}
        for k, tg in timings.iteritems():
            t_avg[k] = tg.sum_time / tg.count  # microseconds
            t_q95[k] = _quantile(tg.arr, 0.95)  # microseconds
            t_q99[k] = _quantile(tg.arr, 0.99)  # microseconds
        self.ctx[self.rearrange_timings] = json.dumps(t_avg)
        self.ctx[self.quantile95_timings] = json.dumps(t_q95)
        self.ctx[self.quantile99_timings] = json.dumps(t_q99)
        filename_timings = channel.sandbox.get_resource(self.ctx[self.timings_resource_id]).path
        logging.info('write timings to: ' + filename_timings)
        with open(filename_timings, 'w') as f:
            tms = {
                self.avg_resp_time: self.ctx[self.avg_resp_time],
                self.rearrange_timings: t_avg,
                self.quantile95_timings: t_q95,
                self.quantile99_timings: t_q99,
            }
            json.dump(tms, f)

    def get_avg_response_time(self, search_component):
        return get_avg_response_time2(search_component=search_component, ctx=self.ctx)

    def check_eventlog_errors(self, eventlog):
        logging.info('eventlog checking disable for this task (hardcode)')


class TimingsStandaloneNoapacheupper2(TimingsStandaloneNoapacheupper):
    pass


def _create_task_COMPARE_BASESEARCH_RESPONSES(responses1, responses2):
    sub_ctx = {
        'basesearch_responses1_resource_id': responses1,
        'basesearch_responses2_resource_id': responses2,
        'fail_on_diff': True,
        'ignore_unanswered': True,
        'compare_same_upper_unanswered': True,
        'limit_of_uncompared_queries': 10,
        'kill_timeout': 21600,
        'response_format': 'noapacheupper',
    }

    return channel.task.create_subtask(
        task_type='COMPARE_BASESEARCH_RESPONSES',
        description="verify noapache responses stability",
        input_parameters=sub_ctx,
        arch='any'
    )
