# -*- coding: utf-8 -*-
"""
Sandbox parameters for common use
"""

import os
import re
import shlex
import six
import tempfile
import textwrap

from sandbox import sdk2
from sandbox.projects.common import file_utils as fu


class EnvironmentVariablesContainer(dict):

    ENV_VAR_PATTERN = r"^\s*(\b[a-zA-Z_]\w*=((\'[^\']*\')|(\"[^\"]*\")|([^\s\'\"]+))(\s+|$))+$"
    VAULT_PATTERN = r"\$\(vault:(?P<dst>file|value):(?P<owner>[^:]+):(?P<name>[^:]+)\)"
    YAV_PATTERN = r"\$\(yav:(?P<sec>sec-.+)#(?P<key>.+)\)"
    DEST_FILE = "file"

    def as_dict_deref(self):
        return {
            k: self.deref(v) for k, v in six.iteritems(self)
        }

    @classmethod
    def deref(cls, s):
        def deref_yav(match):
            secret = sdk2.yav.Secret(match.group('sec')).data()
            return secret[match.group('key')]

        def deref_vault(match):
            secret = sdk2.Vault.data(match.group("owner"), match.group("name"))

            if match.group("dst") == cls.DEST_FILE:
                deref_path = tempfile.NamedTemporaryFile().name
                fu.write_file(deref_path, secret)
                return deref_path

            return secret

        s = re.sub(cls.VAULT_PATTERN, deref_vault, s)
        s = re.sub(cls.YAV_PATTERN, deref_yav, s)

        return s

    @classmethod
    def is_value_valid(cls, env_vars_str):
        return re.match(cls.ENV_VAR_PATTERN, env_vars_str) is not None


class EnvironmentVariables(sdk2.parameters.String):
    name = "env_vars"
    description = textwrap.dedent("""\
        Environment variables (e.g. VAR1=val1 VAR2=\'v a l 2\').

        May be used with Vault:
        - `$(vault:file:owner:name)` - will be replaced with path to the file with secret
        - `$(vault:value:owner:name)` - will be replaced with value of the secret

        or with YaV:
        - `$(yav:secret_id#key)` - will be replaced with the value a secret with key `key` found in the YaV secret with
        id `secret_id`

        Examples:
            - YT_TOKEN='$(vault:value:yazevnul:yazevnul-yt_token)'
            - YT_TOKEN='$(yav:secret_id#key)'
            - YT_TOKEN_PATH='$(vault:file:yazevnul:yazevnul-yt_token)'
    """)
    default_value = ""

    @classmethod
    def apply_patch__runtimetests_115(cls, env_vars):
        """RUNTIMETESTS-115"""

        for key in ('GSID', 'PATH_TO_SOY_BATCH'):
            if key not in env_vars.keys() and key in os.environ.keys():
                env_vars.update({key: os.getenv(key)})

        return env_vars

    @classmethod
    def get_env_vars_from_str(cls, env_vars_str):
        if env_vars_str and not EnvironmentVariablesContainer.is_value_valid(env_vars_str):
            raise ValueError("Incorrect 'Environment variables' parameter '{}'".format(env_vars_str))

        env_vars = EnvironmentVariablesContainer(
            {k: v for k, v in (x.split('=', 1) for x in shlex.split(env_vars_str))}
        )

        cls.apply_patch__runtimetests_115(env_vars)

        return env_vars

    @classmethod
    def __decode__(cls, value):

        value = super(EnvironmentVariables, cls).__decode__(value)

        if isinstance(value, EnvironmentVariablesContainer):
            return value

        if not isinstance(value, dict):
            value = {}

        return EnvironmentVariablesContainer(value)

    @classmethod
    def cast(cls, value):
        if value and isinstance(value, six.string_types):
            return cls.get_env_vars_from_str(value)
        return value
